import _typeof from "@babel/runtime/helpers/esm/typeof";
import _classCallCheck from "@babel/runtime/helpers/esm/classCallCheck";
import _createClass from "@babel/runtime/helpers/esm/createClass";
import { getBinaryImageMIMEType } from '@loaders.gl/images';
import { padTo4Bytes, copyToArray } from '@loaders.gl/loader-utils';
import assert from './utils/assert';
import { getAccessorArrayTypeAndLength, getAccessorTypeFromSize, getComponentTypeFromArray } from './gltf-utils/gltf-utils';

var GLTFScenegraph = function () {
  function GLTFScenegraph(gltf) {
    _classCallCheck(this, GLTFScenegraph);

    if (gltf instanceof GLTFScenegraph) {
      return gltf;
    }

    if (!gltf) {
      gltf = {
        json: {
          version: 2,
          buffers: []
        },
        buffers: []
      };
    }

    this.gltf = gltf;
    assert(this.gltf.json);
  }

  _createClass(GLTFScenegraph, [{
    key: "getApplicationData",
    value: function getApplicationData(key) {
      var data = this.json[key];
      return data;
    }
  }, {
    key: "getExtraData",
    value: function getExtraData(key) {
      var extras = this.json.extras || {};
      return extras[key];
    }
  }, {
    key: "getExtension",
    value: function getExtension(extensionName) {
      var isExtension = this.getUsedExtensions().find(function (name) {
        return name === extensionName;
      });
      var extensions = this.json.extensions || {};
      return isExtension ? extensions[extensionName] || true : null;
    }
  }, {
    key: "getRequiredExtension",
    value: function getRequiredExtension(extensionName) {
      var isRequired = this.getRequiredExtensions().find(function (name) {
        return name === extensionName;
      });
      return isRequired ? this.getExtension(extensionName) : null;
    }
  }, {
    key: "getRequiredExtensions",
    value: function getRequiredExtensions() {
      return this.json.extensionsRequired || [];
    }
  }, {
    key: "getUsedExtensions",
    value: function getUsedExtensions() {
      return this.json.extensionsUsed || [];
    }
  }, {
    key: "getObjectExtension",
    value: function getObjectExtension(object, extensionName) {
      var extensions = object.extensions || {};
      return extensions[extensionName];
    }
  }, {
    key: "getScene",
    value: function getScene(index) {
      return this.getObject('scenes', index);
    }
  }, {
    key: "getNode",
    value: function getNode(index) {
      return this.getObject('nodes', index);
    }
  }, {
    key: "getSkin",
    value: function getSkin(index) {
      return this.getObject('skins', index);
    }
  }, {
    key: "getMesh",
    value: function getMesh(index) {
      return this.getObject('meshes', index);
    }
  }, {
    key: "getMaterial",
    value: function getMaterial(index) {
      return this.getObject('materials', index);
    }
  }, {
    key: "getAccessor",
    value: function getAccessor(index) {
      return this.getObject('accessors', index);
    }
  }, {
    key: "getCamera",
    value: function getCamera(index) {
      return null;
    }
  }, {
    key: "getTexture",
    value: function getTexture(index) {
      return this.getObject('textures', index);
    }
  }, {
    key: "getSampler",
    value: function getSampler(index) {
      return this.getObject('samplers', index);
    }
  }, {
    key: "getImage",
    value: function getImage(index) {
      return this.getObject('images', index);
    }
  }, {
    key: "getBufferView",
    value: function getBufferView(index) {
      return this.getObject('bufferViews', index);
    }
  }, {
    key: "getBuffer",
    value: function getBuffer(index) {
      return this.getObject('buffers', index);
    }
  }, {
    key: "getObject",
    value: function getObject(array, index) {
      if (_typeof(index) === 'object') {
        return index;
      }

      var object = this.json[array] && this.json[array][index];

      if (!object) {
        throw new Error("glTF file error: Could not find ".concat(array, "[").concat(index, "]"));
      }

      return object;
    }
  }, {
    key: "getTypedArrayForBufferView",
    value: function getTypedArrayForBufferView(bufferView) {
      bufferView = this.getBufferView(bufferView);
      var bufferIndex = bufferView.buffer;
      var binChunk = this.gltf.buffers[bufferIndex];
      assert(binChunk);
      var byteOffset = (bufferView.byteOffset || 0) + binChunk.byteOffset;
      return new Uint8Array(binChunk.arrayBuffer, byteOffset, bufferView.byteLength);
    }
  }, {
    key: "getTypedArrayForAccessor",
    value: function getTypedArrayForAccessor(accessor) {
      accessor = this.getAccessor(accessor);
      var bufferView = this.getBufferView(accessor.bufferView);
      var buffer = this.getBuffer(bufferView.buffer);
      var arrayBuffer = buffer.data;

      var _getAccessorArrayType = getAccessorArrayTypeAndLength(accessor, bufferView),
          ArrayType = _getAccessorArrayType.ArrayType,
          length = _getAccessorArrayType.length;

      var byteOffset = bufferView.byteOffset + accessor.byteOffset;
      return new ArrayType(arrayBuffer, byteOffset, length);
    }
  }, {
    key: "getTypedArrayForImageData",
    value: function getTypedArrayForImageData(image) {
      image = this.getAccessor(image);
      var bufferView = this.getBufferView(image.bufferView);
      var buffer = this.getBuffer(bufferView.buffer);
      var arrayBuffer = buffer.data;
      var byteOffset = bufferView.byteOffset || 0;
      return new Uint8Array(arrayBuffer, byteOffset, bufferView.byteLength);
    }
  }, {
    key: "addApplicationData",
    value: function addApplicationData(key, data) {
      this.json[key] = data;
      return this;
    }
  }, {
    key: "addExtraData",
    value: function addExtraData(key, data) {
      this.json.extras = this.json.extras || {};
      this.json.extras[key] = data;
      return this;
    }
  }, {
    key: "addObjectExtension",
    value: function addObjectExtension(object, extensionName, data) {
      assert(data);
      object.extensions = object.extensions || {};
      object.extensions[extensionName] = data;
      this.registerUsedExtension(extensionName);
      return this;
    }
  }, {
    key: "removeObjectExtension",
    value: function removeObjectExtension(object, extensionName) {
      var extensions = object.extensions || {};
      delete extensions[extensionName];
      return this;
    }
  }, {
    key: "addExtension",
    value: function addExtension(extensionName) {
      var extensionData = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      assert(extensionData);
      this.json.extensions = this.json.extensions || {};
      this.json.extensions[extensionName] = extensionData;
      this.registerUsedExtension(extensionName);
      return extensionData;
    }
  }, {
    key: "addRequiredExtension",
    value: function addRequiredExtension(extensionName) {
      var extensionData = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      assert(extensionData);
      this.addExtension(extensionName, extensionData);
      this.registerRequiredExtension(extensionName);
      return extensionData;
    }
  }, {
    key: "registerUsedExtension",
    value: function registerUsedExtension(extensionName) {
      this.json.extensionsUsed = this.json.extensionsUsed || [];

      if (!this.json.extensionsUsed.find(function (ext) {
        return ext === extensionName;
      })) {
        this.json.extensionsUsed.push(extensionName);
      }
    }
  }, {
    key: "registerRequiredExtension",
    value: function registerRequiredExtension(extensionName) {
      this.registerUsedExtension(extensionName);
      this.json.extensionsRequired = this.json.extensionsRequired || [];

      if (!this.json.extensionsRequired.find(function (ext) {
        return ext === extensionName;
      })) {
        this.json.extensionsRequired.push(extensionName);
      }
    }
  }, {
    key: "removeExtension",
    value: function removeExtension(extensionName) {
      if (this.json.extensionsRequired) {
        this._removeStringFromArray(this.json.extensionsRequired, extensionName);
      }

      if (this.json.extensionsUsed) {
        this._removeStringFromArray(this.json.extensionsUsed, extensionName);
      }

      if (this.json.extensions) {
        delete this.json.extensions[extensionName];
      }
    }
  }, {
    key: "setObjectExtension",
    value: function setObjectExtension(object, extensionName, data) {
      var extensions = object.extensions || {};
      extensions[extensionName] = data;
    }
  }, {
    key: "addMesh",
    value: function addMesh(attributes, indices) {
      var mode = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 4;

      var accessors = this._addAttributes(attributes);

      var glTFMesh = {
        primitives: [{
          attributes: accessors,
          indices: indices,
          mode: mode
        }]
      };
      this.json.meshes = this.json.meshes || [];
      this.json.meshes.push(glTFMesh);
      return this.json.meshes.length - 1;
    }
  }, {
    key: "addPointCloud",
    value: function addPointCloud(attributes) {
      var accessorIndices = this._addAttributes(attributes);

      var glTFMesh = {
        primitives: [{
          attributes: accessorIndices,
          mode: 0
        }]
      };
      this.json.meshes = this.json.meshes || [];
      this.json.meshes.push(glTFMesh);
      return this.json.meshes.length - 1;
    }
  }, {
    key: "addImage",
    value: function addImage(imageData, mimeType) {
      mimeType = mimeType || getBinaryImageMIMEType(imageData);
      var bufferViewIndex = this.addBufferView(imageData);
      var glTFImage = {
        bufferView: bufferViewIndex,
        mimeType: mimeType
      };
      this.json.images = this.json.images || [];
      this.json.images.push(glTFImage);
      return this.json.images.length - 1;
    }
  }, {
    key: "addBufferView",
    value: function addBufferView(buffer) {
      var byteLength = buffer.byteLength;
      assert(Number.isFinite(byteLength));
      this.sourceBuffers = this.sourceBuffers || [];
      this.sourceBuffers.push(buffer);
      var glTFBufferView = {
        buffer: 0,
        byteOffset: this.byteLength,
        byteLength: byteLength
      };
      this.byteLength += padTo4Bytes(byteLength);
      this.json.bufferViews = this.json.bufferViews || [];
      this.json.bufferViews.push(glTFBufferView);
      return this.json.bufferViews.length - 1;
    }
  }, {
    key: "addAccessor",
    value: function addAccessor(bufferViewIndex, accessor) {
      var glTFAccessor = {
        bufferView: bufferViewIndex,
        type: getAccessorTypeFromSize(accessor.size),
        componentType: accessor.componentType,
        count: accessor.count
      };
      this.json.accessors = this.json.accessors || [];
      this.json.accessors.push(glTFAccessor);
      return this.json.accessors.length - 1;
    }
  }, {
    key: "addBinaryBuffer",
    value: function addBinaryBuffer(sourceBuffer) {
      var accessor = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {
        size: 3
      };
      var bufferViewIndex = this.addBufferView(sourceBuffer);
      var accessorDefaults = {
        size: accessor.size,
        componentType: getComponentTypeFromArray(sourceBuffer),
        count: Math.round(sourceBuffer.length / accessor.size)
      };
      return this.addAccessor(bufferViewIndex, Object.assign(accessorDefaults, accessor));
    }
  }, {
    key: "createBinaryChunk",
    value: function createBinaryChunk() {
      if (this.arrayBuffer) {
        return;
      }

      var totalByteLength = this.byteLength;
      var arrayBuffer = new ArrayBuffer(totalByteLength);
      var targetArray = new Uint8Array(arrayBuffer);
      var dstByteOffset = 0;

      for (var i = 0; i < this.sourceBuffers.length; i++) {
        var sourceBuffer = this.sourceBuffers[i];
        dstByteOffset = copyToArray(sourceBuffer, targetArray, dstByteOffset);
      }

      this.json.buffers[0].byteLength = totalByteLength;
      this.arrayBuffer = arrayBuffer;
      this.sourceBuffers = [];
    }
  }, {
    key: "_removeStringFromArray",
    value: function _removeStringFromArray(array, string) {
      var found = true;

      while (found) {
        var index = array.indexOf(string);

        if (index > -1) {
          array.splice(index, 1);
        } else {
          found = false;
        }
      }
    }
  }, {
    key: "json",
    get: function get() {
      return this.gltf.json;
    }
  }]);

  return GLTFScenegraph;
}();

export { GLTFScenegraph as default };
//# sourceMappingURL=gltf-scenegraph.js.map