import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

import { Vector3, Matrix4 } from '@math.gl/core';
import { CullingVolume } from '@math.gl/culling';
import { load } from '@loaders.gl/core';
import { assert, path } from '@loaders.gl/loader-utils';
import { TILE_REFINEMENT, TILE_CONTENT_STATE, TILESET_TYPE } from '../constants';
import { createBoundingVolume } from './helpers/bounding-volume';
import { getTiles3DScreenSpaceError } from './helpers/tiles-3d-lod';
import { getI3ScreenSize } from './helpers/i3s-lod';
const scratchVector = new Vector3();

function defined(x) {
  return x !== undefined && x !== null;
}

function updatePriority(tile) {
  if (!tile.isVisible) {
    return -1;
  }

  if (tile.contentState === TILE_CONTENT_STATE.UNLOADED) {
    return -1;
  }

  return Math.max(1e7 - tile._priority, 0) || 0;
}

export default class TileHeader {
  constructor(tileset, header, parentHeader) {
    assert(typeof header === 'object');
    this.header = header;
    this.tileset = tileset;
    this.id = header.id;
    this.url = header.url;
    this.parent = parentHeader;
    this.refine = this._getRefine(header.refine);
    this.type = header.type;
    this.contentUrl = header.contentUrl;
    this.lodMetricType = null;
    this.lodMetricValue = null;
    this.boundingVolume = null;
    this.content = null;
    this.contentState = TILE_CONTENT_STATE.UNLOADED;
    this.gpuMemoryUsageInBytes = 0;
    this.children = [];
    this.depth = 0;
    this._cacheNode = null;
    this._frameNumber = null;
    this._cacheNode = null;

    this._initializeLodMetric(header);

    this._initializeTransforms(header);

    this._initializeBoundingVolumes(header);

    this._initializeContent(header);

    this._initializeRenderingState(header);

    this._lodJudge = null;
    this._expireDate = null;
    this._expiredContent = null;
    Object.seal(this);
  }

  destroy() {
    this.header = null;
  }

  isDestroyed() {
    return this.header === null;
  }

  get selected() {
    return this._selectedFrame === this.tileset._frameNumber;
  }

  get isVisible() {
    return this._visible;
  }

  get isVisibleAndInRequestVolume() {
    return this._visible && this._inRequestVolume;
  }

  get hasRenderContent() {
    return !this.hasEmptyContent && !this.hasTilesetContent;
  }

  get hasChildren() {
    return this.children.length > 0 || this.header.children && this.header.children.length > 0;
  }

  get contentReady() {
    return this.contentState === TILE_CONTENT_STATE.READY || this.hasEmptyContent;
  }

  get contentAvailable() {
    return Boolean(this.contentReady && this.hasRenderContent || this._expiredContent && !this.contentFailed);
  }

  get hasUnloadedContent() {
    return this.hasRenderContent && this.contentUnloaded;
  }

  get contentUnloaded() {
    return this.contentState === TILE_CONTENT_STATE.UNLOADED;
  }

  get contentExpired() {
    return this.contentState === TILE_CONTENT_STATE.EXPIRED;
  }

  get contentFailed() {
    return this.contentState === TILE_CONTENT_STATE.FAILED;
  }

  getScreenSpaceError(frameState, useParentLodMetric) {
    switch (this.tileset.type) {
      case TILESET_TYPE.I3S:
        return getI3ScreenSize(this, frameState);

      case TILESET_TYPE.TILES3D:
        return getTiles3DScreenSpaceError(this, frameState, useParentLodMetric);

      default:
        console.error('Unsupported tileset type');
        return null;
    }
  }

  async loadContent() {
    if (this.hasEmptyContent) {
      return false;
    }

    if (this.content) {
      return true;
    }

    const expired = this.contentExpired;

    if (expired) {
      this._expireDate = null;
    }

    this.contentState = TILE_CONTENT_STATE.LOADING;
    const requestToken = await this.tileset._requestScheduler.scheduleRequest(this.id, updatePriority);

    if (!requestToken) {
      this.contentState = TILE_CONTENT_STATE.UNLOADED;
      return false;
    }

    try {
      const contentUrl = this.tileset.getTileUrl(this.contentUrl);
      const fetchOptions = this.tileset.fetchOptions;
      const loader = this.tileset.loader;

      const options = _objectSpread(_objectSpread({}, fetchOptions), {}, {
        [loader.id]: {
          tile: this.header,
          tileset: this.tileset.tileset
        }
      });

      this.content = await load(contentUrl, loader, options);

      if (this._isTileset()) {
        this.tileset._initializeTileHeaders(this.content, this, path.dirname(this.contentUrl));
      }

      this.contentState = TILE_CONTENT_STATE.READY;

      this._onContentLoaded();

      return true;
    } catch (error) {
      this.contentState = TILE_CONTENT_STATE.FAILED;
      throw error;
    } finally {
      requestToken.done();
    }
  }

  unloadContent() {
    if (this.content && this.content.destroy) {
      this.content.destroy();
    }

    this.content = null;
    this.contentState = TILE_CONTENT_STATE.UNLOADED;
    return true;
  }

  updateVisibility(frameState) {
    if (this._frameNumber === frameState.frameNumber) {
      return;
    }

    const parent = this.parent;
    const parentTransform = parent ? parent.computedTransform : this.tileset.modelMatrix;
    const parentVisibilityPlaneMask = parent ? parent._visibilityPlaneMask : CullingVolume.MASK_INDETERMINATE;

    this._updateTransform(parentTransform);

    this._distanceToCamera = this.distanceToTile(frameState);
    this._screenSpaceError = this.getScreenSpaceError(frameState, false);
    this._visibilityPlaneMask = this.visibility(frameState, parentVisibilityPlaneMask);
    this._visible = this._visibilityPlaneMask !== CullingVolume.MASK_OUTSIDE;
    this._inRequestVolume = this.insideViewerRequestVolume(frameState);
    this._priority = this.lodMetricValue;
    this._frameNumber = frameState.frameNumber;
  }

  visibility(frameState, parentVisibilityPlaneMask) {
    const {
      cullingVolume
    } = frameState;
    const {
      boundingVolume
    } = this;
    return cullingVolume.computeVisibilityWithPlaneMask(boundingVolume, parentVisibilityPlaneMask);
  }

  contentVisibility(frameState) {
    return true;
  }

  distanceToTile(frameState) {
    const boundingVolume = this.boundingVolume;
    return Math.sqrt(Math.max(boundingVolume.distanceSquaredTo(frameState.camera.position), 0));
  }

  cameraSpaceZDepth({
    camera
  }) {
    const boundingVolume = this.boundingVolume;
    scratchVector.subVectors(boundingVolume.center, camera.position);
    return camera.direction.dot(scratchVector);
  }

  insideViewerRequestVolume(frameState) {
    const viewerRequestVolume = this._viewerRequestVolume;
    return !viewerRequestVolume || viewerRequestVolume.distanceToCamera(frameState) === 0.0;
  }

  _initializeLodMetric(header) {
    if ('lodMetricType' in header) {
      this.lodMetricType = header.lodMetricType;
    } else {
      this.lodMetricType = this.parent && this.parent.lodMetricType || this.tileset.lodMetricType;
      console.warn("3D Tile: Required prop lodMetricType is undefined. Using parent lodMetricType");
    }

    if ('lodMetricValue' in header) {
      this.lodMetricValue = header.lodMetricValue;
    } else {
      this.lodMetricValue = this.parent && this.parent.lodMetricValue || this.tileset.lodMetricValue;
      console.warn("3D Tile: Required prop lodMetricValue is undefined. Using parent lodMetricValue");
    }
  }

  _initializeTransforms(tileHeader) {
    this.transform = tileHeader.transform ? new Matrix4(tileHeader.transform) : new Matrix4();
    const parent = this.parent;
    const tileset = this.tileset;
    const parentTransform = parent && parent.computedTransform ? parent.computedTransform.clone() : tileset.modelMatrix.clone();
    this.computedTransform = new Matrix4(parentTransform).multiplyRight(this.transform);
    const parentInitialTransform = parent && parent._initialTransform ? parent._initialTransform.clone() : new Matrix4();
    this._initialTransform = new Matrix4(parentInitialTransform).multiplyRight(this.transform);
  }

  _initializeBoundingVolumes(tileHeader) {
    this._contentBoundingVolume = null;
    this._viewerRequestVolume = null;

    this._updateBoundingVolume(tileHeader);
  }

  _initializeContent(tileHeader) {
    this.content = {
      _tileset: this.tileset,
      _tile: this
    };
    this.hasEmptyContent = true;
    this.contentState = TILE_CONTENT_STATE.UNLOADED;
    this.hasTilesetContent = false;

    if (tileHeader.contentUrl) {
      this.content = null;
      this.hasEmptyContent = false;
    }
  }

  _initializeRenderingState(header) {
    this.depth = header.level;
    this._shouldRefine = false;
    this._distanceToCamera = 0;
    this._centerZDepth = 0;
    this._screenSpaceError = 0;
    this._visibilityPlaneMask = CullingVolume.MASK_INDETERMINATE;
    this._visible = false;
    this._inRequestVolume = false;
    this._stackLength = 0;
    this._selectionDepth = 0;
    this._frameNumber = 0;
    this._touchedFrame = 0;
    this._visitedFrame = 0;
    this._selectedFrame = 0;
    this._requestedFrame = 0;
    this._priority = 0.0;
  }

  _getRefine(refine) {
    return refine || this.parent && this.parent.refine || TILE_REFINEMENT.REPLACE;
  }

  _isTileset() {
    return this.contentUrl.indexOf('.json') !== -1;
  }

  _onContentLoaded() {
    switch (this.content && this.content.type) {
      case 'vctr':
      case 'geom':
        this.tileset.traverser.disableSkipLevelOfDetail = true;
        break;

      default:
    }

    if (this._isTileset()) {
      this.hasTilesetContent = true;
    }
  }

  _updateBoundingVolume(header) {
    this.boundingVolume = createBoundingVolume(header.boundingVolume, this.computedTransform, this.boundingVolume);
    const content = header.content;

    if (!content) {
      return;
    }

    if (content.boundingVolume) {
      this._contentBoundingVolume = createBoundingVolume(content.boundingVolume, this.computedTransform, this._contentBoundingVolume);
    }

    if (header.viewerRequestVolume) {
      this._viewerRequestVolume = createBoundingVolume(header.viewerRequestVolume, this.computedTransform, this._viewerRequestVolume);
    }
  }

  _updateTransform(parentTransform = new Matrix4()) {
    const computedTransform = parentTransform.clone().multiplyRight(this.transform);
    const didTransformChange = !computedTransform.equals(this.computedTransform);

    if (!didTransformChange) {
      return;
    }

    this.computedTransform = computedTransform;

    this._updateBoundingVolume(this.header);
  }

  updateExpiration() {
    if (defined(this._expireDate) && this.contentReady && !this.hasEmptyContent) {
      const now = Date.now();

      if (Date.lessThan(this._expireDate, now)) {
        this.contentState = TILE_CONTENT_STATE.EXPIRED;
        this._expiredContent = this.content;
      }
    }
  }

  get extras() {
    return this.header.extras;
  }

}
//# sourceMappingURL=tile-3d.js.map