"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createBoundingVolume = createBoundingVolume;

var _readOnlyError2 = _interopRequireDefault(require("@babel/runtime/helpers/readOnlyError"));

var _slicedToArray2 = _interopRequireDefault(require("@babel/runtime/helpers/slicedToArray"));

var _core = require("@math.gl/core");

var _culling = require("@math.gl/culling");

var _geospatial = require("@math.gl/geospatial");

var _loaderUtils = require("@loaders.gl/loader-utils");

function defined(x) {
  return x !== undefined && x !== null;
}

var scratchMatrix = new _core.Matrix3();
var scratchScale = new _core.Vector3();
var scratchNorthWest = new _core.Vector3();
var scratchSouthEast = new _core.Vector3();
var scratchTransform = new _core.Matrix4();

function createBoundingVolume(boundingVolumeHeader, transform, result) {
  (0, _loaderUtils.assert)(boundingVolumeHeader, '3D Tile: boundingVolume must be defined');

  if (boundingVolumeHeader.box) {
    return createBox(boundingVolumeHeader.box, transform, result);
  }

  if (boundingVolumeHeader.region) {
    var _boundingVolumeHeader = (0, _slicedToArray2["default"])(boundingVolumeHeader.region, 6),
        west = _boundingVolumeHeader[0],
        south = _boundingVolumeHeader[1],
        east = _boundingVolumeHeader[2],
        north = _boundingVolumeHeader[3],
        minHeight = _boundingVolumeHeader[4],
        maxHeight = _boundingVolumeHeader[5];

    var northWest = _geospatial.Ellipsoid.WGS84.cartographicToCartesian([(0, _core.degrees)(west), (0, _core.degrees)(north), minHeight], scratchNorthWest);

    var southEast = _geospatial.Ellipsoid.WGS84.cartographicToCartesian([(0, _core.degrees)(east), (0, _core.degrees)(south), maxHeight], scratchSouthEast);

    var centerInCartesian = new _core.Vector3().addVectors(northWest, southEast).multiplyScalar(0.5);
    var radius = new _core.Vector3().subVectors(northWest, southEast).len() / 2.0;
    return createSphere([centerInCartesian[0], centerInCartesian[1], centerInCartesian[2], radius], new _core.Matrix4());
  }

  if (boundingVolumeHeader.sphere) {
    return createSphere(boundingVolumeHeader.sphere, transform, result);
  }

  throw new Error('3D Tile: boundingVolume must contain a sphere, region, or box');
}

function createBox(box, transform, result) {
  var center = new _core.Vector3(box[0], box[1], box[2]);
  transform.transform(center, center);
  var xAxis = transform.transformAsVector(box.slice(3, 6));
  var yAxis = transform.transformAsVector(box.slice(6, 9));
  var zAxis = transform.transformAsVector(box.slice(9, 12));
  var halfAxes = new _core.Matrix3([xAxis[0], xAxis[1], xAxis[2], yAxis[0], yAxis[1], yAxis[2], zAxis[0], zAxis[1], zAxis[2]]);

  if (defined(result)) {
    result.center = center;
    result.halfAxes = halfAxes;
    return result;
  }

  return new _culling.OrientedBoundingBox(center, halfAxes);
}

function createBoxFromTransformedRegion(region, transform, initialTransform, result) {
  var rectangle = Rectangle.unpack(region, 0, scratchRectangle);
  var minimumHeight = region[4];
  var maximumHeight = region[5];

  var orientedBoundingBox = _culling.OrientedBoundingBox.fromRectangle(rectangle, minimumHeight, maximumHeight, _geospatial.Ellipsoid.WGS84, scratchOrientedBoundingBox);

  var center = orientedBoundingBox.center;
  var halfAxes = orientedBoundingBox.halfAxes;
  transform = _core.Matrix4.multiplyTransformation(transform, _core.Matrix4.inverseTransformation(initialTransform, scratchTransform), scratchTransform);
  center = ((0, _readOnlyError2["default"])("center"), _core.Matrix4.multiplyByPoint(transform, center, center));

  var rotationScale = _core.Matrix4.getRotation(transform, scratchMatrix);

  halfAxes = ((0, _readOnlyError2["default"])("halfAxes"), _core.Matrix3.multiply(rotationScale, halfAxes, halfAxes));

  if (defined(result) && result instanceof TileOrientedBoundingBox) {
    result.update(center, halfAxes);
    return result;
  }

  return new TileOrientedBoundingBox(center, halfAxes);
}

function createRegion(region, transform, initialTransform, result) {
  if (!_core.Matrix4.equalsEpsilon(transform, initialTransform, CesiumMath.EPSILON8)) {
    return createBoxFromTransformedRegion(region, transform, initialTransform, result);
  }

  if (defined(result)) {
    return result;
  }

  var rectangleRegion = Rectangle.unpack(region, 0, scratchRectangle);
  return new TileBoundingRegion({
    rectangle: rectangleRegion,
    minimumHeight: region[4],
    maximumHeight: region[5]
  });
}

function createSphere(sphere, transform, result) {
  var center = new _core.Vector3(sphere[0], sphere[1], sphere[2]);
  transform.transform(center, center);
  var scale = transform.getScale(scratchScale);
  var uniformScale = Math.max(Math.max(scale[0], scale[1]), scale[2]);
  var radius = sphere[3] * uniformScale;

  if (defined(result)) {
    result.center = center;
    result.radius = radius;
    return result;
  }

  return new _culling.BoundingSphere(center, radius);
}
//# sourceMappingURL=bounding-volume.js.map