import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

import { Matrix4, Vector3 } from '@math.gl/core';
import { Ellipsoid } from '@math.gl/geospatial';
import { Stats } from '@probe.gl/stats';
import { RequestScheduler, assert, path } from '@loaders.gl/loader-utils';
import TilesetCache from './tileset-cache';
import { calculateTransformProps } from './helpers/transform-utils';
import { getFrameState } from './helpers/frame-state';
import { getZoomFromBoundingVolume } from './helpers/zoom';
import Tile3D from './tile-3d';
import Tileset3DTraverser from './traversers/tileset-3d-traverser';
import TilesetTraverser from './traversers/tileset-traverser';
import I3SetTraverser from './traversers/i3s-tilset-traverser';
import { TILESET_TYPE } from '../constants';
const TILES_TOTAL = 'Tiles In Tileset(s)';
const TILES_IN_MEMORY = 'Tiles In Memory';
const TILES_IN_VIEW = 'Tiles In View';
const TILES_RENDERABLE = 'Tiles To Render';
const TILES_LOADED = 'Tiles Loaded';
const TILES_LOADING = 'Tiles Loading';
const TILES_UNLOADED = 'Tiles Unloaded';
const TILES_LOAD_FAILED = 'Failed Tile Loads';
const POINTS_COUNT = 'Points';
const TILES_GPU_MEMORY = 'Tile Memory Use';

function getQueryParamString(queryParams) {
  const queryParamStrings = [];

  for (const key of Object.keys(queryParams)) {
    queryParamStrings.push("".concat(key, "=").concat(queryParams[key]));
  }

  switch (queryParamStrings.length) {
    case 0:
      return '';

    case 1:
      return "?".concat(queryParamStrings[0]);

    default:
      return "?".concat(queryParamStrings.join('&'));
  }
}

const DEFAULT_OPTIONS = {
  ellipsoid: Ellipsoid.WGS84,
  modelMatrix: new Matrix4(),
  throttleRequests: false,
  maximumMemoryUsage: 32,
  onTileLoad: () => {},
  onTileUnload: () => {},
  onTileError: (tile, message, url) => {},
  maximumScreenSpaceError: 8
};
export default class Tileset3D {
  constructor(json, options = {}) {
    assert(json);
    this.options = _objectSpread(_objectSpread({}, DEFAULT_OPTIONS), options);
    this.tileset = json;
    this.loader = json.loader;
    this.type = json.type;
    this.url = json.url;
    this.basePath = json.basePath || path.dirname(this.url);
    this.modelMatrix = this.options.modelMatrix;
    this.ellipsoid = this.options.ellipsoid;
    this.lodMetricType = json.lodMetricType;
    this.lodMetricValue = json.lodMetricValue;
    this.refine = json.root.refine;
    this.fetchOptions = this.options.fetchOptions || {};

    if (this.options.headers) {
      this.fetchOptions.headers = this.options.headers;
    }

    if (this.options.token) {
      this.fetchOptions.token = this.options.token;
    }

    this.root = null;
    this.cartographicCenter = null;
    this.cartesianCenter = null;
    this.zoom = 1;
    this.boundingVolume = null;
    this._traverser = this._initializeTraverser();
    this._cache = new TilesetCache();
    this._requestScheduler = new RequestScheduler({
      throttleRequests: this.options.throttleRequests
    });
    this._frameNumber = 0;
    this._updateFrameNumber = 0;
    this._pendingCount = 0;
    this._tiles = {};
    this.selectedTiles = [];
    this._emptyTiles = [];
    this._requestedTiles = [];
    this._selectedTilesToStyle = [];
    this._queryParams = {};
    this._queryParamsString = null;
    this.maximumMemoryUsage = this.options.maximumMemoryUsage;
    this.gpuMemoryUsageInBytes = 0;
    this.stats = new Stats({
      id: this.url
    });

    this._initializeStats();

    this._hasMixedContent = false;
    this._maximumScreenSpaceError = this.options.maximumScreenSpaceError;
    this._properties = undefined;
    this._extensionsUsed = undefined;
    this._gltfUpAxis = undefined;
    this._dynamicScreenSpaceErrorComputedDensity = 0.0;
    this.extras = null;
    this.asset = {};
    this.credits = {};
    this.description = this.options.description;
    this._defaultGeometrySchema = [];

    this._initializeTileSet(json, this.options);
  }

  isLoaded() {
    return this._pendingCount === 0;
  }

  destroy() {
    this._destroy();
  }

  get tiles() {
    return Object.values(this._tiles);
  }

  getTileUrl(tilePath) {
    const isDataUrl = tilePath.startsWith('data:');

    if (isDataUrl) {
      return tilePath;
    }

    return "".concat(tilePath).concat(this.queryParams);
  }

  update(viewport) {
    this._cache.reset();

    this._frameNumber++;
    this._frameState = getFrameState(viewport, this._frameNumber);

    this._traverser.traverse(this.root, this._frameState, this.options);
  }

  _onTraversalEnd() {
    const selectedTiles = Object.values(this._traverser.selectedTiles);

    if (this._tilesChanged(this.selectedTiles, selectedTiles)) {
      this._updateFrameNumber++;
    }

    this.selectedTiles = selectedTiles;

    for (const tile of this.selectedTiles) {
      this._tiles[tile.id] = tile;
    }

    this._requestedTiles = Object.values(this._traverser.requestedTiles);
    this._emptyTiles = Object.values(this._traverser.emptyTiles);

    this._loadTiles(this._frameState);

    this._unloadTiles();

    this._updateStats();

    return this._updateFrameNumber;
  }

  _tilesChanged(oldSelectedTiles, selectedTiles) {
    if (oldSelectedTiles.length !== selectedTiles.length) {
      return true;
    }

    const set1 = new Set(oldSelectedTiles.map(t => t.id));
    const set2 = new Set(selectedTiles.map(t => t.id));
    let changed = oldSelectedTiles.filter(x => !set2.has(x.id)).length > 0;
    changed = changed || selectedTiles.filter(x => !set1.has(x.id)).length > 0;
    return changed;
  }

  _loadTiles(frameState) {
    for (const tile of this._requestedTiles) {
      if (tile.contentUnloaded) {
        this._loadTile(tile, frameState);
      }
    }
  }

  _unloadTiles() {
    this._cache.unloadTiles(this, (tileset, tile) => tileset._unloadTile(tile));
  }

  _updateStats() {
    let tilesRenderable = 0;
    let pointsRenderable = 0;

    for (const tile of this.selectedTiles) {
      if (tile.contentAvailable) {
        tilesRenderable++;

        if (tile.content.pointCount) {
          pointsRenderable += tile.content.pointCount;
        }
      }
    }

    this.stats.get(TILES_IN_VIEW).count = this.selectedTiles.length;
    this.stats.get(TILES_RENDERABLE).count = tilesRenderable;
    this.stats.get(POINTS_COUNT).count = pointsRenderable;
  }

  _initializeTileSet(tilesetJson) {
    this.root = this._initializeTileHeaders(tilesetJson, null, this.basePath);

    if (this.type === TILESET_TYPE.TILES3D) {
      this._initializeCesiumTileset(tilesetJson);
    }

    if (this.type === TILESET_TYPE.I3S) {
      this._initializeI3STileset(tilesetJson);
    }

    this._calculateViewProps();
  }

  _calculateViewProps() {
    const root = this.root;
    const {
      center
    } = root.boundingVolume;

    if (!center) {
      console.warn('center was not pre-calculated for the root tile');
      this.cartographicCenter = new Vector3();
      this.zoom = 1;
      return;
    }

    this.cartographicCenter = Ellipsoid.WGS84.cartesianToCartographic(center, new Vector3());
    this.cartesianCenter = center;
    this.zoom = getZoomFromBoundingVolume(root.boundingVolume);
  }

  _initializeStats() {
    this.stats.get(TILES_TOTAL);
    this.stats.get(TILES_LOADING);
    this.stats.get(TILES_IN_MEMORY);
    this.stats.get(TILES_IN_VIEW);
    this.stats.get(TILES_RENDERABLE);
    this.stats.get(TILES_LOADED);
    this.stats.get(TILES_UNLOADED);
    this.stats.get(TILES_LOAD_FAILED);
    this.stats.get(POINTS_COUNT, 'memory');
    this.stats.get(TILES_GPU_MEMORY, 'memory');
  }

  _initializeTileHeaders(tilesetJson, parentTileHeader, basePath) {
    const rootTile = new Tile3D(this, tilesetJson.root, parentTileHeader, basePath);

    if (parentTileHeader) {
      parentTileHeader.children.push(rootTile);
      rootTile.depth = parentTileHeader.depth + 1;
    }

    if (this.type === TILESET_TYPE.TILES3D) {
      const stack = [];
      stack.push(rootTile);

      while (stack.length > 0) {
        const tile = stack.pop();
        this.stats.get(TILES_TOTAL).incrementCount();
        const children = tile.header.children || [];

        for (const childHeader of children) {
          const childTile = new Tile3D(this, childHeader, tile, basePath);
          tile.children.push(childTile);
          childTile.depth = tile.depth + 1;
          stack.push(childTile);
        }
      }
    }

    return rootTile;
  }

  _initializeTraverser() {
    let TraverserClass;
    const type = this.type;

    switch (type) {
      case TILESET_TYPE.TILES3D:
        TraverserClass = Tileset3DTraverser;
        break;

      case TILESET_TYPE.I3S:
        TraverserClass = I3SetTraverser;
        break;

      default:
        TraverserClass = TilesetTraverser;
    }

    return new TraverserClass({
      basePath: this.basePath,
      onTraversalEnd: this._onTraversalEnd.bind(this)
    });
  }

  _destroyTileHeaders(parentTile) {
    this._destroySubtree(parentTile);
  }

  async _loadTile(tile, frameState) {
    let loaded;

    try {
      this._onStartTileLoading();

      loaded = await tile.loadContent(frameState);
    } catch (error) {
      this._onTileLoadError(tile, error);
    } finally {
      this._onEndTileLoading();

      this._onTileLoad(tile, loaded);
    }
  }

  _onTileLoadError(tile, error) {
    this.stats.get(TILES_LOAD_FAILED).incrementCount();
    const message = error.message || error.toString();
    const url = tile.url;
    console.error("A 3D tile failed to load: ".concat(tile.url, " ").concat(message));
    this.options.onTileError(tile, message, url);
  }

  _onTileLoad(tile, loaded) {
    if (!loaded) {
      return;
    }

    if (tile && tile.content) {
      calculateTransformProps(tile, tile.content);
    }

    this._addTileToCache(tile);

    this.options.onTileLoad(tile);
  }

  _onStartTileLoading() {
    this._pendingCount++;
    this.stats.get(TILES_LOADING).incrementCount();
  }

  _onEndTileLoading() {
    this._pendingCount--;
    this.stats.get(TILES_LOADING).decrementCount();
  }

  _addTileToCache(tile) {
    this._cache.add(this, tile, tileset => tileset._updateCacheStats(tile));
  }

  _updateCacheStats(tile) {
    this.stats.get(TILES_LOADED).incrementCount();
    this.stats.get(TILES_IN_MEMORY).incrementCount();
    this.gpuMemoryUsageInBytes += tile.content.byteLength || 0;
    this.stats.get(TILES_GPU_MEMORY).count = this.gpuMemoryUsageInBytes;
  }

  _unloadTile(tile) {
    this.gpuMemoryUsageInBytes -= tile.content.byteLength || 0;
    this.stats.get(TILES_IN_MEMORY).decrementCount();
    this.stats.get(TILES_UNLOADED).incrementCount();
    this.stats.get(TILES_GPU_MEMORY).count = this.gpuMemoryUsageInBytes;
    this.options.onTileUnload(tile);
    tile.unloadContent();
  }

  _destroy() {
    const stack = [];

    if (this.root) {
      stack.push(this.root);
    }

    while (stack.length > 0) {
      const tile = stack.pop();

      for (const child of tile.children) {
        stack.push(child);
      }

      this._destroyTile(tile);
    }

    this.root = null;
  }

  _destroySubtree(tile) {
    const root = tile;
    const stack = [];
    stack.push(root);

    while (stack.length > 0) {
      tile = stack.pop();

      for (const child of tile.children) {
        stack.push(child);
      }

      if (tile !== root) {
        this._destroyTile(tile);
      }
    }

    root.children = [];
  }

  _destroyTile(tile) {
    this._cache.unloadTile(this, tile);

    this._unloadTile(tile);

    tile.destroy();
  }

  _initializeCesiumTileset(tilesetJson) {
    this.asset = tilesetJson.asset;

    if (!this.asset) {
      throw new Error('Tileset must have an asset property.');
    }

    if (this.asset.version !== '0.0' && this.asset.version !== '1.0') {
      throw new Error('The tileset must be 3D Tiles version 0.0 or 1.0.');
    }

    if ('tilesetVersion' in this.asset) {
      this._queryParams.v = this.asset.tilesetVersion;
    }

    this.credits = {
      attributions: this.options.attributions || []
    };
    this.description = this.options.description;
    this.properties = tilesetJson.properties;
    this.geometricError = tilesetJson.geometricError;
    this._extensionsUsed = tilesetJson.extensionsUsed;
    this.extras = tilesetJson.extras;
  }

  _initializeI3STileset(tilesetJson) {
    if ('token' in this.options) {
      this._queryParams.token = this.options.token;
    }

    this._defaultGeometrySchema = tilesetJson.store.defaultGeometrySchema;
  }

  hasExtension(extensionName) {
    return Boolean(this._extensionsUsed && this._extensionsUsed.indexOf(extensionName) > -1);
  }

  get queryParams() {
    if (!this._queryParamsString) {
      this._queryParamsString = getQueryParamString(this._queryParams);
    }

    return this._queryParamsString;
  }

}
//# sourceMappingURL=tileset-3d.js.map