import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

import ManagedArray from '../../utils/managed-array';
import { TILE_REFINEMENT } from '../../constants';
import { assert } from '@loaders.gl/loader-utils';
export const DEFAULT_OPTIONS = {
  loadSiblings: false,
  skipLevelOfDetail: false,
  maximumScreenSpaceError: 2
};
export default class TilesetTraverser {
  constructor(options) {
    this.options = _objectSpread(_objectSpread({}, DEFAULT_OPTIONS), options);
    this._traversalStack = new ManagedArray();
    this._emptyTraversalStack = new ManagedArray();
    this._frameNumber = null;
    this.root = null;
    this.selectedTiles = {};
    this.requestedTiles = {};
    this.emptyTiles = {};
  }

  traverse(root, frameState, options) {
    this.root = root;
    this.options = _objectSpread(_objectSpread({}, this.options), options);
    this.reset();
    this.updateTile(root, frameState);
    this._frameNumber = frameState.frameNumber;
    this.executeTraversal(root, frameState);
  }

  reset() {
    this.requestedTiles = {};
    this.selectedTiles = {};
    this.emptyTiles = {};

    this._traversalStack.reset();

    this._emptyTraversalStack.reset();
  }

  executeTraversal(root, frameState) {
    const stack = this._traversalStack;
    stack.push(root);

    while (stack.length > 0) {
      const tile = stack.pop();
      let shouldRefine = false;

      if (this.canTraverse(tile, frameState)) {
        this.updateChildTiles(tile, frameState);
        shouldRefine = this.updateAndPushChildren(tile, frameState, stack);
      }

      const parent = tile.parent;
      const parentRefines = Boolean(!parent || parent._shouldRefine);
      const stoppedRefining = !shouldRefine;

      if (!tile.hasRenderContent) {
        this.emptyTiles[tile.id] = tile;
        this.loadTile(tile, frameState);

        if (stoppedRefining) {
          this.selectTile(tile, frameState);
        }
      } else if (tile.refine === TILE_REFINEMENT.ADD) {
        this.loadTile(tile, frameState);
        this.selectTile(tile, frameState);
      } else if (tile.refine === TILE_REFINEMENT.REPLACE) {
        this.loadTile(tile, frameState);

        if (stoppedRefining) {
          this.selectTile(tile, frameState);
        }
      }

      this.touchTile(tile, frameState);
      tile._shouldRefine = shouldRefine && parentRefines;
    }

    if (this.options.onTraversalEnd) {
      this.options.onTraversalEnd(frameState);
    }
  }

  updateChildTiles(tile, frameState) {
    const children = tile.children;

    for (const child of children) {
      this.updateTile(child, frameState);
    }

    return true;
  }

  updateAndPushChildren(tile, frameState, stack) {
    const {
      loadSiblings,
      skipLevelOfDetail
    } = this.options;
    const children = tile.children;
    children.sort(this.compareDistanceToCamera);
    const checkRefines = !skipLevelOfDetail && tile.refine === TILE_REFINEMENT.REPLACE && tile.hasRenderContent;
    let hasVisibleChild = false;

    for (const child of children) {
      if (child.isVisibleAndInRequestVolume) {
        if (stack.find(child)) {
          stack.delete(child);
        }

        stack.push(child);
        hasVisibleChild = true;
      } else if (checkRefines || loadSiblings) {
        this.loadTile(child, frameState);
        this.touchTile(child, frameState);
      }

      if (checkRefines) {
        let childRefines;

        if (!child._inRequestVolume) {
          childRefines = false;
        } else if (!child.hasRenderContent) {
          childRefines = this.executeEmptyTraversal(child, frameState);
        } else {
          childRefines = child.contentAvailable;
        }

        if (!childRefines) {
          return childRefines;
        }
      }
    }

    return hasVisibleChild;
  }

  updateTile(tile, frameState) {
    this.updateTileVisibility(tile, frameState);
  }

  selectTile(tile, frameState) {
    if (this.shouldSelectTile(tile, frameState)) {
      tile._selectedFrame = frameState.frameNumber;
      this.selectedTiles[tile.id] = tile;
    }
  }

  loadTile(tile, frameState) {
    if (this.shouldLoadTile(tile, frameState)) {
      tile._requestedFrame = frameState.frameNumber;
      tile._priority = this.getPriority(tile);
      this.requestedTiles[tile.id] = tile;
    }
  }

  touchTile(tile, frameState) {
    tile.tileset._cache.touch(tile);

    tile._touchedFrame = frameState.frameNumber;
  }

  canTraverse(tile, frameState, useParentMetric = false, ignoreVisibility = false) {
    if (!ignoreVisibility && !tile.isVisibleAndInRequestVolume) {
      return false;
    }

    if (!tile.hasChildren) {
      return false;
    }

    if (tile.hasTilesetContent) {
      return !tile.contentExpired;
    }

    return this.shouldRefine(tile, frameState, useParentMetric);
  }

  shouldLoadTile(tile, frameState) {
    return tile.hasUnloadedContent || tile.contentExpired;
  }

  shouldSelectTile(tile, frameState) {
    return tile.contentAvailable && !this.options.skipLevelOfDetail;
  }

  shouldRefine(tile, frameState, useParentMetric) {
    let screenSpaceError = tile._screenSpaceError;

    if (useParentMetric) {
      screenSpaceError = tile.getScreenSpaceError(frameState, true);
    }

    return screenSpaceError > this.options.maximumScreenSpaceError;
  }

  updateTileVisibility(tile, frameState) {
    tile.updateVisibility(frameState);
  }

  compareDistanceToCamera(b, a) {
    return b._distanceToCamera - a._distanceToCamera;
  }

  getPriority(tile) {
    const {
      options
    } = this;

    switch (tile.refine) {
      case TILE_REFINEMENT.ADD:
        return tile._distanceToCamera;

      case TILE_REFINEMENT.REPLACE:
        const {
          parent
        } = tile;
        const useParentScreenSpaceError = parent && (!options.skipLevelOfDetail || tile._screenSpaceError === 0.0 || parent.hasTilesetContent);
        const screenSpaceError = useParentScreenSpaceError ? parent._screenSpaceError : tile._screenSpaceError;
        const rootScreenSpaceError = this.root._screenSpaceError;
        return rootScreenSpaceError - screenSpaceError;

      default:
        return assert(false);
    }
  }

  anyChildrenVisible(tile, frameState) {
    let anyVisible = false;

    for (const child of tile.children) {
      child.updateVisibility(frameState);
      anyVisible = anyVisible || child.isVisibleAndInRequestVolume;
    }

    return anyVisible;
  }

  executeEmptyTraversal(root, frameState) {
    let allDescendantsLoaded = true;
    const stack = this._emptyTraversalStack;

    while (stack.length > 0) {
      const tile = stack.pop();
      this.updateTile(tile, frameState);

      if (!tile.isVisibleAndInRequestVolume) {
        this.loadTile(tile, frameState);
        this.touchTile(tile, frameState);
      }

      const traverse = !tile.hasRenderContent && this.canTraverse(tile, frameState, false, true);

      if (!traverse && !tile.contentAvailable) {
        allDescendantsLoaded = false;
      }

      if (traverse) {
        const children = tile.children.filter(c => c);

        for (const child of children) {
          if (stack.find(child)) {
            stack.delete(child);
          }

          stack.push(child);
        }
      }
    }

    return allDescendantsLoaded;
  }

}
//# sourceMappingURL=tileset-traverser.js.map