import _readOnlyError from "@babel/runtime/helpers/esm/readOnlyError";
import _slicedToArray from "@babel/runtime/helpers/esm/slicedToArray";
import { Vector3, Matrix3, Matrix4, degrees } from '@math.gl/core';
import { BoundingSphere, OrientedBoundingBox } from '@math.gl/culling';
import { Ellipsoid } from '@math.gl/geospatial';
import { assert } from '@loaders.gl/loader-utils';

function defined(x) {
  return x !== undefined && x !== null;
}

var scratchMatrix = new Matrix3();
var scratchScale = new Vector3();
var scratchNorthWest = new Vector3();
var scratchSouthEast = new Vector3();
var scratchTransform = new Matrix4();
export function createBoundingVolume(boundingVolumeHeader, transform, result) {
  assert(boundingVolumeHeader, '3D Tile: boundingVolume must be defined');

  if (boundingVolumeHeader.box) {
    return createBox(boundingVolumeHeader.box, transform, result);
  }

  if (boundingVolumeHeader.region) {
    var _boundingVolumeHeader = _slicedToArray(boundingVolumeHeader.region, 6),
        west = _boundingVolumeHeader[0],
        south = _boundingVolumeHeader[1],
        east = _boundingVolumeHeader[2],
        north = _boundingVolumeHeader[3],
        minHeight = _boundingVolumeHeader[4],
        maxHeight = _boundingVolumeHeader[5];

    var northWest = Ellipsoid.WGS84.cartographicToCartesian([degrees(west), degrees(north), minHeight], scratchNorthWest);
    var southEast = Ellipsoid.WGS84.cartographicToCartesian([degrees(east), degrees(south), maxHeight], scratchSouthEast);
    var centerInCartesian = new Vector3().addVectors(northWest, southEast).multiplyScalar(0.5);
    var radius = new Vector3().subVectors(northWest, southEast).len() / 2.0;
    return createSphere([centerInCartesian[0], centerInCartesian[1], centerInCartesian[2], radius], new Matrix4());
  }

  if (boundingVolumeHeader.sphere) {
    return createSphere(boundingVolumeHeader.sphere, transform, result);
  }

  throw new Error('3D Tile: boundingVolume must contain a sphere, region, or box');
}

function createBox(box, transform, result) {
  var center = new Vector3(box[0], box[1], box[2]);
  transform.transform(center, center);
  var xAxis = transform.transformAsVector(box.slice(3, 6));
  var yAxis = transform.transformAsVector(box.slice(6, 9));
  var zAxis = transform.transformAsVector(box.slice(9, 12));
  var halfAxes = new Matrix3([xAxis[0], xAxis[1], xAxis[2], yAxis[0], yAxis[1], yAxis[2], zAxis[0], zAxis[1], zAxis[2]]);

  if (defined(result)) {
    result.center = center;
    result.halfAxes = halfAxes;
    return result;
  }

  return new OrientedBoundingBox(center, halfAxes);
}

function createBoxFromTransformedRegion(region, transform, initialTransform, result) {
  var rectangle = Rectangle.unpack(region, 0, scratchRectangle);
  var minimumHeight = region[4];
  var maximumHeight = region[5];
  var orientedBoundingBox = OrientedBoundingBox.fromRectangle(rectangle, minimumHeight, maximumHeight, Ellipsoid.WGS84, scratchOrientedBoundingBox);
  var center = orientedBoundingBox.center;
  var halfAxes = orientedBoundingBox.halfAxes;
  transform = Matrix4.multiplyTransformation(transform, Matrix4.inverseTransformation(initialTransform, scratchTransform), scratchTransform);
  center = (_readOnlyError("center"), Matrix4.multiplyByPoint(transform, center, center));
  var rotationScale = Matrix4.getRotation(transform, scratchMatrix);
  halfAxes = (_readOnlyError("halfAxes"), Matrix3.multiply(rotationScale, halfAxes, halfAxes));

  if (defined(result) && result instanceof TileOrientedBoundingBox) {
    result.update(center, halfAxes);
    return result;
  }

  return new TileOrientedBoundingBox(center, halfAxes);
}

function createRegion(region, transform, initialTransform, result) {
  if (!Matrix4.equalsEpsilon(transform, initialTransform, CesiumMath.EPSILON8)) {
    return createBoxFromTransformedRegion(region, transform, initialTransform, result);
  }

  if (defined(result)) {
    return result;
  }

  var rectangleRegion = Rectangle.unpack(region, 0, scratchRectangle);
  return new TileBoundingRegion({
    rectangle: rectangleRegion,
    minimumHeight: region[4],
    maximumHeight: region[5]
  });
}

function createSphere(sphere, transform, result) {
  var center = new Vector3(sphere[0], sphere[1], sphere[2]);
  transform.transform(center, center);
  var scale = transform.getScale(scratchScale);
  var uniformScale = Math.max(Math.max(scale[0], scale[1]), scale[2]);
  var radius = sphere[3] * uniformScale;

  if (defined(result)) {
    result.center = center;
    result.radius = radius;
    return result;
  }

  return new BoundingSphere(center, radius);
}
//# sourceMappingURL=bounding-volume.js.map