import { assert } from '@luma.gl/webgl';
import { combineInjects, getQualifierDetails, typeToChannelSuffix } from '@luma.gl/shadertools';
const SAMPLER_UNIFORM_PREFIX = 'transform_uSampler_';
const SIZE_UNIFORM_PREFIX = 'transform_uSize_';
const VS_POS_VARIABLE = 'transform_position';
export function updateForTextures({
  vs,
  sourceTextureMap,
  targetTextureVarying,
  targetTexture
}) {
  const texAttributeNames = Object.keys(sourceTextureMap);
  let sourceCount = texAttributeNames.length;
  let targetTextureType = null;
  const samplerTextureMap = {};
  let updatedVs = vs;
  let finalInject = {};

  if (sourceCount > 0 || targetTextureVarying) {
    const vsLines = updatedVs.split('\n');
    const updateVsLines = vsLines.slice();
    vsLines.forEach((line, index, lines) => {
      if (sourceCount > 0) {
        const updated = processAttributeDefinition(line, sourceTextureMap);

        if (updated) {
          const {
            updatedLine,
            inject
          } = updated;
          updateVsLines[index] = updatedLine;
          finalInject = combineInjects([finalInject, inject]);
          Object.assign(samplerTextureMap, updated.samplerTextureMap);
          sourceCount--;
        }
      }

      if (targetTextureVarying && !targetTextureType) {
        targetTextureType = getVaryingType(line, targetTextureVarying);
      }
    });

    if (targetTextureVarying) {
      assert(targetTexture);
      const sizeName = "".concat(SIZE_UNIFORM_PREFIX).concat(targetTextureVarying);
      const uniformDeclaration = "uniform vec2 ".concat(sizeName, ";\n");
      const posInstructions = "     vec2 ".concat(VS_POS_VARIABLE, " = transform_getPos(").concat(sizeName, ");\n     gl_Position = vec4(").concat(VS_POS_VARIABLE, ", 0, 1.);\n");
      const inject = {
        'vs:#decl': uniformDeclaration,
        'vs:#main-start': posInstructions
      };
      finalInject = combineInjects([finalInject, inject]);
    }

    updatedVs = updateVsLines.join('\n');
  }

  return {
    vs: updatedVs,
    targetTextureType,
    inject: finalInject,
    samplerTextureMap
  };
}
export function getSizeUniforms({
  sourceTextureMap,
  targetTextureVarying,
  targetTexture
}) {
  const uniforms = {};
  let width;
  let height;

  if (targetTextureVarying) {
    ({
      width,
      height
    } = targetTexture);
    uniforms["".concat(SIZE_UNIFORM_PREFIX).concat(targetTextureVarying)] = [width, height];
  }

  for (const textureName in sourceTextureMap) {
    ({
      width,
      height
    } = sourceTextureMap[textureName]);
    uniforms["".concat(SIZE_UNIFORM_PREFIX).concat(textureName)] = [width, height];
  }

  return uniforms;
}

function getAttributeDefinition(line) {
  return getQualifierDetails(line, ['attribute', 'in']);
}

function getSamplerDeclerations(textureName) {
  const samplerName = "".concat(SAMPLER_UNIFORM_PREFIX).concat(textureName);
  const sizeName = "".concat(SIZE_UNIFORM_PREFIX).concat(textureName);
  const uniformDeclerations = "  uniform sampler2D ".concat(samplerName, ";\n  uniform vec2 ").concat(sizeName, ";");
  return {
    samplerName,
    sizeName,
    uniformDeclerations
  };
}

export function getVaryingType(line, varying) {
  const qualaiferDetails = getQualifierDetails(line, ['varying', 'out']);

  if (!qualaiferDetails) {
    return null;
  }

  return qualaiferDetails.name === varying ? qualaiferDetails.type : null;
}
export function processAttributeDefinition(line, textureMap) {
  const samplerTextureMap = {};
  const attributeData = getAttributeDefinition(line);

  if (!attributeData) {
    return null;
  }

  const {
    type,
    name
  } = attributeData;

  if (name && textureMap[name]) {
    const updatedLine = "// ".concat(line, " => Replaced by Transform with a sampler");
    const {
      samplerName,
      sizeName,
      uniformDeclerations
    } = getSamplerDeclerations(name);
    const channels = typeToChannelSuffix(type);
    const sampleInstruction = "  ".concat(type, " ").concat(name, " = transform_getInput(").concat(samplerName, ", ").concat(sizeName, ").").concat(channels, ";\n");
    samplerTextureMap[samplerName] = name;
    const inject = {
      'vs:#decl': uniformDeclerations,
      'vs:#main-start': sampleInstruction
    };
    return {
      updatedLine,
      inject,
      samplerTextureMap
    };
  }

  return null;
}
//# sourceMappingURL=transform-shader-utils.js.map