import { cloneTextureFrom, readPixelsToArray, getShaderVersion, Buffer, Texture2D, Framebuffer } from '@luma.gl/webgl';
import { _transform as transformModule, getPassthroughFS, typeToChannelCount, combineInjects } from '@luma.gl/shadertools';
import { updateForTextures, getSizeUniforms } from './transform-shader-utils';
const SRC_TEX_PARAMETER_OVERRIDES = {
  [10241]: 9728,
  [10240]: 9728,
  [10242]: 33071,
  [10243]: 33071
};
const FS_OUTPUT_VARIABLE = 'transform_output';
export default class TextureTransform {
  constructor(gl, props = {}) {
    this.gl = gl;
    this.currentIndex = 0;
    this._swapTexture = null;
    this.targetTextureVarying = null;
    this.targetTextureType = null;
    this.samplerTextureMap = null;
    this.bindings = [];
    this.resources = {};

    this._initialize(props);

    Object.seal(this);
  }

  updateModelProps(props = {}) {
    const updatedModelProps = this._processVertexShader(props);

    return Object.assign({}, props, updatedModelProps);
  }

  getDrawOptions(opts = {}) {
    const {
      sourceBuffers,
      sourceTextures,
      framebuffer,
      targetTexture
    } = this.bindings[this.currentIndex];
    const attributes = Object.assign({}, sourceBuffers, opts.attributes);
    const uniforms = Object.assign({}, opts.uniforms);
    const parameters = Object.assign({}, opts.parameters);
    let discard = opts.discard;

    if (this.hasSourceTextures || this.hasTargetTexture) {
      attributes.transform_elementID = this.elementIDBuffer;

      for (const sampler in this.samplerTextureMap) {
        const textureName = this.samplerTextureMap[sampler];
        uniforms[sampler] = sourceTextures[textureName];
      }

      this._setSourceTextureParameters();

      const sizeUniforms = getSizeUniforms({
        sourceTextureMap: sourceTextures,
        targetTextureVarying: this.targetTextureVarying,
        targetTexture
      });
      Object.assign(uniforms, sizeUniforms);
    }

    if (this.hasTargetTexture) {
      discard = false;
      parameters.viewport = [0, 0, framebuffer.width, framebuffer.height];
    }

    return {
      attributes,
      framebuffer,
      uniforms,
      discard,
      parameters
    };
  }

  swap() {
    if (this._swapTexture) {
      this.currentIndex = this._getNextIndex();
      return true;
    }

    return false;
  }

  update(opts = {}) {
    this._setupTextures(opts);
  }

  getTargetTexture() {
    const {
      targetTexture
    } = this.bindings[this.currentIndex];
    return targetTexture;
  }

  getData({
    packed = false
  } = {}) {
    const {
      framebuffer
    } = this.bindings[this.currentIndex];
    const pixels = readPixelsToArray(framebuffer);

    if (!packed) {
      return pixels;
    }

    const ArrayType = pixels.constructor;
    const channelCount = typeToChannelCount(this.targetTextureType);
    const packedPixels = new ArrayType(pixels.length * channelCount / 4);
    let packCount = 0;

    for (let i = 0; i < pixels.length; i += 4) {
      for (let j = 0; j < channelCount; j++) {
        packedPixels[packCount++] = pixels[i + j];
      }
    }

    return packedPixels;
  }

  getFramebuffer() {
    const currentResources = this.bindings[this.currentIndex];
    return currentResources.framebuffer;
  }

  delete() {
    if (this.ownTexture) {
      this.ownTexture.delete();
    }

    if (this.elementIDBuffer) {
      this.elementIDBuffer.delete();
    }
  }

  _initialize(props = {}) {
    const {
      _targetTextureVarying,
      _swapTexture
    } = props;
    this._swapTexture = _swapTexture;
    this.targetTextureVarying = _targetTextureVarying;
    this.hasTargetTexture = _targetTextureVarying;

    this._setupTextures(props);
  }

  _createTargetTexture(props) {
    const {
      sourceTextures,
      textureOrReference
    } = props;

    if (textureOrReference instanceof Texture2D) {
      return textureOrReference;
    }

    const refTexture = sourceTextures[textureOrReference];

    if (!refTexture) {
      return null;
    }

    this._targetRefTexName = textureOrReference;
    return this._createNewTexture(refTexture);
  }

  _setupTextures(props = {}) {
    const {
      sourceBuffers,
      _sourceTextures = {},
      _targetTexture
    } = props;

    const targetTexture = this._createTargetTexture({
      sourceTextures: _sourceTextures,
      textureOrReference: _targetTexture
    });

    this.hasSourceTextures = this.hasSourceTextures || _sourceTextures && Object.keys(_sourceTextures).length > 0;

    this._updateBindings({
      sourceBuffers,
      sourceTextures: _sourceTextures,
      targetTexture
    });

    if ('elementCount' in props) {
      this._updateElementIDBuffer(props.elementCount);
    }
  }

  _updateElementIDBuffer(elementCount) {
    if (typeof elementCount !== 'number' || this.elementCount >= elementCount) {
      return;
    }

    const elementIds = new Float32Array(elementCount);
    elementIds.forEach((_, index, array) => {
      array[index] = index;
    });

    if (!this.elementIDBuffer) {
      this.elementIDBuffer = new Buffer(this.gl, {
        data: elementIds,
        accessor: {
          size: 1
        }
      });
    } else {
      this.elementIDBuffer.setData({
        data: elementIds
      });
    }

    this.elementCount = elementCount;
  }

  _updateBindings(opts) {
    this.bindings[this.currentIndex] = this._updateBinding(this.bindings[this.currentIndex], opts);

    if (this._swapTexture) {
      const {
        sourceTextures,
        targetTexture
      } = this._swapTextures(this.bindings[this.currentIndex]);

      const nextIndex = this._getNextIndex();

      this.bindings[nextIndex] = this._updateBinding(this.bindings[nextIndex], {
        sourceTextures,
        targetTexture
      });
    }
  }

  _updateBinding(binding, opts) {
    const {
      sourceBuffers,
      sourceTextures,
      targetTexture
    } = opts;

    if (!binding) {
      binding = {
        sourceBuffers: {},
        sourceTextures: {},
        targetTexture: null
      };
    }

    Object.assign(binding.sourceTextures, sourceTextures);
    Object.assign(binding.sourceBuffers, sourceBuffers);

    if (targetTexture) {
      binding.targetTexture = targetTexture;
      const {
        width,
        height
      } = targetTexture;
      const {
        framebuffer
      } = binding;

      if (framebuffer) {
        framebuffer.update({
          attachments: {
            [36064]: targetTexture
          },
          resizeAttachments: false
        });
        framebuffer.resize({
          width,
          height
        });
      } else {
        binding.framebuffer = new Framebuffer(this.gl, {
          id: "".concat(this.id || 'transform', "-framebuffer"),
          width,
          height,
          attachments: {
            [36064]: targetTexture
          }
        });
      }
    }

    return binding;
  }

  _setSourceTextureParameters() {
    const index = this.currentIndex;
    const {
      sourceTextures
    } = this.bindings[index];

    for (const name in sourceTextures) {
      sourceTextures[name].setParameters(SRC_TEX_PARAMETER_OVERRIDES);
    }
  }

  _swapTextures(opts) {
    if (!this._swapTexture) {
      return null;
    }

    const sourceTextures = Object.assign({}, opts.sourceTextures);
    sourceTextures[this._swapTexture] = opts.targetTexture;
    const targetTexture = opts.sourceTextures[this._swapTexture];
    return {
      sourceTextures,
      targetTexture
    };
  }

  _createNewTexture(refTexture) {
    const texture = cloneTextureFrom(refTexture, {
      parameters: {
        [10241]: 9728,
        [10240]: 9728,
        [10242]: 33071,
        [10243]: 33071
      },
      pixelStore: {
        [37440]: false
      }
    });

    if (this.ownTexture) {
      this.ownTexture.delete();
    }

    this.ownTexture = texture;
    return texture;
  }

  _getNextIndex() {
    return (this.currentIndex + 1) % 2;
  }

  _processVertexShader(props = {}) {
    const {
      sourceTextures,
      targetTexture
    } = this.bindings[this.currentIndex];
    const {
      vs,
      uniforms,
      targetTextureType,
      inject,
      samplerTextureMap
    } = updateForTextures({
      vs: props.vs,
      sourceTextureMap: sourceTextures,
      targetTextureVarying: this.targetTextureVarying,
      targetTexture
    });
    const combinedInject = combineInjects([props.inject || {}, inject]);
    this.targetTextureType = targetTextureType;
    this.samplerTextureMap = samplerTextureMap;
    const fs = props._fs || getPassthroughFS({
      version: getShaderVersion(vs),
      input: this.targetTextureVarying,
      inputType: targetTextureType,
      output: FS_OUTPUT_VARIABLE
    });
    const modules = this.hasSourceTextures || this.targetTextureVarying ? [transformModule].concat(props.modules || []) : props.modules;
    return {
      vs,
      fs,
      modules,
      uniforms,
      inject: combinedInject
    };
  }

}
//# sourceMappingURL=texture-transform.js.map