import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import _classCallCheck from "@babel/runtime/helpers/esm/classCallCheck";
import _createClass from "@babel/runtime/helpers/esm/createClass";
import { isWebGL } from '@luma.gl/gltools';
import ProgramManager from './program-manager';
import { Program, VertexArray, clear as _clear, TransformFeedback, Buffer, log, isObjectEmpty, uid, assert } from '@luma.gl/webgl';
import { getDebugTableForUniforms, getDebugTableForVertexArray, getDebugTableForProgramConfiguration } from '@luma.gl/webgl';
import { getBuffersFromGeometry } from './model-utils';
var LOG_DRAW_PRIORITY = 2;
var LOG_DRAW_TIMEOUT = 10000;
var ERR_MODEL_PARAMS = 'Model needs drawMode and vertexCount';

var NOOP = function NOOP() {};

var DRAW_PARAMS = {};

var Model = function () {
  function Model(gl) {
    var props = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    _classCallCheck(this, Model);

    var _props$id = props.id,
        id = _props$id === void 0 ? uid('model') : _props$id;
    assert(isWebGL(gl));
    this.id = id;
    this.gl = gl;
    this.id = props.id || uid('Model');
    this.lastLogTime = 0;
    this.initialize(props);
  }

  _createClass(Model, [{
    key: "initialize",
    value: function initialize(props) {
      this.props = {};
      this.programManager = props.programManager || ProgramManager.getDefaultProgramManager(this.gl);
      this._programManagerState = -1;
      this._managedProgram = false;
      var _props$program = props.program,
          program = _props$program === void 0 ? null : _props$program,
          vs = props.vs,
          fs = props.fs,
          modules = props.modules,
          defines = props.defines,
          inject = props.inject,
          varyings = props.varyings,
          bufferMode = props.bufferMode,
          transpileToGLSL100 = props.transpileToGLSL100;
      this.programProps = {
        program: program,
        vs: vs,
        fs: fs,
        modules: modules,
        defines: defines,
        inject: inject,
        varyings: varyings,
        bufferMode: bufferMode,
        transpileToGLSL100: transpileToGLSL100
      };
      this.program = null;
      this.vertexArray = null;
      this._programDirty = true;
      this.userData = {};
      this.needsRedraw = true;
      this._attributes = {};
      this.attributes = {};
      this.uniforms = {};
      this.pickable = true;

      this._checkProgram();

      this.setUniforms(Object.assign({}, this.getModuleUniforms(props.moduleSettings)));
      this.drawMode = props.drawMode !== undefined ? props.drawMode : 4;
      this.vertexCount = props.vertexCount || 0;
      this.geometryBuffers = {};
      this.isInstanced = props.isInstanced || props.instanced || props.instanceCount > 0;

      this._setModelProps(props);

      this.geometry = {};
      assert(this.drawMode !== undefined && Number.isFinite(this.vertexCount), ERR_MODEL_PARAMS);
    }
  }, {
    key: "setProps",
    value: function setProps(props) {
      this._setModelProps(props);
    }
  }, {
    key: "delete",
    value: function _delete() {
      for (var key in this._attributes) {
        if (this._attributes[key] !== this.attributes[key]) {
          this._attributes[key]["delete"]();
        }
      }

      if (this._managedProgram) {
        this.programManager.release(this.program);
        this._managedProgram = false;
      }

      this.vertexArray["delete"]();

      this._deleteGeometryBuffers();
    }
  }, {
    key: "getDrawMode",
    value: function getDrawMode() {
      return this.drawMode;
    }
  }, {
    key: "getVertexCount",
    value: function getVertexCount() {
      return this.vertexCount;
    }
  }, {
    key: "getInstanceCount",
    value: function getInstanceCount() {
      return this.instanceCount;
    }
  }, {
    key: "getAttributes",
    value: function getAttributes() {
      return this.attributes;
    }
  }, {
    key: "getProgram",
    value: function getProgram() {
      return this.program;
    }
  }, {
    key: "setProgram",
    value: function setProgram(props) {
      var program = props.program,
          vs = props.vs,
          fs = props.fs,
          modules = props.modules,
          defines = props.defines,
          inject = props.inject,
          varyings = props.varyings,
          bufferMode = props.bufferMode,
          transpileToGLSL100 = props.transpileToGLSL100;
      this.programProps = {
        program: program,
        vs: vs,
        fs: fs,
        modules: modules,
        defines: defines,
        inject: inject,
        varyings: varyings,
        bufferMode: bufferMode,
        transpileToGLSL100: transpileToGLSL100
      };
      this._programDirty = true;
    }
  }, {
    key: "getUniforms",
    value: function getUniforms() {
      return this.uniforms;
    }
  }, {
    key: "setDrawMode",
    value: function setDrawMode(drawMode) {
      this.drawMode = drawMode;
      return this;
    }
  }, {
    key: "setVertexCount",
    value: function setVertexCount(vertexCount) {
      assert(Number.isFinite(vertexCount));
      this.vertexCount = vertexCount;
      return this;
    }
  }, {
    key: "setInstanceCount",
    value: function setInstanceCount(instanceCount) {
      assert(Number.isFinite(instanceCount));
      this.instanceCount = instanceCount;
      return this;
    }
  }, {
    key: "setGeometry",
    value: function setGeometry(geometry) {
      this.drawMode = geometry.drawMode;
      this.vertexCount = geometry.getVertexCount();

      this._deleteGeometryBuffers();

      this.geometryBuffers = getBuffersFromGeometry(this.gl, geometry);
      this.vertexArray.setAttributes(this.geometryBuffers);
      return this;
    }
  }, {
    key: "setAttributes",
    value: function setAttributes() {
      var attributes = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

      if (isObjectEmpty(attributes)) {
        return this;
      }

      var normalizedAttributes = {};

      for (var name in attributes) {
        var attribute = attributes[name];
        normalizedAttributes[name] = attribute.getValue ? attribute.getValue() : attribute;
      }

      this.vertexArray.setAttributes(normalizedAttributes);
      return this;
    }
  }, {
    key: "setUniforms",
    value: function setUniforms() {
      var uniforms = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      Object.assign(this.uniforms, uniforms);
      return this;
    }
  }, {
    key: "getModuleUniforms",
    value: function getModuleUniforms(opts) {
      this._checkProgram();

      var getUniforms = this.programManager.getUniforms(this.program);

      if (getUniforms) {
        return getUniforms(opts);
      }

      return {};
    }
  }, {
    key: "updateModuleSettings",
    value: function updateModuleSettings(opts) {
      var uniforms = this.getModuleUniforms(opts || {});
      return this.setUniforms(uniforms);
    }
  }, {
    key: "clear",
    value: function clear(opts) {
      _clear(this.program.gl, opts);

      return this;
    }
  }, {
    key: "draw",
    value: function draw() {
      var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

      this._checkProgram();

      var _opts$moduleSettings = opts.moduleSettings,
          moduleSettings = _opts$moduleSettings === void 0 ? null : _opts$moduleSettings,
          framebuffer = opts.framebuffer,
          _opts$uniforms = opts.uniforms,
          uniforms = _opts$uniforms === void 0 ? {} : _opts$uniforms,
          _opts$attributes = opts.attributes,
          attributes = _opts$attributes === void 0 ? {} : _opts$attributes,
          _opts$transformFeedba = opts.transformFeedback,
          transformFeedback = _opts$transformFeedba === void 0 ? this.transformFeedback : _opts$transformFeedba,
          _opts$parameters = opts.parameters,
          parameters = _opts$parameters === void 0 ? {} : _opts$parameters,
          _opts$vertexArray = opts.vertexArray,
          vertexArray = _opts$vertexArray === void 0 ? this.vertexArray : _opts$vertexArray;
      this.setAttributes(attributes);
      this.updateModuleSettings(moduleSettings);
      this.setUniforms(uniforms);
      var logPriority;

      if (log.priority >= LOG_DRAW_PRIORITY) {
        logPriority = this._logDrawCallStart(LOG_DRAW_PRIORITY);
      }

      var drawParams = this.vertexArray.getDrawParams();
      var _this$props = this.props,
          _this$props$isIndexed = _this$props.isIndexed,
          isIndexed = _this$props$isIndexed === void 0 ? drawParams.isIndexed : _this$props$isIndexed,
          _this$props$indexType = _this$props.indexType,
          indexType = _this$props$indexType === void 0 ? drawParams.indexType : _this$props$indexType,
          _this$props$indexOffs = _this$props.indexOffset,
          indexOffset = _this$props$indexOffs === void 0 ? drawParams.indexOffset : _this$props$indexOffs,
          _this$props$vertexArr = _this$props.vertexArrayInstanced,
          vertexArrayInstanced = _this$props$vertexArr === void 0 ? drawParams.isInstanced : _this$props$vertexArr;

      if (vertexArrayInstanced && !this.isInstanced) {
        log.warn('Found instanced attributes on non-instanced model', this.id)();
      }

      var isInstanced = this.isInstanced,
          instanceCount = this.instanceCount;
      var _this$props2 = this.props,
          _this$props2$onBefore = _this$props2.onBeforeRender,
          onBeforeRender = _this$props2$onBefore === void 0 ? NOOP : _this$props2$onBefore,
          _this$props2$onAfterR = _this$props2.onAfterRender,
          onAfterRender = _this$props2$onAfterR === void 0 ? NOOP : _this$props2$onAfterR;
      onBeforeRender();
      this.program.setUniforms(this.uniforms);
      var didDraw = this.program.draw(Object.assign(DRAW_PARAMS, opts, {
        logPriority: logPriority,
        uniforms: null,
        framebuffer: framebuffer,
        parameters: parameters,
        drawMode: this.getDrawMode(),
        vertexCount: this.getVertexCount(),
        vertexArray: vertexArray,
        transformFeedback: transformFeedback,
        isIndexed: isIndexed,
        indexType: indexType,
        isInstanced: isInstanced,
        instanceCount: instanceCount,
        offset: isIndexed ? indexOffset : 0
      }));
      onAfterRender();

      if (log.priority >= LOG_DRAW_PRIORITY) {
        this._logDrawCallEnd(logPriority, vertexArray, framebuffer);
      }

      return didDraw;
    }
  }, {
    key: "transform",
    value: function transform() {
      var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      var _opts$discard = opts.discard,
          discard = _opts$discard === void 0 ? true : _opts$discard,
          feedbackBuffers = opts.feedbackBuffers,
          _opts$unbindModels = opts.unbindModels,
          unbindModels = _opts$unbindModels === void 0 ? [] : _opts$unbindModels;
      var parameters = opts.parameters;

      if (feedbackBuffers) {
        this._setFeedbackBuffers(feedbackBuffers);
      }

      if (discard) {
        parameters = Object.assign({}, parameters, _defineProperty({}, 35977, discard));
      }

      unbindModels.forEach(function (model) {
        return model.vertexArray.unbindBuffers();
      });

      try {
        this.draw(Object.assign({}, opts, {
          parameters: parameters
        }));
      } finally {
        unbindModels.forEach(function (model) {
          return model.vertexArray.bindBuffers();
        });
      }

      return this;
    }
  }, {
    key: "render",
    value: function render() {
      var uniforms = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      log.warn('Model.render() is deprecated. Use Model.setUniforms() and Model.draw()')();
      return this.setUniforms(uniforms).draw();
    }
  }, {
    key: "_setModelProps",
    value: function _setModelProps(props) {
      Object.assign(this.props, props);

      if ('uniforms' in props) {
        this.setUniforms(props.uniforms);
      }

      if ('pickable' in props) {
        this.pickable = props.pickable;
      }

      if ('instanceCount' in props) {
        this.instanceCount = props.instanceCount;
      }

      if ('geometry' in props) {
        this.setGeometry(props.geometry);
      }

      if ('attributes' in props) {
        this.setAttributes(props.attributes);
      }

      if ('_feedbackBuffers' in props) {
        this._setFeedbackBuffers(props._feedbackBuffers);
      }
    }
  }, {
    key: "_checkProgram",
    value: function _checkProgram() {
      var shaderCache = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;
      var needsUpdate = this._programDirty || this.programManager.stateHash !== this._programManagerState;

      if (!needsUpdate) {
        return;
      }

      var program = this.programProps.program;

      if (program) {
        this._managedProgram = false;
      } else {
        var _this$programProps = this.programProps,
            vs = _this$programProps.vs,
            fs = _this$programProps.fs,
            modules = _this$programProps.modules,
            inject = _this$programProps.inject,
            defines = _this$programProps.defines,
            varyings = _this$programProps.varyings,
            bufferMode = _this$programProps.bufferMode,
            transpileToGLSL100 = _this$programProps.transpileToGLSL100;
        program = this.programManager.get({
          vs: vs,
          fs: fs,
          modules: modules,
          inject: inject,
          defines: defines,
          varyings: varyings,
          bufferMode: bufferMode,
          transpileToGLSL100: transpileToGLSL100
        });

        if (this.program && this._managedProgram) {
          this.programManager.release(this.program);
        }

        this._programManagerState = this.programManager.stateHash;
        this._managedProgram = true;
      }

      assert(program instanceof Program, 'Model needs a program');
      this._programDirty = false;

      if (program === this.program) {
        return;
      }

      this.program = program;

      if (this.vertexArray) {
        this.vertexArray.setProps({
          program: this.program,
          attributes: this.vertexArray.attributes
        });
      } else {
        this.vertexArray = new VertexArray(this.gl, {
          program: this.program
        });
      }

      this.setUniforms(Object.assign({}, this.getModuleUniforms()));
    }
  }, {
    key: "_deleteGeometryBuffers",
    value: function _deleteGeometryBuffers() {
      for (var name in this.geometryBuffers) {
        var buffer = this.geometryBuffers[name][0] || this.geometryBuffers[name];

        if (buffer instanceof Buffer) {
          buffer["delete"]();
        }
      }
    }
  }, {
    key: "_setAnimationProps",
    value: function _setAnimationProps(animationProps) {
      if (this.animated) {
        assert(animationProps, 'Model.draw(): animated uniforms but no animationProps');

        var animatedUniforms = this._evaluateAnimateUniforms(animationProps);

        Object.assign(this.uniforms, animatedUniforms);
      }
    }
  }, {
    key: "_setFeedbackBuffers",
    value: function _setFeedbackBuffers() {
      var feedbackBuffers = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

      if (isObjectEmpty(feedbackBuffers)) {
        return this;
      }

      var gl = this.program.gl;
      this.transformFeedback = this.transformFeedback || new TransformFeedback(gl, {
        program: this.program
      });
      this.transformFeedback.setBuffers(feedbackBuffers);
      return this;
    }
  }, {
    key: "_logDrawCallStart",
    value: function _logDrawCallStart(logLevel) {
      var logDrawTimeout = logLevel > 3 ? 0 : LOG_DRAW_TIMEOUT;

      if (Date.now() - this.lastLogTime < logDrawTimeout) {
        return undefined;
      }

      this.lastLogTime = Date.now();
      log.group(LOG_DRAW_PRIORITY, ">>> DRAWING MODEL ".concat(this.id), {
        collapsed: log.level <= 2
      })();
      return logLevel;
    }
  }, {
    key: "_logDrawCallEnd",
    value: function _logDrawCallEnd(logLevel, vertexArray, uniforms, framebuffer) {
      if (logLevel === undefined) {
        return;
      }

      var attributeTable = getDebugTableForVertexArray({
        vertexArray: vertexArray,
        header: "".concat(this.id, " attributes"),
        attributes: this._attributes
      });

      var _getDebugTableForUnif = getDebugTableForUniforms({
        header: "".concat(this.id, " uniforms"),
        program: this.program,
        uniforms: Object.assign({}, this.program.uniforms, uniforms)
      }),
          uniformTable = _getDebugTableForUnif.table,
          unusedTable = _getDebugTableForUnif.unusedTable,
          unusedCount = _getDebugTableForUnif.unusedCount;

      var _getDebugTableForUnif2 = getDebugTableForUniforms({
        header: "".concat(this.id, " uniforms"),
        program: this.program,
        uniforms: Object.assign({}, this.program.uniforms, uniforms),
        undefinedOnly: true
      }),
          missingTable = _getDebugTableForUnif2.table,
          missingCount = _getDebugTableForUnif2.count;

      if (missingCount > 0) {
        log.log('MISSING UNIFORMS', Object.keys(missingTable))();
      }

      if (unusedCount > 0) {
        log.log('UNUSED UNIFORMS', Object.keys(unusedTable))();
      }

      var configTable = getDebugTableForProgramConfiguration(this.vertexArray.configuration);
      log.table(logLevel, attributeTable)();
      log.table(logLevel, uniformTable)();
      log.table(logLevel + 1, configTable)();

      if (framebuffer) {
        framebuffer.log({
          logLevel: LOG_DRAW_PRIORITY,
          message: "Rendered to ".concat(framebuffer.id)
        });
      }

      log.groupEnd(LOG_DRAW_PRIORITY, ">>> DRAWING MODEL ".concat(this.id))();
    }
  }]);

  return Model;
}();

export { Model as default };
//# sourceMappingURL=model.js.map