import { Vector3, Matrix4 } from 'math.gl';
import { assert, uid } from '@luma.gl/webgl';
export default class ScenegraphNode {
  constructor(props = {}) {
    const {
      id
    } = props;
    this.id = id || uid(this.constructor.name);
    this.display = true;
    this.position = new Vector3();
    this.rotation = new Vector3();
    this.scale = new Vector3(1, 1, 1);
    this.matrix = new Matrix4();
    this.userData = {};
    this.props = {};

    this._setScenegraphNodeProps(props);
  }

  delete() {}

  setProps(props) {
    this._setScenegraphNodeProps(props);

    return this;
  }

  toString() {
    return "{type: ScenegraphNode, id: ".concat(this.id, ")}");
  }

  setPosition(position) {
    assert(position.length === 3, 'setPosition requires vector argument');
    this.position = position;
    return this;
  }

  setRotation(rotation) {
    assert(rotation.length === 3, 'setRotation requires vector argument');
    this.rotation = rotation;
    return this;
  }

  setScale(scale) {
    assert(scale.length === 3, 'setScale requires vector argument');
    this.scale = scale;
    return this;
  }

  setMatrix(matrix, copyMatrix = true) {
    if (copyMatrix) {
      this.matrix.copy(matrix);
    } else {
      this.matrix = matrix;
    }
  }

  setMatrixComponents({
    position,
    rotation,
    scale,
    update = true
  }) {
    if (position) {
      this.setPosition(position);
    }

    if (rotation) {
      this.setRotation(rotation);
    }

    if (scale) {
      this.setScale(scale);
    }

    if (update) {
      this.updateMatrix();
    }

    return this;
  }

  updateMatrix() {
    const pos = this.position;
    const rot = this.rotation;
    const scale = this.scale;
    this.matrix.identity();
    this.matrix.translate(pos);
    this.matrix.rotateXYZ(rot);
    this.matrix.scale(scale);
    return this;
  }

  update({
    position,
    rotation,
    scale
  } = {}) {
    if (position) {
      this.setPosition(position);
    }

    if (rotation) {
      this.setRotation(rotation);
    }

    if (scale) {
      this.setScale(scale);
    }

    this.updateMatrix();
    return this;
  }

  getCoordinateUniforms(viewMatrix, modelMatrix) {
    assert(viewMatrix);
    modelMatrix = modelMatrix || this.matrix;
    const worldMatrix = new Matrix4(viewMatrix).multiplyRight(modelMatrix);
    const worldInverse = worldMatrix.invert();
    const worldInverseTranspose = worldInverse.transpose();
    return {
      viewMatrix,
      modelMatrix,
      objectMatrix: modelMatrix,
      worldMatrix,
      worldInverseMatrix: worldInverse,
      worldInverseTransposeMatrix: worldInverseTranspose
    };
  }

  _setScenegraphNodeProps(props) {
    if ('display' in props) {
      this.display = props.display;
    }

    if ('position' in props) {
      this.setPosition(props.position);
    }

    if ('rotation' in props) {
      this.setRotation(props.rotation);
    }

    if ('scale' in props) {
      this.setScale(props.scale);
    }

    if ('matrix' in props) {
      this.setMatrix(props.matrix);
    }

    Object.assign(this.props, props);
  }

}
//# sourceMappingURL=scenegraph-node.js.map