import { Buffer, Texture2D, assert } from '@luma.gl/webgl';
import { isWebGL2 } from '@luma.gl/gltools';
import { Transform } from '@luma.gl/engine';
import { default as textureFilterModule } from './texture-filter';
import { POLY_TEX_VS, FILTER_VS } from './shaders';
import * as Polygon from './polygon';
const TEXTURE_SIZE = 512;
export default class GPUPointInPolygon {
  constructor(gl, opts = {}) {
    this.gl = gl;
    assert(isWebGL2(gl));
    this.textureSize = TEXTURE_SIZE;

    this._setupResources();

    this.update(opts);
  }

  update({
    polygons,
    textureSize
  } = {}) {
    if (textureSize) {
      this.textureSize = textureSize;
    }

    if (!polygons || polygons.length === 0) {
      return;
    }

    const {
      vertices,
      indices,
      vertexCount,
      ids
    } = triangulatePolygons(polygons);

    this._updateResources(vertices, indices, ids, vertexCount);
  }

  filter({
    positionBuffer,
    filterValueIndexBuffer,
    count
  }) {
    this.filterTransform.update({
      sourceBuffers: {
        a_position: positionBuffer
      },
      feedbackBuffers: {
        filterValueIndex: filterValueIndexBuffer
      },
      elementCount: count
    });
    const {
      polygonTexture,
      boundingBox
    } = this;
    this.filterTransform.run({
      moduleSettings: {
        boundingBox,
        texture: polygonTexture
      }
    });
  }

  _setupResources() {
    const {
      gl
    } = this;
    this.polygonTexture = new Texture2D(gl, {
      format: 6407,
      type: 5121,
      dataFormat: 6407,
      border: 0,
      mipmaps: false,
      parameters: {
        [10240]: 9728,
        [10241]: 9728,
        [10242]: 33071,
        [10243]: 33071
      }
    });
    this.positionBuffer = new Buffer(gl, {
      accessor: {
        type: 5126,
        size: 2
      }
    });
    this.idBuffer = new Buffer(gl, {
      accessor: {
        type: 5126,
        size: 1
      }
    });
    this.indexBuffer = new Buffer(gl, {
      target: 34963,
      accessor: {
        type: 5123
      }
    });
    this.polyTextureTransform = new Transform(gl, {
      id: "polygon-texture-creation-transform",
      elementCount: 0,
      _targetTexture: this.polygonTexture,
      _targetTextureVarying: 'v_polygonColor',
      vs: POLY_TEX_VS,
      drawMode: 4,
      isIndexed: true,
      sourceBuffers: {
        a_position: this.positionBuffer,
        a_polygonID: this.idBuffer,
        indices: this.indexBuffer
      }
    });
    this.filterTransform = new Transform(gl, {
      id: 'filter transform',
      vs: FILTER_VS,
      modules: [textureFilterModule],
      varyings: ['filterValueIndex']
    });
  }

  _updateResources(vertices, indices, ids, vertexCount) {
    const boundingBox = getBoundingBox(vertices, vertexCount);
    const [xMin, yMin, xMax, yMax] = boundingBox;
    const width = xMax - xMin;
    const height = yMax - yMin;
    const whRatio = width / height;
    const {
      textureSize
    } = this;
    let texWidth = textureSize;
    let texHeight = textureSize;

    if (whRatio > 1) {
      texHeight = texWidth / whRatio;
    } else {
      texWidth = texHeight * whRatio;
    }

    this.boundingBox = boundingBox;
    this.polygonTexture.resize({
      width: texWidth,
      height: texHeight,
      mipmaps: false
    });
    this.positionBuffer.setData(new Float32Array(vertices));
    this.idBuffer.setData(new Float32Array(ids));
    this.indexBuffer.setData(new Uint16Array(indices));
    this.polyTextureTransform.update({
      elementCount: indices.length,
      _targetTexture: this.polygonTexture
    });
    this.polyTextureTransform.run({
      uniforms: {
        boundingBoxOriginSize: [xMin, yMin, width, height]
      }
    });
  }

}

function getBoundingBox(positions, vertexCount) {
  let yMin = Infinity;
  let yMax = -Infinity;
  let xMin = Infinity;
  let xMax = -Infinity;
  let y;
  let x;

  for (let i = 0; i < vertexCount; i++) {
    x = positions[i * 2];
    y = positions[i * 2 + 1];
    yMin = y < yMin ? y : yMin;
    yMax = y > yMax ? y : yMax;
    xMin = x < xMin ? x : xMin;
    xMax = x > xMax ? x : xMax;
  }

  return [xMin, yMin, xMax, yMax];
}

function triangulatePolygons(polygons) {
  const SIZE = 2;
  const vertices = [];
  const indices = [];
  const ids = [];
  let count = 0;
  let polygonId = 0;

  for (let i = 0; i < polygons.length; i++) {
    const normalized = Polygon.normalize(polygons[i], SIZE);
    const curVertices = normalized.positions || normalized;
    const curCount = curVertices.length / SIZE;
    const curIds = new Array(curCount).fill(polygonId);
    vertices.push(...curVertices);
    ids.push(...curIds);
    const curIndices = Polygon.getSurfaceIndices(normalized, SIZE);
    const indexCount = curIndices.length;

    for (let j = 0; j < indexCount; j++) {
      curIndices[j] += count;
    }

    count += curCount;
    indices.push(...curIndices);
    polygonId++;
  }

  assert(count < 65536);
  const vertexCount = Polygon.getVertexCount(vertices, SIZE);
  return {
    vertices,
    indices,
    ids,
    vertexCount
  };
}
//# sourceMappingURL=gpu-point-in-polygon.js.map