import { Buffer, Program } from '@luma.gl/webgl';
import { assembleShaders, fp64 } from '@luma.gl/shadertools';
import { initializeGL, initializeTexTarget, render, getGPUOutput } from '../../../test/gpu-test-utils';
const BUFFER_DATA = new Float32Array([1, 1, -1, 1, 1, -1, -1, -1]);

function fp64ify(a) {
  const hi = Math.fround(a);
  const lo = a - Math.fround(a);
  return new Float32Array([hi, lo]);
}

function getFloat64(upper = 256) {
  return Math.random() * Math.pow(2.0, (Math.random() - 0.5) * upper);
}

export function getRelativeError64(result, reference) {
  const reference64 = reference[0] + reference[1];
  const result64 = result[0] + result[1];

  if (reference64 < 1) {
    return Math.abs(reference64 - result64);
  }

  return Math.abs((reference64 - result64) / reference64);
}
export function getRelativeError(result, reference) {
  return Math.abs((reference - result) / reference);
}

function getBinaryShader(operation) {
  return "attribute vec3 positions;\nuniform vec2 a;\nuniform vec2 b;\nvarying vec4 vColor;\nvoid main(void) {\n  gl_Position = vec4(positions, 1.0);\n  vec2 result = ".concat(operation, "(a, b);\n  vColor = vec4(result.x, result.y, 0.0, 1.0);\n}\n");
}

function getUnaryShader(operation) {
  return "attribute vec3 positions;\nuniform vec2 a;\nvarying vec4 vColor;\nvoid main(void) {\n  gl_Position = vec4(positions, 1.0);\n  vec2 result = ".concat(operation, "(a);\n  vColor = vec4(result.x, result.y, 0.0, 1.0);\n}\n");
}

const FS_RENDER_VCOLOR = "precision highp float;\nvarying vec4 vColor;\nvoid main(void) {\n  gl_FragColor = vColor;\n}\n";

function setupFloatTest(gl, {
  glslFunc,
  binary = false,
  limit = 256,
  op
}) {
  const a = getFloat64(limit);
  const b = getFloat64(limit);
  const expected = op(a, b);
  const a_fp64 = fp64ify(a);
  const b_fp64 = fp64ify(b);
  const expected_fp64 = fp64ify(expected);
  const vs = binary ? getBinaryShader(glslFunc) : getUnaryShader(glslFunc);
  const program = new Program(gl, assembleShaders(gl, {
    vs,
    fs: FS_RENDER_VCOLOR,
    modules: [fp64]
  }));
  program.setBuffers({
    positions: new Buffer(gl, {
      target: 34962,
      data: BUFFER_DATA,
      size: 2
    })
  }).setUniforms({
    a: a_fp64,
    b: b_fp64,
    ONE: 1.0
  });
  return {
    a,
    b,
    expected,
    a_fp64,
    b_fp64,
    expected_fp64,
    program
  };
}

const ITERATIONS = 10;
const EPSILON = 1e-14;
export function testcase(gl, {
  glslFunc,
  binary,
  op,
  limit = 256,
  t
}) {
  for (let idx0 = 0; idx0 < ITERATIONS; idx0++) {
    const {
      a,
      b,
      a_fp64,
      b_fp64,
      expected_fp64
    } = setupFloatTest(gl, {
      glslFunc,
      binary,
      op,
      limit
    });
    render(gl);
    const gpu_result = getGPUOutput(gl);
    const relativeError = getRelativeError64(gpu_result, expected_fp64);
    const args = binary ? "(".concat(a.toPrecision(2), ", ").concat(b.toPrecision(2), ")") : "(".concat(a.toPrecision(2), ")");
    const message = "".concat(glslFunc).concat(args, ": error=").concat(relativeError, ", within ").concat(EPSILON);
    t.ok(relativeError < EPSILON, message);

    if (relativeError >= EPSILON) {
      t.comment(" (tested ".concat(a_fp64.toString(), ", ").concat(b_fp64.toString(), ")"));
    }
  }

  t.end();
}
const canvas = document.createElement('canvas');
canvas.width = 16;
canvas.height = 16;
export const gl = initializeGL(canvas);
initializeTexTarget(gl);

window.onload = () => {
  document.body.appendChild(canvas);
};
//# sourceMappingURL=fp64-test-utils.js.map