"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const builder_1 = require("../core/builder");
const event_loop_spinner_1 = require("./event-loop-spinner");
const objectHash = require("object-hash");
function addLabel(dep, key, value) {
    if (!dep.labels) {
        dep.labels = {};
    }
    dep.labels[key] = value;
}
async function depTreeToGraph(depTree, pkgManagerName) {
    const rootPkg = {
        name: depTree.name,
        version: depTree.version || undefined,
    };
    const pkgManagerInfo = {
        name: pkgManagerName,
    };
    const targetOS = depTree.targetOS;
    if (targetOS) {
        pkgManagerInfo.repositories = [
            {
                alias: `${targetOS.name}:${targetOS.version}`,
            },
        ];
    }
    const builder = new builder_1.DepGraphBuilder(pkgManagerInfo, rootPkg);
    const eventLoopSpinner = new event_loop_spinner_1.EventLoopSpinner();
    await buildGraph(builder, depTree, depTree.name, eventLoopSpinner, true);
    const depGraph = await builder.build();
    return shortenNodeIds(depGraph, eventLoopSpinner);
}
exports.depTreeToGraph = depTreeToGraph;
async function buildGraph(builder, depTree, pkgName, eventLoopSpinner, isRoot = false) {
    const getNodeId = (name, version, hashId) => `${name}@${version || ''}|${hashId}`;
    const depNodesIds = [];
    const hash = crypto.createHash('sha1');
    if (depTree.versionProvenance) {
        hash.update(objectHash(depTree.versionProvenance));
    }
    if (depTree.labels) {
        hash.update(objectHash(depTree.labels));
    }
    const deps = depTree.dependencies || {};
    // filter-out invalid null deps (shouldn't happen - but did...)
    const depNames = Object.keys(deps).filter((d) => !!deps[d]);
    for (const depName of depNames.sort()) {
        const dep = deps[depName];
        const subtreeHash = await buildGraph(builder, dep, depName, eventLoopSpinner);
        const depPkg = {
            name: depName,
            version: dep.version,
        };
        const depNodeId = getNodeId(depPkg.name, depPkg.version, subtreeHash);
        depNodesIds.push(depNodeId);
        const nodeInfo = {};
        if (dep.versionProvenance) {
            nodeInfo.versionProvenance = dep.versionProvenance;
        }
        if (dep.labels) {
            nodeInfo.labels = dep.labels;
        }
        builder.addPkgNode(depPkg, depNodeId, nodeInfo);
        hash.update(depNodeId);
    }
    const treeHash = hash.digest('hex');
    let pkgNodeId;
    if (isRoot) {
        pkgNodeId = builder.rootNodeId;
    }
    else {
        // we don't assume depTree has a .name to support output of `npm list --json`
        const pkg = {
            name: pkgName,
            version: depTree.version,
        };
        pkgNodeId = getNodeId(pkg.name, pkg.version, treeHash);
        const nodeInfo = {};
        if (depTree.versionProvenance) {
            nodeInfo.versionProvenance = depTree.versionProvenance;
        }
        if (depTree.labels) {
            nodeInfo.labels = depTree.labels;
        }
        builder.addPkgNode(pkg, pkgNodeId, nodeInfo);
    }
    for (const depNodeId of depNodesIds) {
        builder.connectDep(pkgNodeId, depNodeId);
    }
    if (depNodesIds.length > 0 && eventLoopSpinner.isStarving()) {
        await eventLoopSpinner.spin();
    }
    return treeHash;
}
async function shortenNodeIds(depGraph, eventLoopSpinner) {
    const builder = new builder_1.DepGraphBuilder(depGraph.pkgManager, depGraph.rootPkg);
    const nodesMap = {};
    // create nodes with shorter ids
    for (const pkg of depGraph.getPkgs()) {
        const nodeIds = depGraph.getPkgNodeIds(pkg);
        for (let i = 0; i < nodeIds.length; i++) {
            const nodeId = nodeIds[i];
            if (nodeId === depGraph.rootNodeId) {
                continue;
            }
            const nodeInfo = depGraph.getNode(nodeId);
            let newNodeId;
            if (nodeIds.length === 1) {
                newNodeId = `${trimAfterLastSep(nodeId, '|')}`;
            }
            else {
                newNodeId = `${trimAfterLastSep(nodeId, '|')}|${i + 1}`;
            }
            nodesMap[nodeId] = newNodeId;
            builder.addPkgNode(pkg, newNodeId, nodeInfo);
        }
        if (eventLoopSpinner.isStarving()) {
            await eventLoopSpinner.spin();
        }
    }
    // connect nodes
    for (const pkg of depGraph.getPkgs()) {
        for (const nodeId of depGraph.getPkgNodeIds(pkg)) {
            for (const depNodeId of depGraph.getNodeDepsNodeIds(nodeId)) {
                const parentNode = nodesMap[nodeId] || nodeId;
                const childNode = nodesMap[depNodeId] || depNodeId;
                builder.connectDep(parentNode, childNode);
            }
        }
        if (eventLoopSpinner.isStarving()) {
            await eventLoopSpinner.spin();
        }
    }
    return builder.build();
}
async function graphToDepTree(depGraphInterface, pkgType, opts = { deduplicateWithinTopLevelDeps: false }) {
    const depGraph = depGraphInterface;
    // TODO: implement cycles support
    if (depGraph.hasCycles()) {
        throw new Error('Conversion to DepTree does not support cyclic graphs yet');
    }
    const eventLoopSpinner = new event_loop_spinner_1.EventLoopSpinner();
    const depTree = await buildSubtree(depGraph, depGraph.rootNodeId, eventLoopSpinner, opts.deduplicateWithinTopLevelDeps ? null : false);
    depTree.type = depGraph.pkgManager.name;
    depTree.packageFormatVersion = constructPackageFormatVersion(pkgType);
    const targetOS = constructTargetOS(depGraph);
    if (targetOS) {
        depTree.targetOS = targetOS;
    }
    return depTree;
}
exports.graphToDepTree = graphToDepTree;
function constructPackageFormatVersion(pkgType) {
    if (pkgType === 'maven') {
        pkgType = 'mvn';
    }
    return `${pkgType}:0.0.1`;
}
function constructTargetOS(depGraph) {
    if (['apk', 'apt', 'deb', 'rpm', 'linux'].indexOf(depGraph.pkgManager.name) ===
        -1) {
        // .targetOS is undefined unless its a linux pkgManager
        return;
    }
    if (!depGraph.pkgManager.repositories ||
        !depGraph.pkgManager.repositories.length ||
        !depGraph.pkgManager.repositories[0].alias) {
        throw new Error('Incomplete .pkgManager, could not create .targetOS');
    }
    const [name, version] = depGraph.pkgManager.repositories[0].alias.split(':');
    return { name, version };
}
async function buildSubtree(depGraph, nodeId, eventLoopSpinner, maybeDeduplicationSet = null) {
    const isRoot = nodeId === depGraph.rootNodeId;
    const nodePkg = depGraph.getNodePkg(nodeId);
    const nodeInfo = depGraph.getNode(nodeId);
    const depTree = {};
    depTree.name = nodePkg.name;
    depTree.version = nodePkg.version;
    if (nodeInfo.versionProvenance) {
        depTree.versionProvenance = nodeInfo.versionProvenance;
    }
    if (nodeInfo.labels) {
        depTree.labels = Object.assign({}, nodeInfo.labels);
    }
    const depInstanceIds = depGraph.getNodeDepsNodeIds(nodeId);
    if (!depInstanceIds || depInstanceIds.length === 0) {
        return depTree;
    }
    if (maybeDeduplicationSet) {
        if (maybeDeduplicationSet.has(nodeId)) {
            if (depInstanceIds.length > 0) {
                addLabel(depTree, 'pruned', 'true');
            }
            return depTree;
        }
        maybeDeduplicationSet.add(nodeId);
    }
    for (const depInstId of depInstanceIds) {
        // Deduplication of nodes occurs only within a scope of a top-level dependency.
        // Therefore, every top-level dep gets an independent set to track duplicates.
        if (isRoot && maybeDeduplicationSet !== false) {
            maybeDeduplicationSet = new Set();
        }
        const subtree = await buildSubtree(depGraph, depInstId, eventLoopSpinner, maybeDeduplicationSet);
        if (!subtree) {
            continue;
        }
        if (!depTree.dependencies) {
            depTree.dependencies = {};
        }
        depTree.dependencies[subtree.name] = subtree;
    }
    if (eventLoopSpinner.isStarving()) {
        await eventLoopSpinner.spin();
    }
    return depTree;
}
function trimAfterLastSep(str, sep) {
    return str.slice(0, str.lastIndexOf(sep));
}
//# sourceMappingURL=index.js.map