"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:core:decoding");
const CodecUtils = __importStar(require("../utils"));
const utils_1 = require("../utils");
const errors_1 = require("../types/errors");
const abi_1 = require("../encode/abi");
const read_1 = __importDefault(require("../read"));
const decode_1 = __importDefault(require("../decode"));
function* decodeVariable(definition, pointer, info) {
    let compiler = info.currentContext.compiler;
    let dataType = utils_1.MakeType.definitionToType(definition, compiler);
    return yield* decode_1.default(dataType, pointer, info); //no need to pass an offset
}
exports.decodeVariable = decodeVariable;
function* decodeCalldata(info) {
    const context = info.currentContext;
    if (context === null) {
        //if we don't know the contract ID, we can't decode
        return {
            kind: "unknown",
            decodingMode: "full",
            data: CodecUtils.Conversion.toHexString(info.state.calldata)
        };
    }
    const compiler = context.compiler;
    const contextHash = context.context;
    const contractType = CodecUtils.ContextUtils.contextToType(context);
    const isConstructor = context.isConstructor;
    const allocations = info.allocations.calldata;
    let allocation;
    let selector;
    //first: is this a creation call?
    if (isConstructor) {
        allocation = allocations.constructorAllocations[contextHash];
    }
    else {
        //skipping any error-handling on this read, as a calldata read can't throw anyway
        let rawSelector = yield* read_1.default({ location: "calldata",
            start: 0,
            length: CodecUtils.EVM.SELECTOR_SIZE
        }, info.state);
        selector = CodecUtils.Conversion.toHexString(rawSelector);
        allocation = allocations.functionAllocations[contextHash][selector];
    }
    if (allocation === undefined) {
        return {
            kind: "message",
            class: contractType,
            abi: context.hasFallback ? CodecUtils.AbiUtils.fallbackAbiForPayability(context.payable) : null,
            data: CodecUtils.Conversion.toHexString(info.state.calldata),
            decodingMode: "full",
        };
    }
    let decodingMode = allocation.allocationMode; //starts out this way, degrades to ABI if necessary
    //you can't map with a generator, so we have to do this map manually
    let decodedArguments = [];
    for (const argumentAllocation of allocation.arguments) {
        let value;
        let dataType = decodingMode === "full" ? argumentAllocation.type : utils_1.abifyType(argumentAllocation.type);
        try {
            value = yield* decode_1.default(dataType, argumentAllocation.pointer, info, {
                abiPointerBase: allocation.offset,
                allowRetry: decodingMode === "full"
            });
        }
        catch (error) {
            if (error instanceof errors_1.StopDecodingError && error.allowRetry && decodingMode === "full") {
                //if a retry happens, we've got to do several things in order to switch to ABI mode:
                //1. mark that we're switching to ABI mode;
                decodingMode = "abi";
                //2. abify all previously decoded values;
                decodedArguments = decodedArguments.map(argumentDecoding => (Object.assign(Object.assign({}, argumentDecoding), { value: utils_1.abifyResult(argumentDecoding.value, info.userDefinedTypes) })));
                //3. retry this particular decode in ABI mode.
                //(no try/catch on this one because we can't actually handle errors here!
                //not that they should be occurring)
                value = yield* decode_1.default(utils_1.abifyType(argumentAllocation.type), //type is now abified!
                argumentAllocation.pointer, info, {
                    abiPointerBase: allocation.offset
                });
                //4. the remaining parameters will then automatically be decoded in ABI mode due to (1),
                //so we don't need to do anything special there.
            }
            else {
                //we shouldn't be getting other exceptions, but if we do, we don't know
                //how to handle them, so uhhhh just rethrow I guess??
                throw error;
            }
        }
        const name = argumentAllocation.name;
        decodedArguments.push(name //deliberate general falsiness test
            ? { name, value }
            : { value });
    }
    if (isConstructor) {
        return {
            kind: "constructor",
            class: contractType,
            arguments: decodedArguments,
            abi: allocation.abi,
            bytecode: CodecUtils.Conversion.toHexString(info.state.calldata.slice(0, allocation.offset)),
            decodingMode
        };
    }
    else {
        return {
            kind: "function",
            class: contractType,
            abi: allocation.abi,
            arguments: decodedArguments,
            selector,
            decodingMode
        };
    }
}
exports.decodeCalldata = decodeCalldata;
//note: this will likely change in the future to take options rather than targetName, but I'm
//leaving it alone for now, as I'm not sure what form those options will take
//(and this is something we're a bit more OK with breaking since it's primarily
//for internal use :) )
function* decodeEvent(info, address, targetName) {
    const allocations = info.allocations.event;
    let rawSelector;
    let selector;
    let contractAllocations; //for non-anonymous events
    let libraryAllocations; //similar
    const topicsCount = info.state.eventtopics.length;
    //yeah, it's not great to read directly from the state like this (bypassing read), but what are you gonna do?
    if (topicsCount > 0) {
        rawSelector = yield* read_1.default({ location: "eventtopic",
            topic: 0
        }, info.state);
        selector = CodecUtils.Conversion.toHexString(rawSelector);
        ({ contract: contractAllocations, library: libraryAllocations } = allocations[topicsCount].bySelector[selector] || { contract: {}, library: {} });
    }
    else {
        //if we don't have a selector, it means we don't have any non-anonymous events
        contractAllocations = {};
        libraryAllocations = {};
    }
    //now: let's get our allocations for anonymous events
    //note: these ones map contract IDs to *arrays* of event allocations, not individual allocations!
    const { contract: contractAnonymousAllocations, library: libraryAnonymousAllocations } = allocations[topicsCount].anonymous;
    //now: what contract are we (probably) dealing with? let's get its code to find out
    const codeBytes = yield {
        type: "code",
        address
    };
    const codeAsHex = CodecUtils.Conversion.toHexString(codeBytes);
    const contractContext = CodecUtils.ContextUtils.findDecoderContext(info.contexts, codeAsHex);
    let possibleContractAllocations; //excludes anonymous events
    let possibleContractAnonymousAllocations;
    if (contractContext) {
        //if we found the contract, maybe it's from that contract
        const contextHash = contractContext.context;
        const contractAllocation = contractAllocations[contextHash];
        const contractAnonymousAllocation = contractAnonymousAllocations[contextHash];
        possibleContractAllocations = contractAllocation
            ? [contractAllocation]
            : [];
        possibleContractAnonymousAllocations = contractAnonymousAllocation || [];
    }
    else {
        //if we couldn't determine the contract, well, we have to assume it's from a library
        possibleContractAllocations = [];
        possibleContractAnonymousAllocations = [];
    }
    //now we get all the library allocations!
    const possibleLibraryAllocations = Object.values(libraryAllocations);
    const possibleLibraryAnonymousAllocations = [].concat(...Object.values(libraryAnonymousAllocations));
    //now we put it all together!
    const possibleAllocations = possibleContractAllocations.concat(possibleLibraryAllocations);
    const possibleAnonymousAllocations = possibleContractAnonymousAllocations.concat(possibleLibraryAnonymousAllocations);
    const possibleAllocationsTotal = possibleAllocations.concat(possibleAnonymousAllocations);
    let decodings = [];
    allocationAttempts: for (const allocation of possibleAllocationsTotal) {
        //first: do a name check so we can skip decoding if name is wrong
        debug("trying allocation: %O", allocation);
        if (targetName !== undefined && allocation.abi.name !== targetName) {
            continue;
        }
        let decodingMode = allocation.allocationMode; //starts out here; degrades to abi if necessary
        const contextHash = allocation.contextHash;
        const attemptContext = info.contexts[contextHash];
        const contractType = CodecUtils.ContextUtils.contextToType(attemptContext);
        //you can't map with a generator, so we have to do this map manually
        let decodedArguments = [];
        for (const argumentAllocation of allocation.arguments) {
            let value;
            //if in full mode, use the allocation's listed data type.
            //if in ABI mode, abify it before use.
            let dataType = decodingMode === "full" ? argumentAllocation.type : utils_1.abifyType(argumentAllocation.type);
            try {
                value = yield* decode_1.default(dataType, argumentAllocation.pointer, info, {
                    strictAbiMode: true,
                    allowRetry: decodingMode === "full" //this option is unnecessary but including for clarity
                });
            }
            catch (error) {
                if (error instanceof errors_1.StopDecodingError && error.allowRetry && decodingMode === "full") {
                    //if a retry happens, we've got to do several things in order to switch to ABI mode:
                    //1. mark that we're switching to ABI mode;
                    decodingMode = "abi";
                    //2. abify all previously decoded values;
                    decodedArguments = decodedArguments.map(argumentDecoding => (Object.assign(Object.assign({}, argumentDecoding), { value: utils_1.abifyResult(argumentDecoding.value, info.userDefinedTypes) })));
                    //3. retry this particular decode in ABI mode.
                    try {
                        value = yield* decode_1.default(utils_1.abifyType(argumentAllocation.type), //type is now abified!
                        argumentAllocation.pointer, info, {
                            strictAbiMode: true,
                        });
                    }
                    catch (_) {
                        //if an error occurred on the retry, this isn't a valid decoding!
                        debug("rejected due to exception on retry");
                        continue allocationAttempts;
                    }
                    //4. the remaining parameters will then automatically be decoded in ABI mode due to (1),
                    //so we don't need to do anything special there.
                }
                else {
                    //if any other sort of error occurred, this isn't a valid decoding!
                    debug("rejected due to exception on first try: %O", error);
                    continue allocationAttempts;
                }
            }
            const name = argumentAllocation.name;
            const indexed = argumentAllocation.pointer.location === "eventtopic";
            decodedArguments.push(name //deliberate general falsiness test
                ? { name, indexed, value }
                : { indexed, value });
        }
        //OK, so, having decoded the result, the question is: does it reencode to the original?
        //first, we have to filter out the indexed arguments, and also get rid of the name information
        const nonIndexedValues = decodedArguments
            .filter(argument => !argument.indexed)
            .map(argument => argument.value);
        //now, we can encode!
        const reEncodedData = abi_1.encodeTupleAbi(nonIndexedValues, info.allocations.abi);
        const encodedData = info.state.eventdata; //again, not great to read this directly, but oh well
        //are they equal?
        if (!CodecUtils.EVM.equalData(reEncodedData, encodedData)) {
            //if not, this allocation doesn't work
            debug("rejected due to [non-indexed] mismatch");
            continue;
        }
        //one last check -- let's check that the indexed arguments match up, too
        const indexedValues = decodedArguments
            .filter(argument => argument.indexed)
            .map(argument => argument.value);
        debug("indexedValues: %O", indexedValues);
        const reEncodedTopics = indexedValues.map(value => abi_1.encodeAbi(value, info.allocations.abi) //NOTE: I should really use a different encoding function here,
        //but I haven't written that function yet
        );
        const encodedTopics = info.state.eventtopics;
        //now: do *these* match?
        const selectorAdjustment = allocation.anonymous ? 0 : 1;
        for (let i = 0; i < reEncodedTopics.length; i++) {
            debug("encodedTopics[i]: %O", encodedTopics[i]);
            if (!CodecUtils.EVM.equalData(reEncodedTopics[i], encodedTopics[i + selectorAdjustment])) {
                debug("rejected due to indexed mismatch");
                continue allocationAttempts;
            }
        }
        //if we've made it here, the allocation works!  hooray!
        debug("allocation accepted!");
        if (allocation.abi.anonymous) {
            decodings.push({
                kind: "anonymous",
                class: contractType,
                abi: allocation.abi,
                arguments: decodedArguments,
                decodingMode
            });
        }
        else {
            decodings.push({
                kind: "event",
                class: contractType,
                abi: allocation.abi,
                arguments: decodedArguments,
                selector,
                decodingMode
            });
        }
    }
    return decodings;
}
exports.decodeEvent = decodeEvent;
//# sourceMappingURL=decoding.js.map