"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:decode:stack");
const CodecUtils = __importStar(require("../utils"));
const utils_1 = require("../utils");
const read_1 = __importDefault(require("../read"));
const value_1 = __importDefault(require("./value"));
const value_2 = require("./value");
const memory_1 = require("./memory");
const storage_1 = require("./storage");
const abi_1 = require("./abi");
function* decodeStack(dataType, pointer, info) {
    let rawValue;
    try {
        rawValue = yield* read_1.default(pointer, info.state);
    }
    catch (error) {
        return {
            type: dataType,
            kind: "error",
            error: error.error
        };
    }
    const literalPointer = { location: "stackliteral", literal: rawValue };
    return yield* decodeLiteral(dataType, literalPointer, info);
}
exports.default = decodeStack;
function* decodeLiteral(dataType, pointer, info) {
    debug("type %O", dataType);
    debug("pointer %o", pointer);
    if (utils_1.TypeUtils.isReferenceType(dataType)) {
        switch (dataType.location) {
            case "memory":
                //first: do we have a memory pointer? if so we can just dispatch to
                //decodeMemoryReference
                return yield* memory_1.decodeMemoryReferenceByAddress(dataType, pointer, info);
            case "storage":
                //next: do we have a storage pointer (which may be a mapping)? if so, we can
                //we dispatch to decodeStorageByAddress
                return yield* storage_1.decodeStorageReferenceByAddress(dataType, pointer, info);
            case "calldata":
                //next: do we have a calldata pointer?
                //if it's a string or bytes, we will interpret the pointer ourself and skip
                //straight to decodeValue.  this is to allow us to correctly handle the
                //case of msg.data used as a mapping key.
                if (dataType.typeClass === "bytes" || dataType.typeClass === "string") {
                    let startAsBN = CodecUtils.Conversion.toBN(pointer.literal.slice(0, CodecUtils.EVM.WORD_SIZE));
                    let lengthAsBN = CodecUtils.Conversion.toBN(pointer.literal.slice(CodecUtils.EVM.WORD_SIZE));
                    let start;
                    let length;
                    try {
                        start = startAsBN.toNumber();
                    }
                    catch (_) {
                        return {
                            type: dataType,
                            kind: "error",
                            error: {
                                kind: "OverlargePointersNotImplementedError",
                                pointerAsBN: startAsBN
                            }
                        };
                    }
                    try {
                        length = lengthAsBN.toNumber();
                    }
                    catch (_) {
                        return {
                            type: dataType,
                            kind: "error",
                            error: {
                                kind: "OverlongArraysAndStringsNotImplementedError",
                                lengthAsBN
                            }
                        };
                    }
                    let newPointer = { location: "calldata", start, length };
                    return yield* value_1.default(dataType, newPointer, info);
                }
                //otherwise, is it a dynamic array?
                if (dataType.typeClass === "array" && dataType.kind === "dynamic") {
                    //in this case, we're actually going to *throw away* the length info,
                    //because it makes the logic simpler -- we'll get the length info back
                    //from calldata
                    let locationOnly = pointer.literal.slice(0, CodecUtils.EVM.WORD_SIZE);
                    //HACK -- in order to read the correct location, we need to add an offset
                    //of -32 (since, again, we're throwing away the length info), so we pass
                    //that in as the "base" value
                    return yield* abi_1.decodeAbiReferenceByAddress(dataType, { location: "stackliteral", literal: locationOnly }, info, { abiPointerBase: -CodecUtils.EVM.WORD_SIZE });
                }
                else {
                    //multivalue case -- this case is straightforward
                    //pass in 0 as the base since this is an absolute pointer
                    //(yeah we don't need to but let's be explicit)
                    return yield* abi_1.decodeAbiReferenceByAddress(dataType, pointer, info, { abiPointerBase: 0 });
                }
        }
    }
    //next: do we have an external function?  these work differently on the stack
    //than elsewhere, so we can't just pass it on to decodeValue.
    if (dataType.typeClass === "function" && dataType.visibility === "external") {
        let address = pointer.literal.slice(0, CodecUtils.EVM.WORD_SIZE);
        let selectorWord = pointer.literal.slice(-CodecUtils.EVM.WORD_SIZE);
        if (!value_2.checkPaddingLeft(address, CodecUtils.EVM.ADDRESS_SIZE)
            || !value_2.checkPaddingLeft(selectorWord, CodecUtils.EVM.SELECTOR_SIZE)) {
            return {
                type: dataType,
                kind: "error",
                error: {
                    kind: "FunctionExternalStackPaddingError",
                    rawAddress: CodecUtils.Conversion.toHexString(address),
                    rawSelector: CodecUtils.Conversion.toHexString(selectorWord)
                }
            };
        }
        let selector = selectorWord.slice(-CodecUtils.EVM.SELECTOR_SIZE);
        return {
            type: dataType,
            kind: "value",
            value: yield* value_2.decodeExternalFunction(address, selector, info)
        };
    }
    //finally, if none of the above hold, we can just dispatch to decodeValue.
    //however, note that because we're on the stack, we use the permissive padding
    //option so that errors won't result due to values with bad padding
    //(of numeric or bytesN type, anyway)
    return yield* value_1.default(dataType, pointer, info, { permissivePadding: true });
}
exports.decodeLiteral = decodeLiteral;
//# sourceMappingURL=stack.js.map