"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:decode:value");
const read_1 = __importDefault(require("../read"));
const CodecUtils = __importStar(require("../utils"));
const utils_1 = require("../utils");
const utf8_1 = __importDefault(require("utf8"));
const errors_1 = require("../types/errors");
function* decodeValue(dataType, pointer, info, options = {}) {
    const { state } = info;
    const { permissivePadding, strictAbiMode: strict } = options; //if these are undefined they'll still be falsy so OK
    let bytes;
    let rawBytes;
    try {
        bytes = yield* read_1.default(pointer, state);
    }
    catch (error) { //error: DecodingError
        debug("segfault, pointer %o, state: %O", pointer, state);
        if (strict) {
            throw new errors_1.StopDecodingError(error.error);
        }
        return {
            type: dataType,
            kind: "error",
            error: error.error
        };
    }
    rawBytes = bytes;
    debug("type %O", dataType);
    debug("pointer %o", pointer);
    switch (dataType.typeClass) {
        case "bool": {
            const numeric = CodecUtils.Conversion.toBN(bytes);
            if (numeric.eqn(0)) {
                return {
                    type: dataType,
                    kind: "value",
                    value: { asBoolean: false }
                };
            }
            else if (numeric.eqn(1)) {
                return {
                    type: dataType,
                    kind: "value",
                    value: { asBoolean: true }
                };
            }
            else {
                let error = {
                    kind: "BoolOutOfRangeError",
                    rawAsBN: numeric
                };
                if (strict) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
        }
        case "uint":
            //first, check padding (if needed)
            if (!permissivePadding && !checkPaddingLeft(bytes, dataType.bits / 8)) {
                let error = {
                    kind: "UintPaddingError",
                    raw: CodecUtils.Conversion.toHexString(bytes)
                };
                if (strict) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
            //now, truncate to appropriate length (keeping the bytes on the right)
            bytes = bytes.slice(-dataType.bits / 8);
            return {
                type: dataType,
                kind: "value",
                value: {
                    asBN: CodecUtils.Conversion.toBN(bytes),
                    rawAsBN: CodecUtils.Conversion.toBN(rawBytes)
                }
            };
        case "int":
            //first, check padding (if needed)
            if (!permissivePadding && !checkPaddingSigned(bytes, dataType.bits / 8)) {
                let error = {
                    kind: "IntPaddingError",
                    raw: CodecUtils.Conversion.toHexString(bytes)
                };
                if (strict) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
            //now, truncate to appropriate length (keeping the bytes on the right)
            bytes = bytes.slice(-dataType.bits / 8);
            return {
                type: dataType,
                kind: "value",
                value: {
                    asBN: CodecUtils.Conversion.toSignedBN(bytes),
                    rawAsBN: CodecUtils.Conversion.toSignedBN(rawBytes)
                }
            };
        case "address":
            if (!permissivePadding && !checkPaddingLeft(bytes, CodecUtils.EVM.ADDRESS_SIZE)) {
                let error = {
                    kind: "AddressPaddingError",
                    raw: CodecUtils.Conversion.toHexString(bytes)
                };
                if (strict) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
            return {
                type: dataType,
                kind: "value",
                value: {
                    asAddress: CodecUtils.Conversion.toAddress(bytes),
                    rawAsHex: CodecUtils.Conversion.toHexString(rawBytes)
                }
            };
        case "contract":
            if (!permissivePadding && !checkPaddingLeft(bytes, CodecUtils.EVM.ADDRESS_SIZE)) {
                let error = {
                    kind: "ContractPaddingError",
                    raw: CodecUtils.Conversion.toHexString(bytes)
                };
                if (strict) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
            const fullType = utils_1.TypeUtils.fullType(dataType, info.userDefinedTypes);
            const contractValueInfo = yield* decodeContract(bytes, info);
            return {
                type: fullType,
                kind: "value",
                value: contractValueInfo
            };
        case "bytes":
            switch (dataType.kind) {
                case "static":
                    //first, check padding (if needed)
                    if (!permissivePadding && !checkPaddingRight(bytes, dataType.length)) {
                        let error = {
                            kind: "BytesPaddingError",
                            raw: CodecUtils.Conversion.toHexString(bytes)
                        };
                        if (strict) {
                            throw new errors_1.StopDecodingError(error);
                        }
                        return {
                            type: dataType,
                            kind: "error",
                            error
                        };
                    }
                    //now, truncate to appropriate length
                    bytes = bytes.slice(0, dataType.length);
                    return {
                        type: dataType,
                        kind: "value",
                        value: {
                            asHex: CodecUtils.Conversion.toHexString(bytes),
                            rawAsHex: CodecUtils.Conversion.toHexString(rawBytes)
                        }
                    };
                case "dynamic":
                    //no need to check padding here
                    return {
                        type: dataType,
                        kind: "value",
                        value: {
                            asHex: CodecUtils.Conversion.toHexString(bytes),
                        }
                    };
            }
        case "string":
            //there is no padding check for strings
            return {
                type: dataType,
                kind: "value",
                value: decodeString(bytes)
            };
        case "function":
            switch (dataType.visibility) {
                case "external":
                    if (!checkPaddingRight(bytes, CodecUtils.EVM.ADDRESS_SIZE + CodecUtils.EVM.SELECTOR_SIZE)) {
                        let error = {
                            kind: "FunctionExternalNonStackPaddingError",
                            raw: CodecUtils.Conversion.toHexString(bytes)
                        };
                        if (strict) {
                            throw new errors_1.StopDecodingError(error);
                        }
                        return {
                            type: dataType,
                            kind: "error",
                            error
                        };
                    }
                    const address = bytes.slice(0, CodecUtils.EVM.ADDRESS_SIZE);
                    const selector = bytes.slice(CodecUtils.EVM.ADDRESS_SIZE, CodecUtils.EVM.ADDRESS_SIZE + CodecUtils.EVM.SELECTOR_SIZE);
                    return {
                        type: dataType,
                        kind: "value",
                        value: yield* decodeExternalFunction(address, selector, info)
                    };
                case "internal":
                    if (strict) {
                        //internal functions don't go in the ABI!
                        //this should never happen, but just to be sure...
                        throw new errors_1.StopDecodingError({ kind: "InternalFunctionInABIError" });
                    }
                    if (!checkPaddingLeft(bytes, 2 * CodecUtils.EVM.PC_SIZE)) {
                        return {
                            type: dataType,
                            kind: "error",
                            error: {
                                kind: "FunctionInternalPaddingError",
                                raw: CodecUtils.Conversion.toHexString(bytes)
                            }
                        };
                    }
                    const deployedPc = bytes.slice(-CodecUtils.EVM.PC_SIZE);
                    const constructorPc = bytes.slice(-CodecUtils.EVM.PC_SIZE * 2, -CodecUtils.EVM.PC_SIZE);
                    return decodeInternalFunction(dataType, deployedPc, constructorPc, info);
            }
            break; //to satisfy TypeScript
        case "enum": {
            const numeric = CodecUtils.Conversion.toBN(bytes);
            const fullType = utils_1.TypeUtils.fullType(dataType, info.userDefinedTypes);
            if (!fullType.options) {
                let error = {
                    kind: "EnumNotFoundDecodingError",
                    type: fullType,
                    rawAsBN: numeric
                };
                if (strict || options.allowRetry) {
                    throw new errors_1.StopDecodingError(error, true);
                    //note that we allow a retry if we couldn't locate the enum type!
                }
                return {
                    type: fullType,
                    kind: "error",
                    error
                };
            }
            const numOptions = fullType.options.length;
            const numBytes = Math.ceil(Math.log2(numOptions) / 8);
            if (numeric.ltn(numOptions)) {
                const name = fullType.options[numeric.toNumber()];
                //NOTE: despite the use of toNumber(), I'm NOT catching exceptions here and returning an
                //error value like elsewhere; I'm just letting this one fail.  Why?  Because if we have
                //an enum with that many options in the first place, we have bigger problems!
                return {
                    type: fullType,
                    kind: "value",
                    value: {
                        name,
                        numericAsBN: numeric
                    }
                };
            }
            else {
                let error = {
                    kind: "EnumOutOfRangeError",
                    type: fullType,
                    rawAsBN: numeric
                };
                if (strict && !checkPaddingLeft(bytes, numBytes)) {
                    //note that second condition -- even in strict mode,
                    //if the enum is merely out of range rather than out of the ABI range,
                    //we do NOT throw an error here!  instead we simply return an error value,
                    //which we normally avoid doing in strict mode.  (the error will be caught
                    //later at the re-encoding step instead.)  why?  because we might be running
                    //in ABI mode, so we may need to abify this "value" rather than just throwing
                    //it out.
                    throw new errors_1.StopDecodingError(error);
                    //note that we do NOT allow a retry here!
                    //if we *can* find the enum type but the value is out of range,
                    //we *know* that it is invalid!
                }
                return {
                    type: fullType,
                    kind: "error",
                    error
                };
            }
        }
        //will have to split these once we actually support fixed-point
        case "fixed": {
            //first, check padding (if needed)
            if (!permissivePadding && !checkPaddingSigned(bytes, dataType.bits / 8)) {
                let error = {
                    kind: "FixedPaddingError",
                    raw: CodecUtils.Conversion.toHexString(bytes)
                };
                if (strict) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
            //now, truncate to appropriate length (keeping the bytes on the right)
            bytes = bytes.slice(-dataType.bits / 8);
            let asBN = CodecUtils.Conversion.toSignedBN(bytes);
            let rawAsBN = CodecUtils.Conversion.toSignedBN(rawBytes);
            let asBig = CodecUtils.Conversion.shiftBigDown(CodecUtils.Conversion.toBig(asBN), dataType.places);
            let rawAsBig = CodecUtils.Conversion.shiftBigDown(CodecUtils.Conversion.toBig(rawAsBN), dataType.places);
            return {
                type: dataType,
                kind: "value",
                value: {
                    asBig,
                    rawAsBig
                }
            };
        }
        case "ufixed": {
            //first, check padding (if needed)
            if (!permissivePadding && !checkPaddingLeft(bytes, dataType.bits / 8)) {
                let error = {
                    kind: "UfixedPaddingError",
                    raw: CodecUtils.Conversion.toHexString(bytes)
                };
                if (strict) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
            //now, truncate to appropriate length (keeping the bytes on the right)
            bytes = bytes.slice(-dataType.bits / 8);
            let asBN = CodecUtils.Conversion.toBN(bytes);
            let rawAsBN = CodecUtils.Conversion.toBN(rawBytes);
            let asBig = CodecUtils.Conversion.shiftBigDown(CodecUtils.Conversion.toBig(asBN), dataType.places);
            let rawAsBig = CodecUtils.Conversion.shiftBigDown(CodecUtils.Conversion.toBig(rawAsBN), dataType.places);
            return {
                type: dataType,
                kind: "value",
                value: {
                    asBig,
                    rawAsBig
                }
            };
        }
    }
}
exports.default = decodeValue;
function decodeString(bytes) {
    //the following line takes our UTF-8 string... and interprets each byte
    //as a UTF-16 bytepair.  Yikes!  Fortunately, we have a library to repair that.
    let badlyEncodedString = String.fromCharCode.apply(undefined, bytes);
    try {
        //this will throw an error if we have malformed UTF-8
        let correctlyEncodedString = utf8_1.default.decode(badlyEncodedString);
        //NOTE: we don't use node's builtin Buffer class to do the UTF-8 decoding
        //here, because that handles malformed UTF-8 by means of replacement characters
        //(U+FFFD).  That loses information.  So we use the utf8 package instead,
        //and... well, see the catch block below.
        return {
            kind: "valid",
            asString: correctlyEncodedString
        };
    }
    catch (_) {
        //we're going to ignore the precise error and just assume it's because
        //the string was malformed (what else could it be?)
        let hexString = CodecUtils.Conversion.toHexString(bytes);
        return {
            kind: "malformed",
            asHex: hexString
        };
    }
}
exports.decodeString = decodeString;
//NOTE that this function returns a ContractValueInfo, not a ContractResult
function* decodeContract(addressBytes, info) {
    return (yield* decodeContractAndContext(addressBytes, info)).contractInfo;
}
exports.decodeContract = decodeContract;
function* decodeContractAndContext(addressBytes, info) {
    let address = CodecUtils.Conversion.toAddress(addressBytes);
    let rawAddress = CodecUtils.Conversion.toHexString(addressBytes);
    let codeBytes = yield {
        type: "code",
        address
    };
    let code = CodecUtils.Conversion.toHexString(codeBytes);
    let context = CodecUtils.ContextUtils.findDecoderContext(info.contexts, code);
    if (context !== null) {
        return {
            context,
            contractInfo: {
                kind: "known",
                address,
                rawAddress,
                class: CodecUtils.ContextUtils.contextToType(context)
            }
        };
    }
    else {
        return {
            context,
            contractInfo: {
                kind: "unknown",
                address,
                rawAddress
            }
        };
    }
}
//note: address can have extra zeroes on the left like elsewhere, but selector should be exactly 4 bytes
//NOTE this again returns a FunctionExternalValueInfo, not a FunctionExternalResult
function* decodeExternalFunction(addressBytes, selectorBytes, info) {
    let { contractInfo: contract, context } = yield* decodeContractAndContext(addressBytes, info);
    let selector = CodecUtils.Conversion.toHexString(selectorBytes);
    if (contract.kind === "unknown") {
        return {
            kind: "unknown",
            contract,
            selector
        };
    }
    let abiEntry = context.abi !== undefined
        ? context.abi[selector]
        : undefined;
    if (abiEntry === undefined) {
        return {
            kind: "invalid",
            contract,
            selector
        };
    }
    return {
        kind: "known",
        contract,
        selector,
        abi: abiEntry
    };
}
exports.decodeExternalFunction = decodeExternalFunction;
//this one works a bit differently -- in order to handle errors, it *does* return a FunctionInternalResult
//also note, I haven't put the same sort of error-handling in this one since it's only intended to run with full info (for now, anyway)
function decodeInternalFunction(dataType, deployedPcBytes, constructorPcBytes, info) {
    let deployedPc = CodecUtils.Conversion.toBN(deployedPcBytes).toNumber();
    let constructorPc = CodecUtils.Conversion.toBN(constructorPcBytes).toNumber();
    let context = CodecUtils.ContextUtils.contextToType(info.currentContext);
    //before anything else: do we even have an internal functions table?
    //if not, we'll just return the info we have without really attemting to decode
    if (!info.internalFunctionsTable) {
        return {
            type: dataType,
            kind: "value",
            value: {
                kind: "unknown",
                context,
                deployedProgramCounter: deployedPc,
                constructorProgramCounter: constructorPc
            }
        };
    }
    //also before we continue: is the PC zero? if so let's just return that
    if (deployedPc === 0 && constructorPc === 0) {
        return {
            type: dataType,
            kind: "value",
            value: {
                kind: "exception",
                context,
                deployedProgramCounter: deployedPc,
                constructorProgramCounter: constructorPc
            }
        };
    }
    //another check: is only the deployed PC zero?
    if (deployedPc === 0 && constructorPc !== 0) {
        return {
            type: dataType,
            kind: "error",
            error: {
                kind: "MalformedInternalFunctionError",
                context,
                deployedProgramCounter: 0,
                constructorProgramCounter: constructorPc
            }
        };
    }
    //one last pre-check: is this a deployed-format pointer in a constructor?
    if (info.currentContext.isConstructor && constructorPc === 0) {
        return {
            type: dataType,
            kind: "error",
            error: {
                kind: "DeployedFunctionInConstructorError",
                context,
                deployedProgramCounter: deployedPc,
                constructorProgramCounter: 0
            }
        };
    }
    //otherwise, we get our function
    let pc = info.currentContext.isConstructor
        ? constructorPc
        : deployedPc;
    let functionEntry = info.internalFunctionsTable[pc];
    if (!functionEntry) {
        //if it's not zero and there's no entry... error!
        return {
            type: dataType,
            kind: "error",
            error: {
                kind: "NoSuchInternalFunctionError",
                context,
                deployedProgramCounter: deployedPc,
                constructorProgramCounter: constructorPc
            }
        };
    }
    if (functionEntry.isDesignatedInvalid) {
        return {
            type: dataType,
            kind: "value",
            value: {
                kind: "exception",
                context,
                deployedProgramCounter: deployedPc,
                constructorProgramCounter: constructorPc
            }
        };
    }
    let name = functionEntry.name;
    let mutability = functionEntry.mutability;
    let definedIn = {
        typeClass: "contract",
        kind: "native",
        id: functionEntry.contractId.toString(),
        typeName: functionEntry.contractName,
        contractKind: functionEntry.contractKind,
        payable: functionEntry.contractPayable
    };
    return {
        type: dataType,
        kind: "value",
        value: {
            kind: "function",
            context,
            deployedProgramCounter: deployedPc,
            constructorProgramCounter: constructorPc,
            name,
            definedIn,
            mutability
        }
    };
}
exports.decodeInternalFunction = decodeInternalFunction;
function checkPaddingRight(bytes, length) {
    let padding = bytes.slice(length); //cut off the first length bytes
    return padding.every(paddingByte => paddingByte === 0);
}
//exporting this one for use in stack.ts
function checkPaddingLeft(bytes, length) {
    let padding = bytes.slice(0, -length); //cut off the last length bytes
    return padding.every(paddingByte => paddingByte === 0);
}
exports.checkPaddingLeft = checkPaddingLeft;
function checkPaddingSigned(bytes, length) {
    let padding = bytes.slice(0, -length); //padding is all but the last length bytes
    let value = bytes.slice(-length); //meanwhile the actual value is those last length bytes
    let signByte = value[0] & 0x80 ? 0xff : 0x00;
    return padding.every(paddingByte => paddingByte === signByte);
}
//# sourceMappingURL=value.js.map