"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:encode:abi");
const conversion_1 = require("../utils/conversion");
const evm_1 = require("../utils/evm");
const abi_1 = require("../allocate/abi");
const lodash_sum_1 = __importDefault(require("lodash.sum"));
const utf8_1 = __importDefault(require("utf8"));
//UGH -- it turns out TypeScript can't handle nested tagged unions
//see: https://github.com/microsoft/TypeScript/issues/18758
//so, I'm just going to have to throw in a bunch of type coercions >_>
//NOTE: Tuple (as opposed to struct) is not supported yet!
//Coming soon though!
function encodeAbi(input, allocations) {
    //errors can't be encoded
    if (input.kind === "error") {
        debug("input: %O", input);
        if (input.error.kind === "IndexedReferenceTypeError") {
            //HACK: errors can't be encoded, *except* for indexed reference parameter errors.
            //really this should go in a different encoding function, not encodeAbi, but I haven't
            //written that function yet.  I'll move this case when I do.
            return conversion_1.Conversion.toBytes(input.error.raw, evm_1.EVM.WORD_SIZE);
        }
        else {
            return undefined;
        }
    }
    let bytes;
    //TypeScript can at least infer in the rest of this that we're looking
    //at a value, not an error!  But that's hardly enough...
    switch (input.type.typeClass) {
        case "mapping":
        case "magic":
            //neither of these can go in the ABI
            return undefined;
        case "uint":
        case "int":
            return conversion_1.Conversion.toBytes(input.value.asBN, evm_1.EVM.WORD_SIZE);
        case "enum":
            return conversion_1.Conversion.toBytes(input.value.numericAsBN, evm_1.EVM.WORD_SIZE);
        case "bool": {
            bytes = new Uint8Array(evm_1.EVM.WORD_SIZE); //is initialized to zeroes
            if (input.value.asBoolean) {
                bytes[evm_1.EVM.WORD_SIZE - 1] = 1;
            }
            return bytes;
        }
        case "bytes":
            bytes = conversion_1.Conversion.toBytes(input.value.asHex);
            switch (input.type.kind) {
                case "static":
                    let padded = new Uint8Array(evm_1.EVM.WORD_SIZE); //initialized to zeroes
                    padded.set(bytes);
                    return padded;
                case "dynamic":
                    return padAndPrependLength(bytes);
            }
        case "address":
            return conversion_1.Conversion.toBytes(input.value.asAddress, evm_1.EVM.WORD_SIZE);
        case "contract":
            return conversion_1.Conversion.toBytes(input.value.address, evm_1.EVM.WORD_SIZE);
        case "string": {
            let coercedInput = input;
            switch (coercedInput.value.kind) {
                case "valid":
                    bytes = stringToBytes(coercedInput.value.asString);
                    break;
                case "malformed":
                    bytes = conversion_1.Conversion.toBytes(coercedInput.value.asHex);
                    break;
            }
            return padAndPrependLength(bytes);
        }
        case "function": {
            switch (input.type.visibility) {
                case "internal":
                    return undefined; //internal functions can't go in the ABI!
                case "external":
                    let coercedInput = input;
                    let encoded = new Uint8Array(evm_1.EVM.WORD_SIZE); //starts filled w/0s
                    let addressBytes = conversion_1.Conversion.toBytes(coercedInput.value.contract.address); //should already be correct length
                    let selectorBytes = conversion_1.Conversion.toBytes(coercedInput.value.selector); //should already be correct length
                    encoded.set(addressBytes);
                    encoded.set(selectorBytes, evm_1.EVM.ADDRESS_SIZE); //set it after the address
                    return encoded;
            }
        }
        case "fixed":
        case "ufixed":
            let bigValue = input.value.asBig;
            let shiftedValue = conversion_1.Conversion.shiftBigUp(bigValue, input.type.places);
            return conversion_1.Conversion.toBytes(shiftedValue, evm_1.EVM.WORD_SIZE);
        case "array": {
            let coercedInput = input;
            if (coercedInput.reference !== undefined) {
                return undefined; //circular values can't be encoded
            }
            let staticEncoding = encodeTupleAbi(coercedInput.value, allocations);
            switch (input.type.kind) {
                case "static":
                    return staticEncoding;
                case "dynamic":
                    let encoded = new Uint8Array(evm_1.EVM.WORD_SIZE + staticEncoding.length); //leave room for length
                    encoded.set(staticEncoding, evm_1.EVM.WORD_SIZE); //again, leave room for length beforehand
                    let lengthBytes = conversion_1.Conversion.toBytes(coercedInput.value.length, evm_1.EVM.WORD_SIZE);
                    encoded.set(lengthBytes); //and now we set the length
                    return encoded;
            }
        }
        case "struct": {
            let coercedInput = input;
            if (coercedInput.reference !== undefined) {
                return undefined; //circular values can't be encoded
            }
            return encodeTupleAbi(coercedInput.value.map(({ value }) => value), allocations);
        }
        case "tuple": {
            //WARNING: This case is written in a way that involves a bunch of unnecessary recomputation!
            //(That may not be apparent from this one line, but it's true)
            //I'm writing it this way anyway for simplicity, to avoid rewriting the encoder
            //However it may be worth revisiting this in the future if performance turns out to be a problem
            return encodeTupleAbi(input.value.map(({ value }) => value), allocations);
        }
    }
}
exports.encodeAbi = encodeAbi;
function stringToBytes(input) {
    input = utf8_1.default.encode(input);
    let bytes = new Uint8Array(input.length);
    for (let i = 0; i < input.length; i++) {
        bytes[i] = input.charCodeAt(i);
    }
    return bytes;
    //NOTE: this will throw an error if the string contained malformed UTF-16!
    //but, well, it shouldn't contain that...
}
exports.stringToBytes = stringToBytes;
function padAndPrependLength(bytes) {
    let length = bytes.length;
    let paddedLength = evm_1.EVM.WORD_SIZE * Math.ceil(length / evm_1.EVM.WORD_SIZE);
    let encoded = new Uint8Array(evm_1.EVM.WORD_SIZE + paddedLength);
    encoded.set(bytes, evm_1.EVM.WORD_SIZE); //start 32 in to leave room for the length beforehand
    let lengthBytes = conversion_1.Conversion.toBytes(length, evm_1.EVM.WORD_SIZE);
    encoded.set(lengthBytes); //and now we set the length
    return encoded;
}
function encodeTupleAbi(tuple, allocations) {
    let elementEncodings = tuple.map(element => encodeAbi(element, allocations));
    if (elementEncodings.some(element => element === undefined)) {
        return undefined;
    }
    let elementSizeInfo = tuple.map(element => abi_1.abiSizeInfo(element.type, allocations));
    //heads and tails here are as discussed in the ABI docs;
    //for a static type the head is the encoding and the tail is empty,
    //for a dynamic type the head is the pointer and the tail is the encoding
    let heads = [];
    let tails = [];
    //but first, we need to figure out where the first tail will start,
    //by adding up the sizes of all the heads (we can easily do this in
    //advance via elementSizeInfo, without needing to know the particular
    //values of the heads)
    let startOfNextTail = lodash_sum_1.default(elementSizeInfo.map(elementInfo => elementInfo.size));
    for (let i = 0; i < tuple.length; i++) {
        let head;
        let tail;
        if (!elementSizeInfo[i].dynamic) {
            //static case
            head = elementEncodings[i];
            tail = new Uint8Array(); //empty array
        }
        else {
            //dynamic case
            head = conversion_1.Conversion.toBytes(startOfNextTail, evm_1.EVM.WORD_SIZE);
            tail = elementEncodings[i];
        }
        heads.push(head);
        tails.push(tail);
        startOfNextTail += tail.length;
    }
    //finally, we need to concatenate everything together!
    //since we're dealing with Uint8Arrays, we have to do this manually
    let totalSize = startOfNextTail;
    let encoded = new Uint8Array(totalSize);
    let position = 0;
    for (let head of heads) {
        encoded.set(head, position);
        position += head.length;
    }
    for (let tail of tails) {
        encoded.set(tail, position);
        position += tail.length;
    }
    return encoded;
}
exports.encodeTupleAbi = encodeTupleAbi;
//# sourceMappingURL=abi.js.map