"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:interface:contract");
const CodecUtils = __importStar(require("../utils"));
const utils_1 = require("../utils");
const Utils = __importStar(require("../utils/interface"));
const bn_js_1 = __importDefault(require("bn.js"));
const storage_1 = require("../allocate/storage");
const decoding_1 = require("../core/decoding");
const storage_2 = require("../utils/storage");
const errors_1 = require("../types/errors");
class TruffleContractDecoder {
    constructor(contract, wireDecoder, address) {
        this.contract = contract;
        this.wireDecoder = wireDecoder;
        this.web3 = wireDecoder.getWeb3();
        this.contexts = wireDecoder.getDeployedContexts();
        this.contractNode = Utils.getContractNode(this.contract);
        if (this.contract.deployedBytecode && this.contract.deployedBytecode !== "0x") {
            const unnormalizedContext = Utils.makeContext(this.contract, this.contractNode);
            this.contextHash = unnormalizedContext.context;
            //we now throw away the unnormalized context, instead fetching the correct one from
            //this.contexts (which is normalized) via the context getter below
        }
        this.allocations = this.wireDecoder.getAllocations();
        if (this.contractNode) {
            this.allocations.storage = storage_1.getStorageAllocations(this.wireDecoder.getReferenceDeclarations(), //redundant stuff will be skipped so this is fine
            { [this.contractNode.id]: this.contractNode }, this.allocations.storage //existing allocations from wire decoder
            );
            debug("done with allocation");
            if (this.allocations.storage[this.contractNode.id]) {
                this.stateVariableReferences = this.allocations.storage[this.contractNode.id].members;
            }
            //if it doesn't exist, we will leave it undefined, and then throw an exception when
            //we attempt to decode
            debug("stateVariableReferences %O", this.stateVariableReferences);
        }
    }
    init() {
        return __awaiter(this, void 0, void 0, function* () {
            this.contractNetwork = (yield this.web3.eth.net.getId()).toString();
        });
    }
    forInstance(address) {
        return __awaiter(this, void 0, void 0, function* () {
            let instanceDecoder = new TruffleContractInstanceDecoder(this, address);
            yield instanceDecoder.init();
            return instanceDecoder;
        });
    }
    decodeTransaction(transaction) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.decodeTransaction(transaction);
        });
    }
    decodeLog(log) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.decodeLog(log);
        });
    }
    decodeLogs(logs) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.decodeLogs(logs);
        });
    }
    events(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.events(options);
        });
    }
    abifyCalldataDecoding(decoding) {
        return this.wireDecoder.abifyCalldataDecoding(decoding);
    }
    abifyLogDecoding(decoding) {
        return this.wireDecoder.abifyLogDecoding(decoding);
    }
    //the following functions are for internal use
    getAllocations() {
        return this.allocations;
    }
    getStateVariableReferences() {
        return this.stateVariableReferences;
    }
    getWireDecoder() {
        return this.wireDecoder;
    }
    getContractInfo() {
        return {
            contract: this.contract,
            contractNode: this.contractNode,
            contractNetwork: this.contractNetwork,
            contextHash: this.contextHash,
        };
    }
}
exports.default = TruffleContractDecoder;
class TruffleContractInstanceDecoder {
    constructor(contractDecoder, address) {
        this.contexts = {}; //deployed contexts only
        this.additionalContexts = {}; //for passing to wire decoder when contract has no deployedBytecode
        this.mappingKeys = [];
        this.storageCache = {};
        this.contractDecoder = contractDecoder;
        if (address !== undefined) {
            this.contractAddress = address;
        }
        this.wireDecoder = this.contractDecoder.getWireDecoder();
        this.web3 = this.wireDecoder.getWeb3();
        this.referenceDeclarations = this.wireDecoder.getReferenceDeclarations();
        this.userDefinedTypes = this.wireDecoder.getUserDefinedTypes();
        this.contexts = this.wireDecoder.getDeployedContexts();
        ({
            contract: this.contract,
            contractNode: this.contractNode,
            contractNetwork: this.contractNetwork,
            contextHash: this.contextHash,
        } = this.contractDecoder.getContractInfo());
        this.allocations = this.contractDecoder.getAllocations();
        this.stateVariableReferences = this.contractDecoder.getStateVariableReferences();
        if (this.contractAddress === undefined) {
            this.contractAddress = this.contract.networks[this.contractNetwork].address;
        }
    }
    init() {
        return __awaiter(this, void 0, void 0, function* () {
            this.contractCode = CodecUtils.Conversion.toHexString(yield this.getCode(this.contractAddress, yield this.web3.eth.getBlockNumber()));
            if (!this.contract.deployedBytecode || this.contract.deployedBytecode === "0x") {
                //if this contract does *not* have the deployedBytecode field, then the decoder core
                //has no way of knowing that contracts or function pointers with its address
                //are of its class; this is an especial problem for function pointers, as it
                //won't be able to determine what the selector points to.
                //so, to get around this, we make an "additional context" for the contract,
                //based on its *actual* deployed bytecode as pulled from the blockchain.
                //This way the decoder core can recognize the address as the class, without us having
                //to make serious modifications to contract decoding.  And while sure this requires
                //a little more work, I mean, it's all cached, so, no big deal.
                let extraContext = Utils.makeContext(this.contract, this.contractNode);
                //now override the binary
                extraContext.binary = this.contractCode;
                this.additionalContexts = { [extraContext.context]: extraContext };
                //the following line only has any effect if we're dealing with a library,
                //since the code we pulled from the blockchain obviously does not have unresolved link references!
                //(it's not strictly necessary even then, but, hey, why not?)
                this.additionalContexts = utils_1.ContextUtils.normalizeContexts(this.additionalContexts);
                //again, since the code did not have unresolved link references, it is safe to just
                //mash these together like I'm about to
                this.contexts = Object.assign(Object.assign({}, this.contexts), this.additionalContexts);
            }
        });
    }
    get context() {
        return this.contexts[this.contextHash];
    }
    checkAllocationSuccess() {
        if (!this.contractNode) {
            throw new errors_1.ContractBeingDecodedHasNoNodeError(this.contract.contractName);
        }
        if (!this.stateVariableReferences) {
            throw new errors_1.ContractAllocationFailedError(this.contractNode.id, this.contract.contractName);
        }
    }
    decodeVariable(variable, block) {
        return __awaiter(this, void 0, void 0, function* () {
            const info = {
                state: {
                    storage: {},
                },
                mappingKeys: this.mappingKeys,
                userDefinedTypes: this.userDefinedTypes,
                allocations: this.allocations,
                contexts: this.contexts,
                currentContext: this.context
            };
            const decoder = decoding_1.decodeVariable(variable.definition, variable.pointer, info);
            let result = decoder.next();
            while (result.done === false) {
                let request = result.value;
                let response;
                switch (request.type) {
                    case "storage":
                        response = yield this.getStorage(this.contractAddress, request.slot, block);
                        break;
                    case "code":
                        response = yield this.getCode(request.address, block);
                        break;
                }
                result = decoder.next(response);
            }
            //at this point, result.value holds the final value
            return {
                name: variable.definition.name,
                class: this.userDefinedTypes[variable.definedIn.id],
                value: result.value,
            };
        });
    }
    state(block = "latest") {
        return __awaiter(this, void 0, void 0, function* () {
            return {
                name: this.contract.contractName,
                code: this.contractCode,
                balanceAsBN: new bn_js_1.default(yield this.web3.eth.getBalance(this.contractAddress, block)),
                nonceAsBN: new bn_js_1.default(yield this.web3.eth.getTransactionCount(this.contractAddress, block)),
            };
        });
    }
    variables(block = "latest") {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkAllocationSuccess();
            let blockNumber = typeof block === "number"
                ? block
                : (yield this.web3.eth.getBlock(block)).number;
            let result = [];
            for (const variable of this.stateVariableReferences) {
                debug("about to decode %s", variable.definition.name);
                const decodedVariable = yield this.decodeVariable(variable, blockNumber);
                debug("decoded");
                result.push(decodedVariable);
            }
            return result;
        });
    }
    //variable may be given by name, ID, or qualified name
    variable(nameOrId, block = "latest") {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkAllocationSuccess();
            let blockNumber = typeof block === "number"
                ? block
                : (yield this.web3.eth.getBlock(block)).number;
            let variable = this.findVariableByNameOrId(nameOrId);
            if (variable === undefined) { //if user put in a bad name
                return undefined;
            }
            return (yield this.decodeVariable(variable, blockNumber)).value;
        });
    }
    findVariableByNameOrId(nameOrId) {
        //case 1: an ID was input
        if (typeof nameOrId === "number" || nameOrId.match(/[0-9]+/)) {
            let id = Number(nameOrId);
            return this.stateVariableReferences.find(({ definition }) => definition.id === nameOrId);
            //there should be exactly one; returns undefined if none
        }
        //case 2: a name was input
        else if (!nameOrId.includes(".")) {
            //we want to search *backwards*, to get most derived version;
            //we use slice().reverse() to clone before reversing since reverse modifies
            return this.stateVariableReferences.slice().reverse().find(({ definition }) => definition.name === nameOrId);
        }
        //case 3: a qualified name was input
        else {
            let [className, variableName] = nameOrId.split(".");
            //again, we'll search backwards, although, uhhh...?
            return this.stateVariableReferences.slice().reverse().find(({ definition, definedIn }) => definition.name === variableName && definedIn.name === className);
        }
    }
    getStorage(address, slot, block) {
        return __awaiter(this, void 0, void 0, function* () {
            //first, set up any preliminary layers as needed
            if (this.storageCache[block] === undefined) {
                this.storageCache[block] = {};
            }
            if (this.storageCache[block][address] === undefined) {
                this.storageCache[block][address] = {};
            }
            //now, if we have it cached, just return it
            if (this.storageCache[block][address][slot.toString()] !== undefined) {
                return this.storageCache[block][address][slot.toString()];
            }
            //otherwise, get it, cache it, and return it
            let word = CodecUtils.Conversion.toBytes(yield this.web3.eth.getStorageAt(address, slot, block), CodecUtils.EVM.WORD_SIZE);
            this.storageCache[block][address][slot.toString()] = word;
            return word;
        });
    }
    getCode(address, block) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.getCode(address, block);
        });
    }
    //EXAMPLE: to watch a.b.c[d][e], use watchMappingKey("a", "b", "c", d, e)
    //(this will watch all ancestors too, or at least ones given by mapping keys)
    //feel free to mix arrays, mappings, and structs here!
    //see the comment on constructSlot for more detail on what forms are accepted
    watchMappingKey(variable, ...indices) {
        this.checkAllocationSuccess();
        let slot = this.constructSlot(variable, ...indices)[0];
        //add mapping key and all ancestors
        debug("slot: %O", slot);
        while (slot !== undefined &&
            this.mappingKeys.every(existingSlot => !storage_2.equalSlots(existingSlot, slot)
            //we put the newness requirement in the while condition rather than a
            //separate if because if we hit one ancestor that's not new, the futher
            //ones won't be either
            )) {
            if (slot.key !== undefined) { //only add mapping keys
                this.mappingKeys = [...this.mappingKeys, slot];
            }
            slot = slot.path;
        }
    }
    //input is similar to watchMappingKey; will unwatch all descendants too
    unwatchMappingKey(variable, ...indices) {
        this.checkAllocationSuccess();
        let slot = this.constructSlot(variable, ...indices)[0];
        if (slot === undefined) {
            return; //not strictly necessary, but may as well
        }
        //remove mapping key and all descendants
        this.mappingKeys = this.mappingKeys.filter(existingSlot => {
            while (existingSlot !== undefined) {
                if (storage_2.equalSlots(existingSlot, slot)) {
                    return false; //if it matches, remove it
                }
                existingSlot = existingSlot.path;
            }
            return true; //if we didn't match, keep the key
        });
    }
    //NOTE: if you decide to add a way to remove a mapping key *without* removing
    //all descendants, you'll need to alter watchMappingKey to use an if rather
    //than a while
    decodeTransaction(transaction) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.decodeTransaction(transaction, this.additionalContexts);
        });
    }
    decodeLog(log) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.decodeLog(log, {}, this.additionalContexts);
        });
    }
    decodeLogs(logs) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.decodeLogs(logs, {}, this.additionalContexts);
        });
    }
    abifyCalldataDecoding(decoding) {
        return this.wireDecoder.abifyCalldataDecoding(decoding);
    }
    abifyLogDecoding(decoding) {
        return this.wireDecoder.abifyLogDecoding(decoding);
    }
    //note: by default restricts address to address of this
    //contract, but you can override this (including by specifying
    //address undefined to not filter by adddress)
    events(options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.wireDecoder.events(Object.assign({ address: this.contractAddress }, options), this.additionalContexts);
        });
    }
    //in addition to returning the slot we want, it also returns a definition
    //used in the recursive call
    //HOW TO USE:
    //variable may be a variable id (number or numeric string) or name (string) or qualified name (also string)
    //struct members may be given either by id (number) or name (string)
    //array indices and numeric mapping keys may be BN, number, or numeric string
    //string mapping keys should be given as strings. duh.
    //bytes mapping keys should be given as hex strings beginning with "0x"
    //address mapping keys are like bytes; checksum case is not required
    //boolean mapping keys may be given either as booleans, or as string "true" or "false"
    constructSlot(variable, ...indices) {
        //base case: we need to locate the variable and its definition
        if (indices.length === 0) {
            let allocation = this.findVariableByNameOrId(variable);
            let definition = allocation.definition;
            let pointer = allocation.pointer;
            if (pointer.location !== "storage") { //i.e., if it's a constant
                return [undefined, undefined];
            }
            return [pointer.range.from.slot, definition];
        }
        //main case
        let parentIndices = indices.slice(0, -1); //remove last index
        let [parentSlot, parentDefinition] = this.constructSlot(variable, ...parentIndices);
        if (parentSlot === undefined) {
            return [undefined, undefined];
        }
        let rawIndex = indices[indices.length - 1];
        let index;
        let key;
        let slot;
        let definition;
        switch (utils_1.Definition.typeClass(parentDefinition)) {
            case "array":
                if (rawIndex instanceof bn_js_1.default) {
                    index = rawIndex.clone();
                }
                else {
                    index = new bn_js_1.default(rawIndex);
                }
                definition = parentDefinition.baseType || parentDefinition.typeName.baseType;
                let size = storage_1.storageSize(definition, this.referenceDeclarations, this.allocations.storage);
                if (!storage_2.isWordsLength(size)) {
                    return [undefined, undefined];
                }
                slot = {
                    path: parentSlot,
                    offset: index.muln(size.words),
                    hashPath: utils_1.Definition.isDynamicArray(parentDefinition)
                };
                break;
            case "mapping":
                let keyDefinition = parentDefinition.keyType || parentDefinition.typeName.keyType;
                key = utils_1.wrapElementaryViaDefinition(rawIndex, keyDefinition, this.contract.compiler);
                definition = parentDefinition.valueType || parentDefinition.typeName.valueType;
                slot = {
                    path: parentSlot,
                    key,
                    offset: new bn_js_1.default(0)
                };
                break;
            case "struct":
                let parentId = utils_1.Definition.typeId(parentDefinition);
                let allocation;
                if (typeof rawIndex === "number") {
                    index = rawIndex;
                    allocation = this.allocations.storage[parentId].members[index];
                    definition = allocation.definition;
                }
                else {
                    allocation = Object.values(this.allocations.storage[parentId].members)
                        .find(({ definition }) => definition.name === rawIndex); //there should be exactly one
                    definition = allocation.definition;
                    index = definition.id; //not really necessary, but may as well
                }
                slot = {
                    path: parentSlot,
                    //need type coercion here -- we know structs don't contain constants but the compiler doesn't
                    offset: allocation.pointer.range.from.slot.offset.clone()
                };
                break;
            default:
                return [undefined, undefined];
        }
        return [slot, definition];
    }
}
exports.TruffleContractInstanceDecoder = TruffleContractInstanceDecoder;
//# sourceMappingURL=contract.js.map