"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:interface:wire");
const CodecUtils = __importStar(require("../utils"));
const utils_1 = require("../utils");
const Utils = __importStar(require("../utils/interface"));
const web3_1 = __importDefault(require("web3"));
const abi_1 = require("../allocate/abi");
const storage_1 = require("../allocate/storage");
const decoding_1 = require("../core/decoding");
class TruffleWireDecoder {
    constructor(contracts, provider) {
        this.contracts = {};
        this.contractNodes = {};
        this.contexts = {}; //all contexts
        this.deployedContexts = {};
        this.codeCache = {};
        this.web3 = new web3_1.default(provider);
        let contractsAndContexts = [];
        for (let contract of contracts) {
            let node = Utils.getContractNode(contract);
            let deployedContext = undefined;
            let constructorContext = undefined;
            if (node !== undefined) {
                this.contracts[node.id] = contract;
                this.contractNodes[node.id] = node;
            }
            if (contract.deployedBytecode && contract.deployedBytecode !== "0x") {
                deployedContext = Utils.makeContext(contract, node);
                this.contexts[deployedContext.context] = deployedContext;
                //note that we don't set up deployedContexts until after normalization!
            }
            if (contract.bytecode && contract.bytecode !== "0x") {
                constructorContext = Utils.makeContext(contract, node, true);
                this.contexts[constructorContext.context] = constructorContext;
            }
            contractsAndContexts.push({ contract, node, deployedContext, constructorContext });
        }
        this.contexts = utils_1.ContextUtils.normalizeContexts(this.contexts);
        this.deployedContexts = Object.assign({}, ...Object.values(this.contexts).map(context => !context.isConstructor ? { [context.context]: context } : {}));
        for (let contractAndContexts of contractsAndContexts) {
            //change everythign to normalized version
            if (contractAndContexts.deployedContext) {
                contractAndContexts.deployedContext = this.contexts[contractAndContexts.deployedContext.context]; //get normalized version
            }
            if (contractAndContexts.constructorContext) {
                contractAndContexts.constructorContext = this.contexts[contractAndContexts.constructorContext.context]; //get normalized version
            }
        }
        ({ definitions: this.referenceDeclarations, types: this.userDefinedTypes } = this.collectUserDefinedTypes());
        let allocationInfo = contractsAndContexts.map(({ contract: { abi, compiler }, node, deployedContext, constructorContext }) => ({
            abi: utils_1.AbiUtils.schemaAbiToAbi(abi),
            compiler,
            contractNode: node,
            deployedContext,
            constructorContext
        }));
        debug("allocationInfo: %O", allocationInfo);
        this.allocations = {};
        this.allocations.abi = abi_1.getAbiAllocations(this.userDefinedTypes);
        this.allocations.storage = storage_1.getStorageAllocations(this.referenceDeclarations, {}); //not used by wire decoder itself, but used by contract decoder
        this.allocations.calldata = abi_1.getCalldataAllocations(allocationInfo, this.referenceDeclarations, this.userDefinedTypes, this.allocations.abi);
        this.allocations.event = abi_1.getEventAllocations(allocationInfo, this.referenceDeclarations, this.userDefinedTypes, this.allocations.abi);
        debug("done with allocation");
    }
    collectUserDefinedTypes() {
        let references = {};
        let types = {};
        for (const id in this.contracts) {
            const compiler = this.contracts[id].compiler;
            //first, add the contract itself
            const contractNode = this.contractNodes[id];
            references[id] = contractNode;
            types[id] = utils_1.MakeType.definitionToStoredType(contractNode, compiler);
            //now, add its struct and enum definitions
            for (const node of contractNode.nodes) {
                if (node.nodeType === "StructDefinition" || node.nodeType === "EnumDefinition") {
                    references[node.id] = node;
                    //HACK even though we don't have all the references, we only need one:
                    //the reference to the contract itself, which we just added, so we're good
                    types[node.id] = utils_1.MakeType.definitionToStoredType(node, compiler, references);
                }
            }
        }
        return { definitions: references, types };
    }
    //for internal use
    getCode(address, block) {
        return __awaiter(this, void 0, void 0, function* () {
            //first, set up any preliminary layers as needed
            if (this.codeCache[block] === undefined) {
                this.codeCache[block] = {};
            }
            //now, if we have it cached, just return it
            if (this.codeCache[block][address] !== undefined) {
                return this.codeCache[block][address];
            }
            //otherwise, get it, cache it, and return it
            let code = CodecUtils.Conversion.toBytes(yield this.web3.eth.getCode(address, block));
            this.codeCache[block][address] = code;
            return code;
        });
    }
    //NOTE: additionalContexts parameter is for internal use only.
    decodeTransaction(transaction, additionalContexts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            debug("transaction: %O", transaction);
            const block = transaction.blockNumber;
            const context = yield this.getContextByAddress(transaction.to, block, transaction.input, additionalContexts);
            const data = CodecUtils.Conversion.toBytes(transaction.input);
            const info = {
                state: {
                    storage: {},
                    calldata: data,
                },
                userDefinedTypes: this.userDefinedTypes,
                allocations: this.allocations,
                contexts: Object.assign(Object.assign({}, this.deployedContexts), additionalContexts),
                currentContext: context
            };
            const decoder = decoding_1.decodeCalldata(info);
            let result = decoder.next();
            while (result.done === false) {
                let request = result.value;
                let response;
                switch (request.type) {
                    case "code":
                        response = yield this.getCode(request.address, block);
                        break;
                    //not writing a storage case as it shouldn't occur here!
                }
                result = decoder.next(response);
            }
            //at this point, result.value holds the final value
            const decoding = result.value;
            return Object.assign(Object.assign({}, transaction), { decoding });
        });
    }
    //NOTE: options is meant for internal use; do not rely on it
    //NOTE: additionalContexts parameter is for internal use only.
    decodeLog(log, options = {}, additionalContexts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            const block = log.blockNumber;
            const data = CodecUtils.Conversion.toBytes(log.data);
            const topics = log.topics.map(CodecUtils.Conversion.toBytes);
            const info = {
                state: {
                    storage: {},
                    eventdata: data,
                    eventtopics: topics
                },
                userDefinedTypes: this.userDefinedTypes,
                allocations: this.allocations,
                contexts: Object.assign(Object.assign({}, this.deployedContexts), additionalContexts)
            };
            const decoder = decoding_1.decodeEvent(info, log.address, options.name);
            let result = decoder.next();
            while (result.done === false) {
                let request = result.value;
                let response;
                switch (request.type) {
                    case "code":
                        response = yield this.getCode(request.address, block);
                        break;
                    //not writing a storage case as it shouldn't occur here!
                }
                result = decoder.next(response);
            }
            //at this point, result.value holds the final value
            const decodings = result.value;
            return Object.assign(Object.assign({}, log), { decodings });
        });
    }
    //NOTE: options is meant for internal use; do not rely on it
    //NOTE: additionalContexts parameter is for internal use only.
    decodeLogs(logs, options = {}, additionalContexts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield Promise.all(logs.map(log => this.decodeLog(log, options, additionalContexts)));
        });
    }
    //NOTE: additionalContexts parameter is for internal use only.
    events(options = {}, additionalContexts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            let { address, name, fromBlock, toBlock } = options;
            const logs = yield this.web3.eth.getPastLogs({
                address,
                fromBlock,
                toBlock,
            });
            let events = yield this.decodeLogs(logs, options, additionalContexts);
            debug("events: %o", events);
            //if a target name was specified, we'll restrict to events that decoded
            //to something with that name.  (note that only decodings with that name
            //will have been returned from decodeLogs in the first place)
            if (name !== undefined) {
                events = events.filter(event => event.decodings.length > 0);
            }
            return events;
        });
    }
    abifyCalldataDecoding(decoding) {
        return utils_1.abifyCalldataDecoding(decoding, this.userDefinedTypes);
    }
    abifyLogDecoding(decoding) {
        return utils_1.abifyLogDecoding(decoding, this.userDefinedTypes);
    }
    //normally, this function gets the code of the given address at the given block,
    //and checks this against the known contexts to determine the contract type
    //however, if this fails and constructorBinary is passed in, it will then also
    //attempt to determine it from that
    getContextByAddress(address, block, constructorBinary, additionalContexts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            let code;
            if (address !== null) {
                code = CodecUtils.Conversion.toHexString(yield this.getCode(address, block));
            }
            else if (constructorBinary) {
                code = constructorBinary;
            }
            //if neither of these hold... we have a problem
            let contexts = Object.assign(Object.assign({}, this.contexts), additionalContexts);
            return utils_1.ContextUtils.findDecoderContext(contexts, code);
        });
    }
    //the following functions are intended for internal use only
    getReferenceDeclarations() {
        return this.referenceDeclarations;
    }
    getUserDefinedTypes() {
        return this.userDefinedTypes;
    }
    getAllocations() {
        return {
            abi: this.allocations.abi,
            storage: this.allocations.storage
        };
    }
    getWeb3() {
        return this.web3;
    }
    getDeployedContexts() {
        return this.deployedContexts;
    }
}
exports.default = TruffleWireDecoder;
//# sourceMappingURL=wire.js.map