"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:decode:abi");
const read_1 = __importDefault(require("../read"));
const Conversion = __importStar(require("../conversion"));
const Format = __importStar(require("../format"));
const value_1 = __importDefault(require("./value"));
const Evm = __importStar(require("../evm"));
const abi_1 = require("../allocate/abi");
const errors_1 = require("./errors");
function* decodeAbi(dataType, pointer, info, options = {}) {
    if (Format.Types.isReferenceType(dataType) ||
        dataType.typeClass === "tuple") {
        //I don't want tuples to be considered a reference type, but it makes sense
        //to group them for this purpose
        let dynamic;
        try {
            dynamic = abi_1.abiSizeInfo(dataType, info.allocations.abi).dynamic;
        }
        catch (error) {
            if (options.strictAbiMode) {
                throw new errors_1.StopDecodingError(error.error);
            }
            return {
                //dunno why TS is failing at this inference
                type: dataType,
                kind: "error",
                error: error.error
            };
        }
        if (dynamic) {
            return yield* decodeAbiReferenceByAddress(dataType, pointer, info, options);
        }
        else {
            return yield* decodeAbiReferenceStatic(dataType, pointer, info, options);
        }
    }
    else {
        debug("pointer %o", pointer);
        return yield* value_1.default(dataType, pointer, info, options);
    }
}
exports.default = decodeAbi;
function* decodeAbiReferenceByAddress(dataType, pointer, info, options = {}) {
    let { strictAbiMode: strict, abiPointerBase: base } = options;
    base = base || 0; //in case base was undefined
    const { allocations: { abi: allocations }, state } = info;
    debug("pointer %o", pointer);
    //this variable holds the location we should look to *next*
    const location = pointer.location === "eventdata" ? "eventdata" : "calldata"; //stack pointers (& stack literal pointers) point to calldata, not the stack
    let rawValue;
    try {
        rawValue = yield* read_1.default(pointer, state);
    }
    catch (error) {
        if (strict) {
            throw new errors_1.StopDecodingError(error.error);
        }
        return {
            //dunno why TS is failing here
            type: dataType,
            kind: "error",
            error: error.error
        };
    }
    let rawValueAsBN = Conversion.toBN(rawValue);
    let rawValueAsNumber;
    try {
        rawValueAsNumber = rawValueAsBN.toNumber();
    }
    catch (_) {
        let error = {
            kind: "OverlargePointersNotImplementedError",
            pointerAsBN: rawValueAsBN
        };
        if (strict) {
            throw new errors_1.StopDecodingError(error);
        }
        return {
            //again with the TS failures...
            type: dataType,
            kind: "error",
            error
        };
    }
    let startPosition = rawValueAsNumber + base;
    debug("startPosition %d", startPosition);
    let dynamic;
    let size;
    try {
        ({ dynamic, size } = abi_1.abiSizeInfo(dataType, allocations));
    }
    catch (error) {
        if (strict) {
            throw new errors_1.StopDecodingError(error.error);
        }
        return {
            //dunno why TS is failing here
            type: dataType,
            kind: "error",
            error: error.error
        };
    }
    if (!dynamic) {
        //this will only come up when called from stack.ts
        let staticPointer = {
            location,
            start: startPosition,
            length: size
        };
        return yield* decodeAbiReferenceStatic(dataType, staticPointer, info, options);
    }
    let length;
    let lengthAsBN;
    let rawLength;
    switch (dataType.typeClass) {
        case "bytes":
        case "string":
            //initial word contains length
            try {
                rawLength = yield* read_1.default({
                    location,
                    start: startPosition,
                    length: Evm.Utils.WORD_SIZE
                }, state);
            }
            catch (error) {
                if (strict) {
                    throw new errors_1.StopDecodingError(error.error);
                }
                return {
                    //dunno why TS is failing here
                    type: dataType,
                    kind: "error",
                    error: error.error
                };
            }
            lengthAsBN = Conversion.toBN(rawLength);
            if (strict && lengthAsBN.gtn(state[location].length)) {
                //you may notice that the comparison is a bit crude; that's OK, this is
                //just to prevent huge numbers from DOSing us, other errors will still
                //be caught regardless
                throw new errors_1.StopDecodingError({
                    kind: "OverlongArrayOrStringStrictModeError",
                    lengthAsBN,
                    dataLength: state[location].length
                });
            }
            try {
                length = lengthAsBN.toNumber();
            }
            catch (_) {
                //note: if we're in this situation, we can assume we're not in strict mode,
                //as the strict case was handled above
                return {
                    //again with the TS failures...
                    type: dataType,
                    kind: "error",
                    error: {
                        kind: "OverlongArraysAndStringsNotImplementedError",
                        lengthAsBN
                    }
                };
            }
            let childPointer = {
                location,
                start: startPosition + Evm.Utils.WORD_SIZE,
                length
            };
            return yield* value_1.default(dataType, childPointer, info, options);
        case "array":
            switch (dataType.kind) {
                case "dynamic":
                    //initial word contains array length
                    try {
                        rawLength = yield* read_1.default({
                            location,
                            start: startPosition,
                            length: Evm.Utils.WORD_SIZE
                        }, state);
                    }
                    catch (error) {
                        //error: DecodingError
                        if (strict) {
                            throw new errors_1.StopDecodingError(error.error);
                        }
                        return {
                            type: dataType,
                            kind: "error",
                            error: error.error
                        };
                    }
                    lengthAsBN = Conversion.toBN(rawLength);
                    startPosition += Evm.Utils.WORD_SIZE; //increment startPosition
                    //to next word, as first word was used for length
                    break;
                case "static":
                    lengthAsBN = dataType.length;
                    break;
            }
            if (strict && lengthAsBN.gtn(state[location].length)) {
                //you may notice that the comparison is a bit crude; that's OK, this is
                //just to prevent huge numbers from DOSing us, other errors will still
                //be caught regardless
                throw new errors_1.StopDecodingError({
                    kind: "OverlongArraysAndStringsNotImplementedError",
                    lengthAsBN,
                    dataLength: state[location].length
                });
            }
            try {
                length = lengthAsBN.toNumber();
            }
            catch (_) {
                //again, if we get here, we can assume we're not in strict mode
                return {
                    type: dataType,
                    kind: "error",
                    error: {
                        kind: "OverlongArraysAndStringsNotImplementedError",
                        lengthAsBN
                    }
                };
            }
            //note: I've written this fairly generically, but it is worth noting that
            //since this array is of dynamic type, we know that if it's static length
            //then size must be EVM.WORD_SIZE
            let baseSize;
            try {
                baseSize = abi_1.abiSizeInfo(dataType.baseType, allocations).size;
            }
            catch (error) {
                if (strict) {
                    throw new errors_1.StopDecodingError(error.error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error: error.error
                };
            }
            let decodedChildren = [];
            for (let index = 0; index < length; index++) {
                decodedChildren.push(yield* decodeAbi(dataType.baseType, {
                    location,
                    start: startPosition + index * baseSize,
                    length: baseSize
                }, info, Object.assign(Object.assign({}, options), { abiPointerBase: startPosition }))); //pointer base is always start of list, never the length
            }
            return {
                type: dataType,
                kind: "value",
                value: decodedChildren
            };
        case "struct":
            return yield* decodeAbiStructByPosition(dataType, location, startPosition, info, options);
        case "tuple":
            return yield* decodeAbiTupleByPosition(dataType, location, startPosition, info, options);
    }
}
exports.decodeAbiReferenceByAddress = decodeAbiReferenceByAddress;
function* decodeAbiReferenceStatic(dataType, pointer, info, options = {}) {
    debug("static");
    debug("pointer %o", pointer);
    const location = pointer.location;
    switch (dataType.typeClass) {
        case "array":
            //we're in the static case, so we know the array must be statically sized
            const lengthAsBN = dataType.length;
            let length;
            try {
                length = lengthAsBN.toNumber();
            }
            catch (_) {
                //note: since this is the static case, we don't bother including the stronger
                //strict-mode guard against getting DOSed by large array sizes, since in this
                //case we're not reading the size from the input; if there's a huge static size
                //array, well, we'll just have to deal with it
                let error = {
                    kind: "OverlongArraysAndStringsNotImplementedError",
                    lengthAsBN
                };
                if (options.strictAbiMode) {
                    throw new errors_1.StopDecodingError(error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error
                };
            }
            let baseSize;
            try {
                baseSize = abi_1.abiSizeInfo(dataType.baseType, info.allocations.abi).size;
            }
            catch (error) {
                //error: DecodingError
                if (options.strictAbiMode) {
                    throw new errors_1.StopDecodingError(error.error);
                }
                return {
                    type: dataType,
                    kind: "error",
                    error: error.error
                };
            }
            let decodedChildren = [];
            for (let index = 0; index < length; index++) {
                decodedChildren.push(yield* decodeAbi(dataType.baseType, {
                    location,
                    start: pointer.start + index * baseSize,
                    length: baseSize
                }, info, options));
            }
            return {
                type: dataType,
                kind: "value",
                value: decodedChildren
            };
        case "struct":
            return yield* decodeAbiStructByPosition(dataType, location, pointer.start, info, options);
        case "tuple":
            return yield* decodeAbiTupleByPosition(dataType, location, pointer.start, info, options);
    }
}
exports.decodeAbiReferenceStatic = decodeAbiReferenceStatic;
//note that this function takes the start position as a *number*; it does not take a pointer
function* decodeAbiStructByPosition(dataType, location, startPosition, info, options = {}) {
    const { userDefinedTypes, allocations: { abi: allocations } } = info;
    const typeLocation = location === "eventdata"
        ? null //eventdata is not a valid location for a type
        : location;
    const typeId = dataType.id;
    const structAllocation = allocations[parseInt(typeId)];
    if (!structAllocation) {
        let error = {
            kind: "UserDefinedTypeNotFoundError",
            type: dataType
        };
        if (options.strictAbiMode || options.allowRetry) {
            throw new errors_1.StopDecodingError(error, true);
            //note that we allow a retry if we couldn't locate the allocation!
        }
        return {
            type: dataType,
            kind: "error",
            error
        };
    }
    let decodedMembers = [];
    for (let index = 0; index < structAllocation.members.length; index++) {
        const memberAllocation = structAllocation.members[index];
        const memberPointer = memberAllocation.pointer;
        const childPointer = {
            location,
            start: startPosition + memberPointer.start,
            length: memberPointer.length
        };
        let memberName = memberAllocation.name;
        let storedType = userDefinedTypes[typeId];
        if (!storedType) {
            let error = {
                kind: "UserDefinedTypeNotFoundError",
                type: dataType
            };
            if (options.strictAbiMode || options.allowRetry) {
                throw new errors_1.StopDecodingError(error, true);
                //similarly we allow a retry if we couldn't locate the type
            }
            return {
                type: dataType,
                kind: "error",
                error
            };
        }
        let storedMemberType = storedType.memberTypes[index].type;
        let memberType = Format.Types.specifyLocation(storedMemberType, typeLocation);
        decodedMembers.push({
            name: memberName,
            value: yield* decodeAbi(memberType, childPointer, info, Object.assign(Object.assign({}, options), { abiPointerBase: startPosition }))
            //note that the base option is only needed in the dynamic case, but we're being indiscriminate
        });
    }
    return {
        type: dataType,
        kind: "value",
        value: decodedMembers
    };
}
//note that this function takes the start position as a *number*; it does not take a pointer
function* decodeAbiTupleByPosition(dataType, location, startPosition, info, options = {}) {
    //WARNING: This case is written in a way that involves a bunch of unnecessary recomputation!
    //I'm writing it this way anyway for simplicity, to avoid rewriting the decoder
    //However it may be worth revisiting this in the future if performance turns out to be a problem
    //(changing this may be pretty hard though)
    let decodedMembers = [];
    let position = startPosition;
    for (const { name, type: memberType } of dataType.memberTypes) {
        const memberSize = abi_1.abiSizeInfo(memberType, info.allocations.abi).size;
        const childPointer = {
            location,
            start: position,
            length: memberSize
        };
        decodedMembers.push({
            name,
            value: yield* decodeAbi(memberType, childPointer, info, Object.assign(Object.assign({}, options), { abiPointerBase: startPosition }))
            //note that the base option is only needed in the dynamic case, but we're being indiscriminate
        });
        position += memberSize;
    }
    return {
        type: dataType,
        kind: "value",
        value: decodedMembers
    };
}
//# sourceMappingURL=abi.js.map