"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:utils:conversion");
const bn_js_1 = __importDefault(require("bn.js"));
const big_js_1 = __importDefault(require("big.js"));
const web3_1 = __importDefault(require("web3"));
const constants_1 = require("./constants");
const format_1 = require("../format");
const inspect_1 = require("./inspect");
const interface_1 = require("../types/interface");
var Conversion;
(function (Conversion) {
    /**
     * @param bytes - undefined | string | number | BN | Uint8Array | Big
     * @return {BN}
     */
    function toBN(bytes) {
        if (bytes === undefined) {
            return undefined;
        }
        else if (typeof bytes == "string") {
            return new bn_js_1.default(bytes, 16);
        }
        else if (typeof bytes == "number" || bn_js_1.default.isBN(bytes)) {
            return new bn_js_1.default(bytes);
        }
        else if (bytes instanceof big_js_1.default) {
            return new bn_js_1.default(bytes.toFixed()); //warning, better hope input is integer!
            //note: going through string may seem silly but it's actually not terrible here,
            //since BN is binary-based and Big is decimal-based
            //[toFixed is like toString except it guarantees scientific notation is not used]
        }
        else if (typeof bytes.reduce === "function") {
            return bytes.reduce((num, byte) => num.shln(8).addn(byte), new bn_js_1.default(0));
        }
    }
    Conversion.toBN = toBN;
    /**
     * @param bytes - Uint8Array
     * @return {BN}
     */
    function toSignedBN(bytes) {
        if (bytes[0] < 0x80) { // if first bit is 0
            return toBN(bytes);
        }
        else {
            return toBN(bytes.map((b) => 0xff - b)).addn(1).neg();
        }
    }
    Conversion.toSignedBN = toSignedBN;
    function toBig(value) {
        //note: going through string may seem silly but it's actually not terrible here,
        //since BN (& number) is binary-based and Big is decimal-based
        return new big_js_1.default(value.toString());
    }
    Conversion.toBig = toBig;
    /**
     * @param bytes - Uint8Array | BN
     * @param padLength - number - minimum desired byte length (left-pad with zeroes)
     * @return {string}
     */
    function toHexString(bytes, padLength = 0) {
        if (bn_js_1.default.isBN(bytes)) {
            bytes = toBytes(bytes);
        }
        const pad = (s) => `${"00".slice(0, 2 - s.length)}${s}`;
        //                                          0  1  2  3  4
        //                                 0  1  2  3  4  5  6  7
        // bytes.length:        5  -  0x(          e5 c2 aa 09 11 )
        // length (preferred):  8  -  0x( 00 00 00 e5 c2 aa 09 11 )
        //                                `--.---'
        //                                     offset 3
        if (bytes.length < padLength) {
            let prior = bytes;
            bytes = new Uint8Array(padLength);
            bytes.set(prior, padLength - prior.length);
        }
        debug("bytes: %o", bytes);
        let string = bytes.reduce((str, byte) => `${str}${pad(byte.toString(16))}`, "");
        return `0x${string}`;
    }
    Conversion.toHexString = toHexString;
    function toAddress(bytes) {
        if (typeof bytes === "string") {
            //in this case, we can do some simple string manipulation and
            //then pass to web3
            let hex = bytes; //just renaming for clarity
            if (hex.startsWith("0x")) {
                hex = hex.slice(2);
            }
            if (hex.length < 2 * constants_1.Constants.ADDRESS_SIZE) {
                hex = hex.padStart(2 * constants_1.Constants.ADDRESS_SIZE, "0");
            }
            if (hex.length > 2 * constants_1.Constants.ADDRESS_SIZE) {
                hex = "0x" + hex.slice(hex.length - 2 * constants_1.Constants.ADDRESS_SIZE);
            }
            return web3_1.default.utils.toChecksumAddress(hex);
        }
        //otherwise, we're in the Uint8Array case, which we can't fully handle ourself
        //truncate *on left* to 20 bytes
        if (bytes.length > constants_1.Constants.ADDRESS_SIZE) {
            bytes = bytes.slice(bytes.length - constants_1.Constants.ADDRESS_SIZE, bytes.length);
        }
        //now, convert to hex string and apply checksum case that second argument
        //(which ensures it's padded to 20 bytes) shouldn't actually ever be
        //needed, but I'll be safe and include it
        return web3_1.default.utils.toChecksumAddress(toHexString(bytes, constants_1.Constants.ADDRESS_SIZE));
    }
    Conversion.toAddress = toAddress;
    function toBytes(data, length = 0) {
        //note that length is a minimum output length
        //strings will be 0-padded on left
        //numbers/BNs will be sign-padded on left
        //NOTE: if a number/BN is passed in that is too big for the given length,
        //you will get an error!
        //(note that strings passed in should be hex strings; this is not for converting
        //generic strings to hex)
        if (typeof data === "string") {
            let hex = data; //renaming for clarity
            if (hex.startsWith("0x")) {
                hex = hex.slice(2);
            }
            if (hex === "") {
                //this special case is necessary because the match below will return null,
                //not an empty array, when given an empty string
                return new Uint8Array(0);
            }
            if (hex.length % 2 == 1) {
                hex = `0${hex}`;
            }
            let bytes = new Uint8Array(hex.match(/.{2}/g)
                .map((byte) => parseInt(byte, 16)));
            if (bytes.length < length) {
                let prior = bytes;
                bytes = new Uint8Array(length);
                bytes.set(prior, length - prior.length);
            }
            return bytes;
        }
        else {
            // BN/Big/number case
            if (typeof data === "number") {
                data = new bn_js_1.default(data);
            }
            else if (data instanceof big_js_1.default) {
                //note: going through string may seem silly but it's actually not terrible here,
                //since BN is binary-based and Big is decimal-based
                data = new bn_js_1.default(data.toFixed());
                //[toFixed is like toString except it guarantees scientific notation is not used]
            }
            //note that the argument for toTwos is given in bits
            return new Uint8Array(data.toTwos(length * 8).toArrayLike(Buffer, "be", length)); //big-endian
        }
    }
    Conversion.toBytes = toBytes;
    //computes value * 10**decimalPlaces
    function shiftBigUp(value, decimalPlaces) {
        let newValue = new big_js_1.default(value);
        newValue.e += decimalPlaces;
        return newValue;
    }
    Conversion.shiftBigUp = shiftBigUp;
    //computes value * 10**-decimalPlaces
    function shiftBigDown(value, decimalPlaces) {
        let newValue = new big_js_1.default(value);
        newValue.e -= decimalPlaces;
        return newValue;
    }
    Conversion.shiftBigDown = shiftBigDown;
    //we don't need this yet, but we will eventually
    function countDecimalPlaces(value) {
        return Math.max(0, value.c.length - value.e - 1);
    }
    Conversion.countDecimalPlaces = countDecimalPlaces;
    //NOTE: Definitely do not use this in real code!  For tests only!
    //for convenience: invokes the nativize method on all the given variables
    function nativizeVariables(variables) {
        return Object.assign({}, ...Object.entries(variables).map(([name, value]) => ({ [name]: nativize(value) })));
    }
    Conversion.nativizeVariables = nativizeVariables;
    //NOTE: Definitely do not use this in real code!  For tests only!
    //for convenience: invokes the nativize method on all the given variables, and changes them to
    //the old format
    function nativizeDecoderVariables(variables) {
        return Object.assign({}, ...variables.map(({ name, value }) => ({ [name]: nativize(value) })));
        //note that the assignments are processed in order, so if multiple have same name, later
        //(i.e. more derived) will overwrite earlier (i.e. baser)... be aware!  I mean, this is the
        //right way to do overwriting, but it's still overwriting so still dangerous.
        //Again, don't use this in real code!
    }
    Conversion.nativizeDecoderVariables = nativizeDecoderVariables;
    //converts out of range booleans to true; something of a HACK
    //NOTE: does NOT do this recursively inside structs, arrays, etc!
    //I mean, those aren't elementary and therefore aren't in the domain
    //anyway, but still
    function cleanBool(result) {
        switch (result.kind) {
            case "value":
                return result;
            case "error":
                switch (result.error.kind) {
                    case "BoolOutOfRangeError":
                        //return true
                        return {
                            type: result.type,
                            kind: "value",
                            value: {
                                asBoolean: true
                            }
                        };
                    default:
                        return result;
                }
        }
    }
    Conversion.cleanBool = cleanBool;
    //HACK! Avoid using! Only use this if:
    //1. you absolutely have to, or
    //2. it's just testing, not real code
    function nativize(result) {
        if (result.kind === "error") {
            return undefined;
        }
        switch (result.type.typeClass) {
            case "uint":
            case "int":
                return result.value.asBN.toNumber(); //WARNING
            case "bool":
                return result.value.asBoolean;
            case "bytes":
                return result.value.asHex;
            case "address":
                return result.value.asAddress;
            case "string": {
                let coercedResult = result;
                switch (coercedResult.value.kind) {
                    case "valid":
                        return coercedResult.value.asString;
                    case "malformed":
                        // this will turn malformed utf-8 into replacement characters (U+FFFD) (WARNING)
                        // note we need to cut off the 0x prefix
                        return Buffer.from(coercedResult.value.asHex.slice(2), 'hex').toString();
                }
            }
            case "fixed":
            case "ufixed":
                //HACK: Big doesn't have a toNumber() method, so we convert to string and then parse with Number
                //NOTE: we don't bother setting the magic variables Big.NE or Big.PE first, as the choice of
                //notation shouldn't affect the result (can you believe I have to write this? @_@)
                return Number(result.value.asBig.toString()); //WARNING
            case "array": //WARNING: circular case not handled; will loop infinitely
                return result.value.map(nativize);
            case "mapping":
                return Object.assign({}, ...result.value.map(({ key, value }) => ({ [nativize(key).toString()]: nativize(value) })));
            case "struct": //WARNING: circular case not handled; will loop infinitely
                return Object.assign({}, ...result.value.map(({ name, value }) => ({ [name]: nativize(value) })));
            case "tuple":
                return result.value.map(({ value }) => nativize(value));
            case "magic":
                return Object.assign({}, ...Object.entries(result.value).map(([key, value]) => ({ [key]: nativize(value) })));
            case "enum":
                return inspect_1.enumFullName(result);
            case "contract": {
                let coercedResult = result;
                switch (coercedResult.value.kind) {
                    case "known":
                        return `${coercedResult.value.class.typeName}(${coercedResult.value.address})`;
                    case "unknown":
                        return coercedResult.value.address;
                }
                break; //to satisfy typescript
            }
            case "function":
                switch (result.type.visibility) {
                    case "external": {
                        let coercedResult = result;
                        switch (coercedResult.value.kind) {
                            case "known":
                                return `${coercedResult.value.contract.class.typeName}(${coercedResult.value.contract.address}).${coercedResult.value.abi.name}`;
                            case "invalid":
                                return `${coercedResult.value.contract.class.typeName}(${coercedResult.value.contract.address}).call(${coercedResult.value.selector}...)`;
                            case "unknown":
                                return `${coercedResult.value.contract.address}.call(${coercedResult.value.selector}...)`;
                        }
                    }
                    case "internal": {
                        let coercedResult = result;
                        switch (coercedResult.value.kind) {
                            case "function":
                                return `${coercedResult.value.definedIn.typeName}.${coercedResult.value.name}`;
                            case "exception":
                                return coercedResult.value.deployedProgramCounter === 0
                                    ? `<zero>`
                                    : `assert(false)`;
                            case "unknown":
                                return `<decoding not supported>`;
                        }
                    }
                }
        }
    }
    Conversion.nativize = nativize;
})(Conversion = exports.Conversion || (exports.Conversion = {}));
//# sourceMappingURL=conversion.js.map