"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:utils:definition");
const ast_1 = require("../types/ast");
const compiler_1 = require("../types/compiler");
const bn_js_1 = __importDefault(require("bn.js"));
const lodash_clonedeep_1 = __importDefault(require("lodash.clonedeep"));
var Definition;
(function (Definition) {
    function typeIdentifier(definition) {
        return definition.typeDescriptions.typeIdentifier;
    }
    Definition.typeIdentifier = typeIdentifier;
    function typeString(definition) {
        return definition.typeDescriptions.typeString;
    }
    Definition.typeString = typeString;
    //returns the type string, but with location (if any) stripped off the end
    function typeStringWithoutLocation(definition) {
        return typeString(definition).replace(/ (storage|memory|calldata)$/, "");
    }
    Definition.typeStringWithoutLocation = typeStringWithoutLocation;
    /**
     * returns basic type class for a variable definition node
     * e.g.:
     *  `t_uint256` becomes `uint`
     *  `t_struct$_Thing_$20_memory_ptr` becomes `struct`
     */
    function typeClass(definition) {
        return typeIdentifier(definition).match(/t_([^$_0-9]+)/)[1];
    }
    Definition.typeClass = typeClass;
    /**
     * similar to typeClass, but includes any numeric qualifiers
     * e.g.:
     * `t_uint256` becomes `uint256`
     */
    function typeClassLongForm(definition) {
        return typeIdentifier(definition).match(/t_([^$_]+)/)[1];
    }
    Definition.typeClassLongForm = typeClassLongForm;
    //for user-defined types -- structs, enums, contracts
    //often you can get these from referencedDeclaration, but not
    //always
    function typeId(definition) {
        debug("definition %O", definition);
        return parseInt(typeIdentifier(definition)
            .match(/\$(\d+)(_(storage|memory|calldata)(_ptr)?)?$/)[1]);
    }
    Definition.typeId = typeId;
    /**
     * For function types; returns internal or external
     * (not for use on other types! will cause an error!)
     * should only return "internal" or "external"
     */
    function visibility(definition) {
        return (definition.typeName ?
            definition.typeName.visibility : definition.visibility);
    }
    Definition.visibility = visibility;
    /**
     * e.g. uint48 -> 6
     * @return size in bytes for explicit type size, or `null` if not stated
     */
    function specifiedSize(definition) {
        let specified = typeIdentifier(definition).match(/t_[a-z]+([0-9]+)/);
        if (!specified) {
            return null;
        }
        let num = parseInt(specified[1]);
        switch (typeClass(definition)) {
            case "int":
            case "uint":
            case "fixed":
            case "ufixed":
                return num / 8;
            case "bytes":
                return num;
            default:
                debug("Unknown type for size specification: %s", typeIdentifier(definition));
        }
    }
    Definition.specifiedSize = specifiedSize;
    /**
     * for fixed-point types, obviously
     */
    function decimalPlaces(definition) {
        return parseInt(typeIdentifier(definition).match(/t_[a-z]+[0-9]+x([0-9]+)/)[1]);
    }
    Definition.decimalPlaces = decimalPlaces;
    function isArray(definition) {
        return typeIdentifier(definition).match(/^t_array/) != null;
    }
    Definition.isArray = isArray;
    function isDynamicArray(definition) {
        return isArray(definition) && (
        //NOTE: we do this by parsing the type identifier, rather than by just
        //checking the length field, because we might be using this on a faked-up
        //definition
        typeIdentifier(definition).match(/\$dyn_(storage|memory|calldata)(_ptr)?$/) != null);
    }
    Definition.isDynamicArray = isDynamicArray;
    //length of a statically sized array -- please only use for arrays
    //already verified to be static!
    function staticLength(definition) {
        //NOTE: we do this by parsing the type identifier, rather than by just
        //checking the length field, because we might be using this on a faked-up
        //definition
        return parseInt(staticLengthAsString(definition));
    }
    Definition.staticLength = staticLength;
    //see staticLength for explanation
    function staticLengthAsString(definition) {
        return typeIdentifier(definition).match(/\$(\d+)_(storage|memory|calldata)(_ptr)?$/)[1];
    }
    Definition.staticLengthAsString = staticLengthAsString;
    function isStruct(definition) {
        return typeIdentifier(definition).match(/^t_struct/) != null;
    }
    Definition.isStruct = isStruct;
    function isMapping(definition) {
        return typeIdentifier(definition).match(/^t_mapping/) != null;
    }
    Definition.isMapping = isMapping;
    function isEnum(definition) {
        return typeIdentifier(definition).match(/^t_enum/) != null;
    }
    Definition.isEnum = isEnum;
    function isReference(definition) {
        return typeIdentifier(definition).match(/_(memory|storage|calldata)(_ptr)?$/) != null;
    }
    Definition.isReference = isReference;
    //note: only use this on things already verified to be references
    function referenceType(definition) {
        return typeIdentifier(definition).match(/_([^_]+)(_ptr)?$/)[1];
    }
    Definition.referenceType = referenceType;
    //only for contract types, obviously! will yield nonsense otherwise!
    function contractKind(definition) {
        return typeString(definition).split(" ")[0];
    }
    Definition.contractKind = contractKind;
    //stack size, in words, of a given type
    function stackSize(definition) {
        if (typeClass(definition) === "function" &&
            visibility(definition) === "external") {
            return 2;
        }
        if (isReference(definition) && referenceType(definition) === "calldata") {
            if (typeClass(definition) === "string" ||
                typeClass(definition) === "bytes") {
                return 2;
            }
            if (isDynamicArray(definition)) {
                return 2;
            }
        }
        return 1;
    }
    Definition.stackSize = stackSize;
    function isSimpleConstant(definition) {
        const types = ["stringliteral", "rational"];
        return types.includes(typeClass(definition));
    }
    Definition.isSimpleConstant = isSimpleConstant;
    //definition: a storage reference definition
    //location: the location you want it to refer to instead
    function spliceLocation(definition, location) {
        debug("definition %O", definition);
        return Object.assign(Object.assign({}, definition), { typeDescriptions: Object.assign(Object.assign({}, definition.typeDescriptions), { typeIdentifier: definition.typeDescriptions.typeIdentifier
                    .replace(/_(storage|memory|calldata)(?=_ptr$|$)/, "_" + location) }) });
    }
    Definition.spliceLocation = spliceLocation;
    //adds "_ptr" on to the end of type identifiers that might need it; note that
    //this operats on identifiers, not definitions
    function restorePtr(identifier) {
        return identifier.replace(/(?<=_(storage|memory|calldata))$/, "_ptr");
    }
    Definition.restorePtr = restorePtr;
    //extract the actual numerical value from a node of type rational.
    //currently assumes result will be integer (currently returns BN)
    function rationalValue(definition) {
        let identifier = typeIdentifier(definition);
        let absoluteValue = identifier.match(/_(\d+)_by_1$/)[1];
        let isNegative = identifier.match(/_minus_/) != null;
        return isNegative ? new bn_js_1.default(absoluteValue).neg() : new bn_js_1.default(absoluteValue);
    }
    Definition.rationalValue = rationalValue;
    function baseDefinition(definition) {
        if (definition.typeName && definition.typeName.baseType) {
            return definition.typeName.baseType;
        }
        if (definition.baseType) {
            return definition.baseType;
        }
        //otherwise, we'll have to spoof it up ourselves
        let baseIdentifier = typeIdentifier(definition)
            .match(/^t_array\$_(.*)_\$/)[1];
        //greedy match to extract everything from first to last dollar sign
        // HACK - internal types for memory or storage also seem to be pointers
        if (baseIdentifier.match(/_(memory|storage|calldata)$/) != null) {
            baseIdentifier = `${baseIdentifier}_ptr`;
        }
        // another HACK - we get away with it because we're only using that one property
        let result = lodash_clonedeep_1.default(definition);
        result.typeDescriptions.typeIdentifier = baseIdentifier;
        return result;
        //WARNING -- these hacks do *not* correctly handle all cases!
        //they do, however, handle the cases we currently need.
    }
    Definition.baseDefinition = baseDefinition;
    //for use for mappings and arrays only!
    //for arrays, fakes up a uint definition
    function keyDefinition(definition, scopes) {
        let result;
        switch (typeClass(definition)) {
            case "mapping":
                //first: is there a key type already there? if so just use that
                if (definition.keyType) {
                    return definition.keyType;
                }
                if (definition.typeName && definition.typeName.keyType) {
                    return definition.typeName.keyType;
                }
                //otherwise: is there a referencedDeclaration? if so try using that
                let baseDeclarationId = definition.referencedDeclaration;
                debug("baseDeclarationId %d", baseDeclarationId);
                //if there's a referencedDeclaration, we'll use that
                if (baseDeclarationId !== undefined) {
                    let baseDeclaration = scopes[baseDeclarationId].definition;
                    return baseDeclaration.keyType || baseDeclaration.typeName.keyType;
                }
                //otherwise, we'll need to perform some hackery, similarly to in baseDefinition;
                //we'll have to spoof it up ourselves
                let keyIdentifier = typeIdentifier(definition)
                    .match(/^t_mapping\$_(.*?)_\$/)[1];
                //use *non*-greedy match; note that if the key type could include
                //dollar signs, this could cause a problem, but user-defined types
                //are not allowed as key types, so this can't come up
                // HACK - internal types for memory or storage also seem to be pointers
                if (keyIdentifier.match(/_(memory|storage|calldata)$/) != null) {
                    keyIdentifier = `${keyIdentifier}_ptr`;
                }
                let keyString = typeString(definition)
                    .match(/mapping\((.*?) => .*\)( storage)?$/)[1];
                //use *non*-greedy match; note that if the key type could include
                //"=>", this could cause a problem, but mappings are not allowed as key
                //types, so this can't come up
                // another HACK - we get away with it because we're only using that one property
                result = lodash_clonedeep_1.default(definition);
                result.typeDescriptions = {
                    typeIdentifier: keyIdentifier,
                    typeString: keyString
                };
                return result;
            case "array":
                //HACK -- again we should get away with it because for a uint256 we don't
                //really need to inspect the other properties
                result = lodash_clonedeep_1.default(definition);
                result.typeDescriptions = {
                    typeIdentifier: "t_uint256",
                    typeString: "uint256"
                };
                return result;
            default:
                debug("unrecognized index access!");
        }
    }
    Definition.keyDefinition = keyDefinition;
    //returns input parameters, then output parameters
    //NOTE: ONLY FOR VARIABLE DECLARATIONS OF FUNCTION TYPE
    //NOT FOR FUNCTION DEFINITIONS
    function parameters(definition) {
        let typeObject = definition.typeName || definition;
        return [typeObject.parameterTypes.parameters, typeObject.returnParameterTypes.parameters];
    }
    Definition.parameters = parameters;
    //compatibility function, since pre-0.5.0 functions don't have node.kind
    //returns undefined if you don't put in a function node
    function functionKind(node) {
        if (node.nodeType !== "FunctionDefinition") {
            return undefined;
        }
        if (node.kind !== undefined) {
            //if we're dealing with 0.5.x, we can just read node.kind
            return node.kind;
        }
        //otherwise, we need this little shim
        if (node.isConstructor) {
            return "constructor";
        }
        return node.name === ""
            ? "fallback"
            : "function";
    }
    Definition.functionKind = functionKind;
    //similar compatibility function for mutability for pre-0.4.16 versions
    //returns undefined if you don't give it a FunctionDefinition or
    //VariableDeclaration
    function mutability(node) {
        node = node.typeName || node;
        if (node.nodeType !== "FunctionDefinition" && node.nodeType !== "FunctionTypeName") {
            return undefined;
        }
        if (node.stateMutability !== undefined) {
            //if we're dealing with 0.4.16 or later, we can just read node.stateMutability
            return node.stateMutability;
        }
        //otherwise, we need this little shim
        if (node.payable) {
            return "payable";
        }
        if (node.constant) {
            //yes, it means "view" even if you're looking at a variable declaration!
            //old Solidity was weird!
            return "view";
        }
        return "nonpayable";
    }
    Definition.mutability = mutability;
    //takes a contract definition and asks, does it have a payable fallback function?
    function isContractPayable(definition) {
        let fallback = definition.nodes.find(node => node.nodeType === "FunctionDefinition" &&
            functionKind(node) === "fallback");
        if (!fallback) {
            return false;
        }
        return mutability(fallback) === "payable";
    }
    Definition.isContractPayable = isContractPayable;
    //spoofed definitions we'll need
    //we'll give them id -1 to indicate that they're spoofed
    Definition.NOW_DEFINITION = {
        id: -1,
        src: "0:0:-1",
        name: "now",
        nodeType: "VariableDeclaration",
        typeDescriptions: {
            typeIdentifier: "t_uint256",
            typeString: "uint256"
        }
    };
    Definition.MSG_DEFINITION = {
        id: -1,
        src: "0:0:-1",
        name: "msg",
        nodeType: "VariableDeclaration",
        typeDescriptions: {
            typeIdentifier: "t_magic_message",
            typeString: "msg"
        }
    };
    Definition.TX_DEFINITION = {
        id: -1,
        src: "0:0:-1",
        name: "tx",
        nodeType: "VariableDeclaration",
        typeDescriptions: {
            typeIdentifier: "t_magic_transaction",
            typeString: "tx"
        }
    };
    Definition.BLOCK_DEFINITION = {
        id: -1,
        src: "0:0:-1",
        name: "block",
        nodeType: "VariableDeclaration",
        typeDescriptions: {
            typeIdentifier: "t_magic_block",
            typeString: "block"
        }
    };
    function spoofThisDefinition(contractName, contractId, contractKind) {
        let formattedName = contractName.replace(/\$/g, "$$".repeat(3));
        //note that string.replace treats $'s specially in the replacement string;
        //we want 3 $'s for each $ in the input, so we need to put *6* $'s in the
        //replacement string
        return {
            id: -1,
            src: "0:0:-1",
            name: "this",
            nodeType: "VariableDeclaration",
            typeDescriptions: {
                typeIdentifier: "t_contract$_" + formattedName + "_$" + contractId,
                typeString: contractKind + " " + contractName
            }
        };
    }
    Definition.spoofThisDefinition = spoofThisDefinition;
})(Definition = exports.Definition || (exports.Definition = {}));
//# sourceMappingURL=definition.js.map