"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:format:maketype");
const bn_js_1 = __importDefault(require("bn.js"));
const Ast = __importStar(require("../types/ast"));
const definition_1 = require("./definition");
const compiler_1 = require("../types/compiler");
const compiler_2 = require("./compiler");
const abi_1 = require("../types/abi");
const format_1 = require("../format");
var MakeType;
(function (MakeType) {
    //NOTE: the following function will *not* work for arbitrary nodes! It will,
    //however, work for the ones we need (i.e., variable definitions, and arbitrary
    //things of elementary type)
    //NOTE: set forceLocation to *null* to force no location. leave it undefined
    //to not force a location.
    function definitionToType(definition, compiler, forceLocation) {
        debug("definition %O", definition);
        let typeClass = definition_1.Definition.typeClass(definition);
        let typeHint = definition_1.Definition.typeStringWithoutLocation(definition);
        switch (typeClass) {
            case "bool":
                return {
                    typeClass,
                    typeHint
                };
            case "address": {
                switch (compiler_2.solidityFamily(compiler)) {
                    case "pre-0.5.0":
                        return {
                            typeClass,
                            kind: "general",
                            typeHint
                        };
                    case "0.5.x":
                        return {
                            typeClass,
                            kind: "specific",
                            payable: definition_1.Definition.typeIdentifier(definition) === "t_address_payable"
                        };
                }
                break; //to satisfy typescript
            }
            case "uint": {
                let bytes = definition_1.Definition.specifiedSize(definition);
                return {
                    typeClass,
                    bits: bytes * 8,
                    typeHint
                };
            }
            case "int": { //typeScript won't let me group these for some reason
                let bytes = definition_1.Definition.specifiedSize(definition);
                return {
                    typeClass,
                    bits: bytes * 8,
                    typeHint
                };
            }
            case "fixed": { //typeScript won't let me group these for some reason
                let bytes = definition_1.Definition.specifiedSize(definition);
                let places = definition_1.Definition.decimalPlaces(definition);
                return {
                    typeClass,
                    bits: bytes * 8,
                    places,
                    typeHint
                };
            }
            case "ufixed": {
                let bytes = definition_1.Definition.specifiedSize(definition);
                let places = definition_1.Definition.decimalPlaces(definition);
                return {
                    typeClass,
                    bits: bytes * 8,
                    places,
                    typeHint
                };
            }
            case "string": {
                if (forceLocation === null) {
                    return {
                        typeClass,
                        typeHint
                    };
                }
                let location = forceLocation || definition_1.Definition.referenceType(definition);
                return {
                    typeClass,
                    location,
                    typeHint
                };
            }
            case "bytes": {
                let length = definition_1.Definition.specifiedSize(definition);
                if (length !== null) {
                    return {
                        typeClass,
                        kind: "static",
                        length,
                        typeHint
                    };
                }
                else {
                    if (forceLocation === null) {
                        return {
                            typeClass,
                            kind: "dynamic",
                            typeHint
                        };
                    }
                    let location = forceLocation || definition_1.Definition.referenceType(definition);
                    return {
                        typeClass,
                        kind: "dynamic",
                        location,
                        typeHint
                    };
                }
            }
            case "array": {
                let baseDefinition = definition_1.Definition.baseDefinition(definition);
                let baseType = definitionToType(baseDefinition, compiler, forceLocation);
                let location = forceLocation || definition_1.Definition.referenceType(definition);
                if (definition_1.Definition.isDynamicArray(definition)) {
                    if (forceLocation !== null) {
                        return {
                            typeClass,
                            baseType,
                            kind: "dynamic",
                            location,
                            typeHint
                        };
                    }
                    else {
                        return {
                            typeClass,
                            baseType,
                            kind: "dynamic",
                            typeHint
                        };
                    }
                }
                else {
                    let length = new bn_js_1.default(definition_1.Definition.staticLengthAsString(definition));
                    if (forceLocation !== null) {
                        return {
                            typeClass,
                            baseType,
                            kind: "static",
                            length,
                            location,
                            typeHint
                        };
                    }
                    else {
                        return {
                            typeClass,
                            baseType,
                            kind: "static",
                            length,
                            typeHint
                        };
                    }
                }
            }
            case "mapping": {
                let keyDefinition = definition_1.Definition.keyDefinition(definition);
                //note that we can skip the scopes argument here! that's only needed when
                //a general node, rather than a declaration, is being passed in
                let keyType = definitionToType(keyDefinition, compiler, null);
                //suppress the location on the key type (it'll be given as memory but
                //this is meaningless)
                //also, we have to tell TypeScript ourselves that this will be an elementary
                //type; it has no way of knowing that
                let valueDefinition = definition.valueType || definition.typeName.valueType;
                let valueType = definitionToType(valueDefinition, compiler, forceLocation);
                if (forceLocation === null) {
                    return {
                        typeClass,
                        keyType,
                        valueType
                    };
                }
                return {
                    typeClass,
                    keyType,
                    valueType,
                    location: "storage"
                };
            }
            case "function": {
                let visibility = definition_1.Definition.visibility(definition);
                let mutability = definition_1.Definition.mutability(definition);
                let [inputParameters, outputParameters] = definition_1.Definition.parameters(definition);
                //note: don't force a location on these! use the listed location!
                let inputParameterTypes = inputParameters.map(parameter => definitionToType(parameter, compiler));
                let outputParameterTypes = outputParameters.map(parameter => definitionToType(parameter, compiler));
                switch (visibility) {
                    case "internal":
                        return {
                            typeClass,
                            visibility,
                            mutability,
                            inputParameterTypes,
                            outputParameterTypes
                        };
                    case "external":
                        return {
                            typeClass,
                            visibility,
                            kind: "specific",
                            mutability,
                            inputParameterTypes,
                            outputParameterTypes
                        };
                }
                break; //to satisfy typescript
            }
            case "struct": {
                let id = definition_1.Definition.typeId(definition).toString();
                let qualifiedName = definition_1.Definition.typeStringWithoutLocation(definition).match(/struct (.*)/)[1];
                let [definingContractName, typeName] = qualifiedName.split(".");
                if (forceLocation === null) {
                    return {
                        typeClass,
                        kind: "local",
                        id,
                        typeName,
                        definingContractName
                    };
                }
                let location = forceLocation || definition_1.Definition.referenceType(definition);
                return {
                    typeClass,
                    kind: "local",
                    id,
                    typeName,
                    definingContractName,
                    location
                };
            }
            case "enum": {
                let id = definition_1.Definition.typeId(definition).toString();
                let qualifiedName = definition_1.Definition.typeStringWithoutLocation(definition).match(/enum (.*)/)[1];
                let [definingContractName, typeName] = qualifiedName.split(".");
                return {
                    typeClass,
                    kind: "local",
                    id,
                    typeName,
                    definingContractName
                };
            }
            case "contract": {
                let id = definition_1.Definition.typeId(definition).toString();
                let typeName = definition.typeName
                    ? definition.typeName.name
                    : definition.name;
                let contractKind = definition_1.Definition.contractKind(definition);
                return {
                    typeClass,
                    kind: "native",
                    id,
                    typeName,
                    contractKind
                };
            }
            case "magic": {
                let typeIdentifier = definition_1.Definition.typeIdentifier(definition);
                let variable = typeIdentifier.match(/^t_magic_(.*)$/)[1];
                return {
                    typeClass,
                    variable
                };
            }
        }
    }
    MakeType.definitionToType = definitionToType;
    //whereas the above takes variable definitions, this takes the actual type
    //definition
    function definitionToStoredType(definition, compiler, referenceDeclarations) {
        switch (definition.nodeType) {
            case "StructDefinition": {
                let id = definition.id.toString();
                let [definingContractName, typeName] = definition.canonicalName.split(".");
                let memberTypes = definition.members.map(member => ({ name: member.name, type: definitionToType(member, compiler, null) }));
                let definingContract;
                if (referenceDeclarations) {
                    let contractDefinition = Object.values(referenceDeclarations).find(node => node.nodeType === "ContractDefinition" &&
                        node.nodes.some((subNode) => subNode.id.toString() === id));
                    definingContract = definitionToStoredType(contractDefinition, compiler); //can skip reference declarations
                }
                return {
                    typeClass: "struct",
                    kind: "local",
                    id,
                    typeName,
                    definingContractName,
                    definingContract,
                    memberTypes
                };
            }
            case "EnumDefinition": {
                let id = definition.id.toString();
                let [definingContractName, typeName] = definition.canonicalName.split(".");
                let options = definition.members.map(member => member.name);
                let definingContract;
                if (referenceDeclarations) {
                    let contractDefinition = Object.values(referenceDeclarations).find(node => node.nodeType === "ContractDefinition" &&
                        node.nodes.some((subNode) => subNode.id.toString() === id));
                    definingContract = definitionToStoredType(contractDefinition, compiler); //can skip reference declarations
                }
                return {
                    typeClass: "enum",
                    kind: "local",
                    id,
                    typeName,
                    definingContractName,
                    definingContract,
                    options
                };
            }
            case "ContractDefinition": {
                let id = definition.id.toString();
                let typeName = definition.name;
                let contractKind = definition.contractKind;
                let payable = definition_1.Definition.isContractPayable(definition);
                return {
                    typeClass: "contract",
                    kind: "native",
                    id,
                    typeName,
                    contractKind,
                    payable
                };
            }
        }
    }
    MakeType.definitionToStoredType = definitionToStoredType;
    function abiParameterToType(abi) {
        let typeName = abi.type;
        let typeHint = abi.internalType;
        //first: is it an array?
        let arrayMatch = typeName.match(/(.*)\[(\d*)\]$/);
        if (arrayMatch) {
            let baseTypeName = arrayMatch[1];
            let lengthAsString = arrayMatch[2]; //may be empty!
            let baseAbi = Object.assign(Object.assign({}, abi), { type: baseTypeName });
            let baseType = abiParameterToType(baseAbi);
            if (lengthAsString === "") {
                return {
                    typeClass: "array",
                    kind: "dynamic",
                    baseType,
                    typeHint
                };
            }
            else {
                let length = new bn_js_1.default(lengthAsString);
                return {
                    typeClass: "array",
                    kind: "static",
                    length,
                    baseType,
                    typeHint
                };
            }
        }
        //otherwise, here are the simple cases
        let typeClass = typeName.match(/^([^0-9]+)/)[1];
        switch (typeClass) {
            case "uint":
            case "int": {
                let bits = typeName.match(/^u?int([0-9]+)/)[1];
                return {
                    typeClass,
                    bits: parseInt(bits),
                    typeHint
                };
            }
            case "bytes":
                let length = typeName.match(/^bytes([0-9]*)/)[1];
                if (length === "") {
                    return {
                        typeClass,
                        kind: "dynamic",
                        typeHint
                    };
                }
                else {
                    return {
                        typeClass,
                        kind: "static",
                        length: parseInt(length),
                        typeHint
                    };
                }
            case "address":
                return {
                    typeClass,
                    kind: "general",
                    typeHint
                };
            case "string":
            case "bool":
                return {
                    typeClass,
                    typeHint
                };
            case "fixed":
            case "ufixed": {
                let [_, bits, places] = typeName.match(/^u?fixed([0-9]+)x([0-9]+)/);
                return {
                    typeClass,
                    bits: parseInt(bits),
                    places: parseInt(places),
                    typeHint
                };
            }
            case "function":
                return {
                    typeClass,
                    visibility: "external",
                    kind: "general",
                    typeHint
                };
            case "tuple":
                let memberTypes = abi.components.map(component => ({
                    name: component.name || undefined,
                    type: abiParameterToType(component)
                }));
                return {
                    typeClass,
                    memberTypes,
                    typeHint
                };
        }
    }
    MakeType.abiParameterToType = abiParameterToType;
})(MakeType = exports.MakeType || (exports.MakeType = {}));
//# sourceMappingURL=maketype.js.map