"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:read:storage");
const CodecUtils = __importStar(require("../utils"));
const storage_1 = require("../utils/storage");
const errors_1 = require("../utils/errors");
const errors_2 = require("../types/errors");
/**
 * read slot from storage
 *
 * @param slot - see slotAddress() code to understand how these work
 * @param offset - for array, offset from the keccak determined location
 */
function* read(storage, slot) {
    debug("Slot printout: %s", errors_1.slotAddressPrintout(slot));
    const address = storage_1.slotAddress(slot);
    // debug("reading slot: %o", CodecUtils.toHexString(address));
    const hexAddress = CodecUtils.Conversion.toHexString(address, CodecUtils.EVM.WORD_SIZE);
    let word = storage[hexAddress];
    //if we can't find the word in the map, we place a request to the invoker to supply it
    //(contract-decoder will look it up from the blockchain, while the debugger will just
    //say 0)
    if (word === undefined) {
        word = yield {
            type: "storage",
            slot: address
        };
    }
    return word;
}
exports.read = read;
/**
 * read all bytes in some range.
 *
 * parameters `from` and `to` are objects with the following properties:
 *
 *   slot - see the slotAddress() code to understand how these work
 *
 *     ref: https://solidity.readthedocs.io/en/v0.4.23/miscellaneous.html#layout-of-state-variables-in-storage
 *     (search "concatenation")
 *
 *  index - (default: 0) byte index in word
 *
 * @param from - location (see ^)
 * @param to - location (see ^). inclusive.
 * @param length - instead of `to`, number of bytes after `from`
 */
function* readRange(storage, range) {
    debug("readRange %o", range);
    let { from, to, length } = range;
    from = {
        slot: from.slot,
        index: from.index || 0
    };
    if (length !== undefined) {
        to = {
            slot: {
                path: from.slot.path || undefined,
                offset: from.slot.offset.addn(Math.floor((from.index + length - 1) / CodecUtils.EVM.WORD_SIZE))
            },
            index: (from.index + length - 1) % CodecUtils.EVM.WORD_SIZE
        };
    }
    debug("normalized readRange %o", { from, to });
    let totalWordsAsBN = to.slot.offset.sub(from.slot.offset).addn(1);
    let totalWords;
    try {
        totalWords = totalWordsAsBN.toNumber();
    }
    catch (_) {
        throw new errors_2.DecodingError({
            kind: "ReadErrorStorage",
            range
        });
    }
    let data = new Uint8Array(totalWords * CodecUtils.EVM.WORD_SIZE);
    for (let i = 0; i < totalWords; i++) {
        let offset = from.slot.offset.addn(i);
        const word = yield* read(storage, Object.assign(Object.assign({}, from.slot), { offset }));
        if (typeof word !== "undefined") {
            data.set(word, i * CodecUtils.EVM.WORD_SIZE);
        }
    }
    debug("words %o", data);
    data = data.slice(from.index, (totalWords - 1) * CodecUtils.EVM.WORD_SIZE + to.index + 1);
    debug("data: %o", data);
    return data;
}
exports.readRange = readRange;
//# sourceMappingURL=storage.js.map