"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const errors_1 = require("../utils/errors");
class UnknownBaseContractIdError extends Error {
    constructor(derivedId, derivedName, derivedKind, baseId) {
        const message = `Cannot locate base contract ID ${baseId} of ${derivedKind} ${derivedName} (ID ${derivedId})`;
        super(message);
        this.name = "UnknownBaseContractIdError";
        this.derivedId = derivedId;
        this.derivedName = derivedName;
        this.derivedKind = derivedKind;
        this.baseId = baseId;
    }
}
exports.UnknownBaseContractIdError = UnknownBaseContractIdError;
//For when we need to throw an error, here's a wrapper class that extends Error.
//Apologies about the confusing name, but I wanted something that would make
//sense should it not be caught and thus accidentally exposed to the outside.
class DecodingError extends Error {
    constructor(error) {
        super(errors_1.message(error));
        this.error = error;
        this.name = "DecodingError";
    }
}
exports.DecodingError = DecodingError;
//used to stop decoding; like DecodingError, but used in contexts
//where I don't expect it to be caught
//NOTE: currently we don't actually check the type of a thrown error,
//we just rely on context.  still, I think it makes sense to be a separate
//type.
class StopDecodingError extends Error {
    //when decoding in full mode, we allow an ABI-mode retry.  (if we were already in
    //ABI mode, we give up.)
    constructor(error, allowRetry) {
        const message = `Stopping decoding: ${error.kind}`; //sorry about the bare-bones message,
        //but again, users shouldn't actually see this, so I think this should suffice for now
        super(message);
        this.error = error;
        this.allowRetry = Boolean(allowRetry);
    }
}
exports.StopDecodingError = StopDecodingError;
class UnknownUserDefinedTypeError extends Error {
    constructor(id, typeString) {
        const message = `Cannot locate definition for ${typeString} (ID ${id})`;
        super(message);
        this.name = "UnknownUserDefinedTypeError";
        this.id = id;
        this.typeString = typeString;
    }
}
exports.UnknownUserDefinedTypeError = UnknownUserDefinedTypeError;
class ContractBeingDecodedHasNoNodeError extends Error {
    constructor(contractName) {
        const message = `Contract ${contractName} does not appear to have been compiled with Solidity (cannot locate contract node)`;
        super(message);
        this.name = "ContractBeingDecodedHasNoNodeError";
    }
}
exports.ContractBeingDecodedHasNoNodeError = ContractBeingDecodedHasNoNodeError;
class ContractAllocationFailedError extends Error {
    constructor(id, contractName) {
        super(`No allocation found for contract ID ${id} (${contractName})`);
        this.name = "ContractAllocationFailedError";
    }
}
exports.ContractAllocationFailedError = ContractAllocationFailedError;
//# sourceMappingURL=errors.js.map