"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default("codec:utils:definition2abi");
const definition_1 = require("./definition");
const errors_1 = require("../types/errors");
//the main function. just does some dispatch.
//returns undefined on bad input
function definitionToAbi(node, referenceDeclarations) {
    switch (node.nodeType) {
        case "FunctionDefinition":
            if (node.visibility === "public" || node.visibility === "external") {
                return functionDefinitionToAbi(node, referenceDeclarations);
            }
            else {
                return undefined;
            }
        case "EventDefinition":
            return eventDefinitionToAbi(node, referenceDeclarations);
        case "VariableDeclaration":
            if (node.visibility === "public") {
                return getterDefinitionToAbi(node, referenceDeclarations);
            }
            else {
                return undefined;
            }
        default:
            return undefined;
    }
}
exports.definitionToAbi = definitionToAbi;
//note: not for FunctionTypeNames or VariableDeclarations
function functionDefinitionToAbi(node, referenceDeclarations) {
    let kind = definition_1.Definition.functionKind(node);
    let stateMutability = definition_1.Definition.mutability(node);
    let payable = stateMutability === "payable";
    let constant = stateMutability === "view" || stateMutability == "pure";
    let inputs;
    switch (kind) {
        case "function":
            let name = node.name;
            let outputs = parametersToAbi(node.returnParameters.parameters, referenceDeclarations);
            inputs = parametersToAbi(node.parameters.parameters, referenceDeclarations);
            return {
                type: "function",
                name,
                inputs,
                outputs,
                stateMutability,
                constant,
                payable
            };
        case "constructor":
            inputs = parametersToAbi(node.parameters.parameters, referenceDeclarations);
            //note: need to coerce because of mutability restrictions
            return {
                type: "constructor",
                inputs,
                stateMutability,
                payable
            };
        case "fallback":
            //note: need to coerce because of mutability restrictions
            return {
                type: "fallback",
                stateMutability,
                payable
            };
    }
}
function eventDefinitionToAbi(node, referenceDeclarations) {
    let inputs = parametersToAbi(node.parameters.parameters, referenceDeclarations, true);
    let name = node.name;
    let anonymous = node.anonymous;
    return {
        type: "event",
        inputs,
        name,
        anonymous
    };
}
function parametersToAbi(nodes, referenceDeclarations, checkIndexed = false) {
    return nodes.map(node => parameterToAbi(node, referenceDeclarations, checkIndexed));
}
function parameterToAbi(node, referenceDeclarations, checkIndexed = false) {
    let name = node.name; //may be the empty string... or even undefined for a base type
    let components;
    let indexed;
    if (checkIndexed) {
        indexed = node.indexed; //note: may be undefined for a base type
    }
    let internalType = definition_1.Definition.typeStringWithoutLocation(node);
    //is this an array? if so use separate logic
    if (definition_1.Definition.typeClass(node) === "array") {
        let baseType = node.typeName ? node.typeName.baseType : node.baseType;
        let baseAbi = parameterToAbi(baseType, referenceDeclarations, checkIndexed);
        let arraySuffix = definition_1.Definition.isDynamicArray(node)
            ? `[]`
            : `[${definition_1.Definition.staticLength(node)}]`;
        return {
            name,
            type: baseAbi.type + arraySuffix,
            indexed,
            components: baseAbi.components,
            internalType
        };
    }
    let abiTypeString = toAbiType(node, referenceDeclarations);
    //otherwise... is it a struct? if so we need to populate components
    if (definition_1.Definition.typeClass(node) === "struct") {
        let id = definition_1.Definition.typeId(node);
        let referenceDeclaration = referenceDeclarations[id];
        if (referenceDeclaration === undefined) {
            let typeToDisplay = definition_1.Definition.typeString(node);
            throw new errors_1.UnknownUserDefinedTypeError(id.toString(), typeToDisplay);
        }
        components = parametersToAbi(referenceDeclaration.members, referenceDeclarations, checkIndexed);
    }
    return {
        name,
        type: abiTypeString,
        indexed,
        components,
        internalType
    };
}
//note: this is only meant for non-array types that can go in the ABI
//it returns how that type is notated in the ABI -- just the string,
//to be clear, not components of tuples
//again, NOT FOR ARRAYS
function toAbiType(node, referenceDeclarations) {
    let basicType = definition_1.Definition.typeClassLongForm(node); //get that whole first segment!
    switch (basicType) {
        case "contract":
            return "address";
        case "struct":
            return "tuple"; //the more detailed checking will be handled elsewhere
        case "enum":
            let referenceId = definition_1.Definition.typeId(node);
            let referenceDeclaration = referenceDeclarations[referenceId];
            if (referenceDeclaration === undefined) {
                let typeToDisplay = definition_1.Definition.typeString(node);
                throw new errors_1.UnknownUserDefinedTypeError(referenceId.toString(), typeToDisplay);
            }
            let numOptions = referenceDeclaration.members.length;
            let bits = 8 * Math.ceil(Math.log2(numOptions) / 8);
            return `uint${bits}`;
        default:
            return basicType;
        //note that: int/uint/fixed/ufixed/bytes will have their size and such left on;
        //address will have "payable" left off;
        //external functions will be reduced to "function" (and internal functions shouldn't
        //be passed in!)
        //(mappings shouldn't be passed in either obviously)
        //(nor arrays :P )
    }
}
function getterDefinitionToAbi(node, referenceDeclarations) {
    debug("getter node: %O", node);
    let name = node.name;
    let { inputs, outputs } = getterParameters(node, referenceDeclarations);
    let inputsAbi = parametersToAbi(inputs, referenceDeclarations);
    let outputsAbi = parametersToAbi(outputs, referenceDeclarations);
    return {
        type: "function",
        name,
        inputs: inputsAbi,
        outputs: outputsAbi,
        stateMutability: "view",
        constant: true,
        payable: false
    };
}
//how getter parameters work:
//INPUT:
//types other than arrays and mappings take no input.
//array getters take uint256 input. mapping getters take input of their key type.
//if arrays, mappings, stacked, then takes multiple inputs, in order from outside
//to in.
//These parameters are unnamed.
//OUTPUT:
//if base type (beneath mappings & arrays) is not a struct, returns that.
//(This return parameter has no name -- it is *not* named for the variable!)
//if it is a struct, returns multiple outputs, one for each member of the struct,
//*except* arrays and mappings.  (And they have names, the names of the members.)
//important note: inner structs within a struct are just returned, not
//partially destructured like the outermost struct!  Yes, this is confusing.
//here's a simplified function that just does the inputs. it's for use by the
//allocator. I'm keeping it separate because it doesn't require a
//referenceDeclarations argument.
function getterInputs(node) {
    node = node.typeName || node;
    let inputs = [];
    while (definition_1.Definition.typeClass(node) === "array" || definition_1.Definition.typeClass(node) === "mapping") {
        let keyNode = definition_1.Definition.keyDefinition(node); //note: if node is an array, this spoofs up a uint256 definition
        inputs.push(Object.assign(Object.assign({}, keyNode), { name: "" })); //getter input params have no name
        switch (definition_1.Definition.typeClass(node)) {
            case "array":
                node = node.baseType;
                break;
            case "mapping":
                node = node.valueType;
                break;
        }
    }
    return inputs;
}
exports.getterInputs = getterInputs;
//again, despite the duplication, this function is kept separate from the
//more straightforward getterInputs function because, since it has to handle
//outputs too, it requires referenceDeclarations
function getterParameters(node, referenceDeclarations) {
    let baseNode = node.typeName || node;
    let inputs = [];
    while (definition_1.Definition.typeClass(baseNode) === "array" || definition_1.Definition.typeClass(baseNode) === "mapping") {
        let keyNode = definition_1.Definition.keyDefinition(baseNode); //note: if baseNode is an array, this spoofs up a uint256 definition
        inputs.push(Object.assign(Object.assign({}, keyNode), { name: "" })); //again, getter input params have no name
        switch (definition_1.Definition.typeClass(baseNode)) {
            case "array":
                baseNode = baseNode.baseType;
                break;
            case "mapping":
                baseNode = baseNode.valueType;
                break;
        }
    }
    //at this point, baseNode should hold the base type
    //now we face the question: is it a struct?
    if (definition_1.Definition.typeClass(baseNode) === "struct") {
        let id = definition_1.Definition.typeId(baseNode);
        let referenceDeclaration = referenceDeclarations[id];
        if (referenceDeclaration === undefined) {
            let typeToDisplay = definition_1.Definition.typeString(baseNode);
            throw new errors_1.UnknownUserDefinedTypeError(id.toString(), typeToDisplay);
        }
        let outputs = referenceDeclaration.members.filter(member => definition_1.Definition.typeClass(member) !== "array" && definition_1.Definition.typeClass(member) !== "mapping");
        return { inputs, outputs }; //no need to wash name!
    }
    else {
        //only one output; it's just the base node, with its name washed
        return { inputs, outputs: [Object.assign(Object.assign({}, baseNode), { name: "" })] };
    }
}
//# sourceMappingURL=definition2abi.js.map