"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.put = exports.get = exports.request = exports.Method = void 0;
const tslib_1 = require("tslib");
const got_1 = tslib_1.__importDefault(require("got"));
const https_1 = require("https");
const http_1 = require("http");
const micromatch_1 = tslib_1.__importDefault(require("micromatch"));
const p_limit_1 = tslib_1.__importDefault(require("p-limit"));
const tunnel_1 = tslib_1.__importDefault(require("tunnel"));
const url_1 = require("url");
const NETWORK_CONCURRENCY = 8;
const limit = p_limit_1.default(NETWORK_CONCURRENCY);
const cache = new Map();
const globalHttpAgent = new http_1.Agent({ keepAlive: true });
const globalHttpsAgent = new https_1.Agent({ keepAlive: true });
function parseProxy(specifier) {
    const url = new url_1.URL(specifier);
    const proxy = { host: url.hostname, headers: {} };
    if (url.port)
        proxy.port = Number(url.port);
    return { proxy };
}
var Method;
(function (Method) {
    Method["GET"] = "GET";
    Method["PUT"] = "PUT";
})(Method = exports.Method || (exports.Method = {}));
async function request(target, body, { configuration, headers, json, method = Method.GET }) {
    if (!configuration.get(`enableNetwork`))
        throw new Error(`Network access have been disabled by configuration (${method} ${target})`);
    const url = new url_1.URL(target);
    if (url.protocol === `http:` && !micromatch_1.default.isMatch(url.hostname, configuration.get(`unsafeHttpWhitelist`)))
        throw new Error(`Unsafe http requests must be explicitly whitelisted in your configuration (${url.hostname})`);
    const httpProxy = configuration.get(`httpProxy`);
    const httpsProxy = configuration.get(`httpsProxy`);
    const agent = {
        http: httpProxy
            ? tunnel_1.default.httpOverHttp(parseProxy(httpProxy))
            : globalHttpAgent,
        https: httpsProxy
            ? tunnel_1.default.httpsOverHttp(parseProxy(httpsProxy))
            : globalHttpsAgent,
    };
    const gotOptions = { agent, headers, method };
    gotOptions.responseType = json
        ? `json`
        : `buffer`;
    if (body !== null) {
        if (typeof body === `string` || Buffer.isBuffer(body)) {
            gotOptions.body = body;
        }
        else {
            gotOptions.json = body;
        }
    }
    const timeout = configuration.get(`httpTimeout`);
    const retry = configuration.get(`httpRetry`);
    //@ts-ignore
    const gotClient = got_1.default.extend({
        timeout,
        retry,
        ...gotOptions,
    });
    return limit(() => gotClient(target));
}
exports.request = request;
async function get(target, { configuration, json, ...rest }) {
    let entry = cache.get(target);
    if (!entry) {
        entry = request(target, null, { configuration, ...rest }).then(response => {
            cache.set(target, response.body);
            return response.body;
        });
        cache.set(target, entry);
    }
    if (Buffer.isBuffer(entry) === false)
        entry = await entry;
    if (json) {
        return JSON.parse(entry.toString());
    }
    else {
        return entry;
    }
}
exports.get = get;
async function put(target, body, options) {
    const response = await request(target, body, { ...options, method: Method.PUT });
    return response.body;
}
exports.put = put;
