"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getIdentVendorPath = exports.prettyWorkspace = exports.sortDescriptors = exports.prettyLocatorNoColors = exports.prettyLocator = exports.prettyReference = exports.prettyDescriptor = exports.prettyRange = exports.prettyIdent = exports.slugifyLocator = exports.slugifyIdent = exports.stringifyLocator = exports.stringifyDescriptor = exports.stringifyIdent = exports.requirableIdent = exports.convertToManifestRange = exports.makeRange = exports.parseFileStyleRange = exports.parseRange = exports.tryParseLocator = exports.parseLocator = exports.tryParseDescriptor = exports.parseDescriptor = exports.tryParseIdent = exports.parseIdent = exports.areVirtualPackagesEquivalent = exports.areLocatorsEqual = exports.areDescriptorsEqual = exports.areIdentsEqual = exports.bindLocator = exports.bindDescriptor = exports.devirtualizeLocator = exports.devirtualizeDescriptor = exports.isVirtualLocator = exports.isVirtualDescriptor = exports.virtualizePackage = exports.virtualizeDescriptor = exports.copyPackage = exports.renamePackage = exports.convertPackageToLocator = exports.convertLocatorToDescriptor = exports.convertDescriptorToLocator = exports.convertToIdent = exports.makeLocator = exports.makeDescriptor = exports.makeIdent = void 0;
const tslib_1 = require("tslib");
const fslib_1 = require("@yarnpkg/fslib");
const querystring_1 = tslib_1.__importDefault(require("querystring"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const Configuration_1 = require("./Configuration");
const hashUtils = tslib_1.__importStar(require("./hashUtils"));
const miscUtils = tslib_1.__importStar(require("./miscUtils"));
const VIRTUAL_PROTOCOL = `virtual:`;
const VIRTUAL_ABBREVIATE = 5;
function makeIdent(scope, name) {
    return { identHash: hashUtils.makeHash(scope, name), scope, name };
}
exports.makeIdent = makeIdent;
function makeDescriptor(ident, range) {
    return { identHash: ident.identHash, scope: ident.scope, name: ident.name, descriptorHash: hashUtils.makeHash(ident.identHash, range), range };
}
exports.makeDescriptor = makeDescriptor;
function makeLocator(ident, reference) {
    return { identHash: ident.identHash, scope: ident.scope, name: ident.name, locatorHash: hashUtils.makeHash(ident.identHash, reference), reference };
}
exports.makeLocator = makeLocator;
function convertToIdent(source) {
    return { identHash: source.identHash, scope: source.scope, name: source.name };
}
exports.convertToIdent = convertToIdent;
function convertDescriptorToLocator(descriptor) {
    return { identHash: descriptor.identHash, scope: descriptor.scope, name: descriptor.name, locatorHash: descriptor.descriptorHash, reference: descriptor.range };
}
exports.convertDescriptorToLocator = convertDescriptorToLocator;
function convertLocatorToDescriptor(locator) {
    return { identHash: locator.identHash, scope: locator.scope, name: locator.name, descriptorHash: locator.locatorHash, range: locator.reference };
}
exports.convertLocatorToDescriptor = convertLocatorToDescriptor;
function convertPackageToLocator(pkg) {
    return { identHash: pkg.identHash, scope: pkg.scope, name: pkg.name, locatorHash: pkg.locatorHash, reference: pkg.reference };
}
exports.convertPackageToLocator = convertPackageToLocator;
function renamePackage(pkg, locator) {
    return {
        identHash: locator.identHash,
        scope: locator.scope,
        name: locator.name,
        locatorHash: locator.locatorHash,
        reference: locator.reference,
        version: pkg.version,
        languageName: pkg.languageName,
        linkType: pkg.linkType,
        dependencies: new Map(pkg.dependencies),
        peerDependencies: new Map(pkg.peerDependencies),
        dependenciesMeta: new Map(pkg.dependenciesMeta),
        peerDependenciesMeta: new Map(pkg.peerDependenciesMeta),
        bin: new Map(pkg.bin),
    };
}
exports.renamePackage = renamePackage;
function copyPackage(pkg) {
    return renamePackage(pkg, pkg);
}
exports.copyPackage = copyPackage;
function virtualizeDescriptor(descriptor, entropy) {
    if (entropy.includes(`#`))
        throw new Error(`Invalid entropy`);
    return makeDescriptor(descriptor, `virtual:${entropy}#${descriptor.range}`);
}
exports.virtualizeDescriptor = virtualizeDescriptor;
function virtualizePackage(pkg, entropy) {
    if (entropy.includes(`#`))
        throw new Error(`Invalid entropy`);
    return renamePackage(pkg, makeLocator(pkg, `virtual:${entropy}#${pkg.reference}`));
}
exports.virtualizePackage = virtualizePackage;
function isVirtualDescriptor(descriptor) {
    return descriptor.range.startsWith(VIRTUAL_PROTOCOL);
}
exports.isVirtualDescriptor = isVirtualDescriptor;
function isVirtualLocator(locator) {
    return locator.reference.startsWith(VIRTUAL_PROTOCOL);
}
exports.isVirtualLocator = isVirtualLocator;
function devirtualizeDescriptor(descriptor) {
    if (!isVirtualDescriptor(descriptor))
        throw new Error(`Not a virtual descriptor`);
    return makeDescriptor(descriptor, descriptor.range.replace(/^[^#]*#/, ``));
}
exports.devirtualizeDescriptor = devirtualizeDescriptor;
function devirtualizeLocator(locator) {
    if (!isVirtualLocator(locator))
        throw new Error(`Not a virtual descriptor`);
    return makeLocator(locator, locator.reference.replace(/^[^#]*#/, ``));
}
exports.devirtualizeLocator = devirtualizeLocator;
function bindDescriptor(descriptor, params) {
    if (descriptor.range.includes(`::`))
        return descriptor;
    return makeDescriptor(descriptor, `${descriptor.range}::${querystring_1.default.stringify(params)}`);
}
exports.bindDescriptor = bindDescriptor;
function bindLocator(locator, params) {
    if (locator.reference.includes(`::`))
        return locator;
    return makeLocator(locator, `${locator.reference}::${querystring_1.default.stringify(params)}`);
}
exports.bindLocator = bindLocator;
function areIdentsEqual(a, b) {
    return a.identHash === b.identHash;
}
exports.areIdentsEqual = areIdentsEqual;
function areDescriptorsEqual(a, b) {
    return a.descriptorHash === b.descriptorHash;
}
exports.areDescriptorsEqual = areDescriptorsEqual;
function areLocatorsEqual(a, b) {
    return a.locatorHash === b.locatorHash;
}
exports.areLocatorsEqual = areLocatorsEqual;
/**
 * Virtual packages are considered equivalent when they belong to the same
 * package identity and have the same dependencies. Note that equivalence
 * is not the same as equality, as the references may be different.
 */
function areVirtualPackagesEquivalent(a, b) {
    if (!isVirtualLocator(a))
        throw new Error(`Invalid package type`);
    if (!isVirtualLocator(b))
        throw new Error(`Invalid package type`);
    if (!areIdentsEqual(a, b))
        return false;
    if (a.dependencies.size !== b.dependencies.size)
        return false;
    for (const dependencyDescriptorA of a.dependencies.values()) {
        const dependencyDescriptorB = b.dependencies.get(dependencyDescriptorA.identHash);
        if (!dependencyDescriptorB)
            return false;
        if (!areDescriptorsEqual(dependencyDescriptorA, dependencyDescriptorB)) {
            return false;
        }
    }
    return true;
}
exports.areVirtualPackagesEquivalent = areVirtualPackagesEquivalent;
function parseIdent(string) {
    const ident = tryParseIdent(string);
    if (!ident)
        throw new Error(`Invalid ident (${string})`);
    return ident;
}
exports.parseIdent = parseIdent;
function tryParseIdent(string) {
    const match = string.match(/^(?:@([^/]+?)\/)?([^/]+)$/);
    if (!match)
        return null;
    const [, scope, name] = match;
    const realScope = typeof scope !== `undefined`
        ? scope
        : null;
    return makeIdent(realScope, name);
}
exports.tryParseIdent = tryParseIdent;
function parseDescriptor(string, strict = false) {
    const descriptor = tryParseDescriptor(string, strict);
    if (!descriptor)
        throw new Error(`Invalid descriptor (${string})`);
    return descriptor;
}
exports.parseDescriptor = parseDescriptor;
function tryParseDescriptor(string, strict = false) {
    const match = strict
        ? string.match(/^(?:@([^/]+?)\/)?([^/]+?)(?:@(.+))$/)
        : string.match(/^(?:@([^/]+?)\/)?([^/]+?)(?:@(.+))?$/);
    if (!match)
        return null;
    const [, scope, name, range] = match;
    if (range === `unknown`)
        throw new Error(`Invalid range (${string})`);
    const realScope = typeof scope !== `undefined`
        ? scope
        : null;
    const realRange = typeof range !== `undefined`
        ? range
        : `unknown`;
    return makeDescriptor(makeIdent(realScope, name), realRange);
}
exports.tryParseDescriptor = tryParseDescriptor;
function parseLocator(string, strict = false) {
    const locator = tryParseLocator(string, strict);
    if (!locator)
        throw new Error(`Invalid locator (${string})`);
    return locator;
}
exports.parseLocator = parseLocator;
function tryParseLocator(string, strict = false) {
    const match = strict
        ? string.match(/^(?:@([^/]+?)\/)?([^/]+?)(?:@(.+))$/)
        : string.match(/^(?:@([^/]+?)\/)?([^/]+?)(?:@(.+))?$/);
    if (!match)
        return null;
    const [, scope, name, reference] = match;
    if (reference === `unknown`)
        throw new Error(`Invalid reference (${string})`);
    const realScope = typeof scope !== `undefined`
        ? scope
        : null;
    const realReference = typeof reference !== `undefined`
        ? reference
        : `unknown`;
    return makeLocator(makeIdent(realScope, name), realReference);
}
exports.tryParseLocator = tryParseLocator;
function parseRange(range, opts) {
    const match = range.match(/^([^#:]*:)?((?:(?!::)[^#])*)(?:#((?:(?!::).)*))?(?:::(.*))?$/);
    if (match === null)
        throw new Error(`Invalid range (${range})`);
    const protocol = typeof match[1] !== `undefined`
        ? match[1]
        : null;
    if (typeof (opts === null || opts === void 0 ? void 0 : opts.requireProtocol) === `string` && protocol !== opts.requireProtocol)
        throw new Error(`Invalid protocol (${protocol})`);
    else if ((opts === null || opts === void 0 ? void 0 : opts.requireProtocol) && protocol === null)
        throw new Error(`Missing protocol (${protocol})`);
    const source = typeof match[3] !== `undefined`
        ? decodeURIComponent(match[2])
        : null;
    if ((opts === null || opts === void 0 ? void 0 : opts.requireSource) && source === null)
        throw new Error(`Missing source (${range})`);
    const rawSelector = typeof match[3] !== `undefined`
        ? decodeURIComponent(match[3])
        : decodeURIComponent(match[2]);
    const selector = (opts === null || opts === void 0 ? void 0 : opts.parseSelector)
        ? querystring_1.default.parse(rawSelector)
        : rawSelector;
    const params = typeof match[4] !== `undefined`
        ? querystring_1.default.parse(match[4])
        : null;
    return {
        // @ts-ignore
        protocol,
        // @ts-ignore
        source,
        // @ts-ignore
        selector,
        // @ts-ignore
        params,
    };
}
exports.parseRange = parseRange;
function parseFileStyleRange(range, { protocol }) {
    const { selector, params } = parseRange(range, {
        requireProtocol: protocol,
        requireBindings: true,
    });
    if (typeof params.locator !== `string`)
        throw new Error(`Assertion failed: Invalid bindings for ${range}`);
    const parentLocator = parseLocator(params.locator, true);
    const path = selector;
    return { parentLocator, path };
}
exports.parseFileStyleRange = parseFileStyleRange;
function encodeUnsafeCharacters(str) {
    str = str.replace(/%/g, `%25`);
    str = str.replace(/:/g, `%3A`);
    str = str.replace(/#/g, `%23`);
    return str;
}
function hasParams(params) {
    if (params === null)
        return false;
    return Object.entries(params).length > 0;
}
function makeRange({ protocol, source, selector, params }) {
    let range = ``;
    if (protocol !== null)
        range += `${protocol}`;
    if (source !== null)
        range += `${encodeUnsafeCharacters(source)}#`;
    range += encodeUnsafeCharacters(selector);
    if (hasParams(params))
        range += `::${querystring_1.default.stringify(params)}`;
    return range;
}
exports.makeRange = makeRange;
/**
 * The range used internally may differ from the range stored in the
 * Manifest (package.json). This removes any params indicated for internal use.
 * An internal param starts with "__".
 * @param range range to convert
 */
function convertToManifestRange(range) {
    const { params, protocol, source, selector } = parseRange(range);
    for (const name in params)
        if (name.startsWith(`__`))
            delete params[name];
    return makeRange({ protocol, source, params, selector });
}
exports.convertToManifestRange = convertToManifestRange;
function requirableIdent(ident) {
    if (ident.scope) {
        return `@${ident.scope}/${ident.name}`;
    }
    else {
        return `${ident.name}`;
    }
}
exports.requirableIdent = requirableIdent;
function stringifyIdent(ident) {
    if (ident.scope) {
        return `@${ident.scope}/${ident.name}`;
    }
    else {
        return `${ident.name}`;
    }
}
exports.stringifyIdent = stringifyIdent;
function stringifyDescriptor(descriptor) {
    if (descriptor.scope) {
        return `@${descriptor.scope}/${descriptor.name}@${descriptor.range}`;
    }
    else {
        return `${descriptor.name}@${descriptor.range}`;
    }
}
exports.stringifyDescriptor = stringifyDescriptor;
function stringifyLocator(locator) {
    if (locator.scope) {
        return `@${locator.scope}/${locator.name}@${locator.reference}`;
    }
    else {
        return `${locator.name}@${locator.reference}`;
    }
}
exports.stringifyLocator = stringifyLocator;
function slugifyIdent(ident) {
    if (ident.scope !== null) {
        return `@${ident.scope}-${ident.name}`;
    }
    else {
        return ident.name;
    }
}
exports.slugifyIdent = slugifyIdent;
function slugifyLocator(locator) {
    const { protocol, selector } = parseRange(locator.reference);
    const humanProtocol = protocol !== null
        ? protocol.replace(/:$/, ``)
        : `exotic`;
    const humanVersion = semver_1.default.valid(selector);
    const humanReference = humanVersion !== null
        ? `${humanProtocol}-${humanVersion}`
        : `${humanProtocol}`;
    // 10 hex characters means that 47 different entries have 10^-9 chances of
    // causing a hash collision. Since this hash is joined with the package name
    // (making it highly unlikely you'll have more than a handful of instances
    // of any single package), this should provide a good enough guard in most
    // cases.
    //
    // Also note that eCryptfs eats some bytes, so the theoretical maximum for a
    // file size is around 140 bytes (but we don't need as much, as explained).
    const hashTruncate = 10;
    const slug = locator.scope
        ? `${slugifyIdent(locator)}-${humanReference}-${locator.locatorHash.slice(0, hashTruncate)}`
        : `${slugifyIdent(locator)}-${humanReference}-${locator.locatorHash.slice(0, hashTruncate)}`;
    return fslib_1.toFilename(slug);
}
exports.slugifyLocator = slugifyLocator;
function prettyIdent(configuration, ident) {
    if (ident.scope) {
        return `${configuration.format(`@${ident.scope}/`, Configuration_1.FormatType.SCOPE)}${configuration.format(ident.name, Configuration_1.FormatType.NAME)}`;
    }
    else {
        return `${configuration.format(ident.name, Configuration_1.FormatType.NAME)}`;
    }
}
exports.prettyIdent = prettyIdent;
function prettyRangeNoColors(range) {
    if (range.startsWith(VIRTUAL_PROTOCOL)) {
        const nested = prettyRangeNoColors(range.substr(range.indexOf(`#`) + 1));
        const abbrev = range.substr(VIRTUAL_PROTOCOL.length, VIRTUAL_ABBREVIATE);
        // I'm not satisfied of how the virtual packages appear in the output
        // eslint-disable-next-line no-constant-condition
        return false ? `${nested} (virtual:${abbrev})` : `${nested} [${abbrev}]`;
    }
    else {
        return range.replace(/\?.*/, `?[...]`);
    }
}
function prettyRange(configuration, range) {
    return `${configuration.format(prettyRangeNoColors(range), Configuration_1.FormatType.RANGE)}`;
}
exports.prettyRange = prettyRange;
function prettyDescriptor(configuration, descriptor) {
    return `${prettyIdent(configuration, descriptor)}${configuration.format(`@`, Configuration_1.FormatType.RANGE)}${prettyRange(configuration, descriptor.range)}`;
}
exports.prettyDescriptor = prettyDescriptor;
function prettyReference(configuration, reference) {
    return `${configuration.format(prettyRangeNoColors(reference), Configuration_1.FormatType.REFERENCE)}`;
}
exports.prettyReference = prettyReference;
function prettyLocator(configuration, locator) {
    return `${prettyIdent(configuration, locator)}${configuration.format(`@`, Configuration_1.FormatType.REFERENCE)}${prettyReference(configuration, locator.reference)}`;
}
exports.prettyLocator = prettyLocator;
function prettyLocatorNoColors(locator) {
    return `${stringifyIdent(locator)}@${prettyRangeNoColors(locator.reference)}`;
}
exports.prettyLocatorNoColors = prettyLocatorNoColors;
function sortDescriptors(descriptors) {
    return miscUtils.sortMap(descriptors, [
        descriptor => stringifyIdent(descriptor),
        descriptor => descriptor.range,
    ]);
}
exports.sortDescriptors = sortDescriptors;
function prettyWorkspace(configuration, workspace) {
    return prettyIdent(configuration, workspace.locator);
}
exports.prettyWorkspace = prettyWorkspace;
/**
 * The presence of a `node_modules` directory in the path is extremely common
 * in the JavaScript ecosystem to denote whether a path belongs to a vendor
 * or not. I considered using a more generic path for packages that aren't
 * always JS-only (such as when using the Git fetcher), but that unfortunately
 * caused various JS apps to start showing errors when working with git repos.
 *
 * As a result, all packages from all languages will follow this convention. At
 * least it'll be consistent, and linkers will always have the ability to remap
 * them to a different location if that's a critical requirement.
 */
function getIdentVendorPath(ident) {
    return `node_modules/${requirableIdent(ident)}`;
}
exports.getIdentVendorPath = getIdentVendorPath;
