"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractArchiveTo = exports.convertToZip = exports.makeArchiveFromDirectory = void 0;
const fslib_1 = require("@yarnpkg/fslib");
const libzip_1 = require("@yarnpkg/libzip");
const tar_1 = require("tar");
async function makeArchiveFromDirectory(source, { baseFs = new fslib_1.NodeFS(), prefixPath = fslib_1.PortablePath.root, compressionLevel } = {}) {
    const tmpFolder = await fslib_1.xfs.mktempPromise();
    const tmpFile = fslib_1.ppath.join(tmpFolder, `archive.zip`);
    const zipFs = new fslib_1.ZipFS(tmpFile, { create: true, libzip: await libzip_1.getLibzipPromise(), level: compressionLevel });
    const target = fslib_1.ppath.resolve(fslib_1.PortablePath.root, prefixPath);
    await zipFs.copyPromise(target, source, { baseFs, stableTime: true, stableSort: true });
    return zipFs;
}
exports.makeArchiveFromDirectory = makeArchiveFromDirectory;
async function convertToZip(tgz, opts) {
    const tmpFolder = await fslib_1.xfs.mktempPromise();
    const tmpFile = fslib_1.ppath.join(tmpFolder, `archive.zip`);
    const { compressionLevel, ...bufferOpts } = opts;
    return await extractArchiveTo(tgz, new fslib_1.ZipFS(tmpFile, { create: true, libzip: await libzip_1.getLibzipPromise(), level: compressionLevel }), bufferOpts);
}
exports.convertToZip = convertToZip;
async function extractArchiveTo(tgz, targetFs, { stripComponents = 0, prefixPath = fslib_1.PortablePath.dot } = {}) {
    // 1980-01-01, like Fedora
    const defaultTime = 315532800;
    // @ts-ignore: Typescript doesn't want me to use new
    const parser = new tar_1.Parse();
    function ignore(entry) {
        // Disallow absolute paths; might be malicious (ex: /etc/passwd)
        if (entry[0] === `/`)
            return true;
        const parts = entry.path.split(/\//g);
        // We also ignore paths that could lead to escaping outside the archive
        if (parts.some((part) => part === `..`))
            return true;
        if (parts.length <= stripComponents)
            return true;
        return false;
    }
    parser.on(`entry`, (entry) => {
        if (ignore(entry)) {
            entry.resume();
            return;
        }
        const parts = fslib_1.ppath.normalize(fslib_1.npath.toPortablePath(entry.path)).replace(/\/$/, ``).split(/\//g);
        if (parts.length <= stripComponents) {
            entry.resume();
            return;
        }
        const slicePath = parts.slice(stripComponents).join(`/`);
        const mappedPath = fslib_1.ppath.join(prefixPath, slicePath);
        const chunks = [];
        let mode = 0o644;
        // If a single executable bit is set, normalize so that all are
        if (entry.type === `Directory` || (entry.mode & 0o111) !== 0)
            mode |= 0o111;
        entry.on(`data`, (chunk) => {
            chunks.push(chunk);
        });
        entry.on(`end`, () => {
            var _a;
            switch (entry.type) {
                case `Directory`:
                    {
                        targetFs.mkdirpSync(fslib_1.ppath.dirname(mappedPath), { chmod: 0o755, utimes: [defaultTime, defaultTime] });
                        targetFs.mkdirSync(mappedPath);
                        targetFs.chmodSync(mappedPath, mode);
                        targetFs.utimesSync(mappedPath, defaultTime, defaultTime);
                    }
                    break;
                case `OldFile`:
                case `File`:
                    {
                        targetFs.mkdirpSync(fslib_1.ppath.dirname(mappedPath), { chmod: 0o755, utimes: [defaultTime, defaultTime] });
                        targetFs.writeFileSync(mappedPath, Buffer.concat(chunks));
                        targetFs.chmodSync(mappedPath, mode);
                        targetFs.utimesSync(mappedPath, defaultTime, defaultTime);
                    }
                    break;
                case `SymbolicLink`:
                    {
                        targetFs.mkdirpSync(fslib_1.ppath.dirname(mappedPath), { chmod: 0o755, utimes: [defaultTime, defaultTime] });
                        targetFs.symlinkSync(entry.linkpath, mappedPath);
                        (_a = targetFs.lutimesSync) === null || _a === void 0 ? void 0 : _a.call(targetFs, mappedPath, defaultTime, defaultTime);
                    }
                    break;
            }
        });
    });
    return await new Promise((resolve, reject) => {
        parser.on(`error`, (error) => {
            reject(error);
        });
        parser.on(`close`, () => {
            resolve(targetFs);
        });
        parser.end(tgz);
    });
}
exports.extractArchiveTo = extractArchiveTo;
