"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.copyPromise = void 0;
const tslib_1 = require("tslib");
const fs_1 = tslib_1.__importDefault(require("fs"));
const path_1 = require("../path");
// 1980-01-01, like Fedora
const defaultTime = 315532800;
async function copyPromise(destinationFs, destination, sourceFs, source, opts) {
    const normalizedDestination = destinationFs.pathUtils.normalize(destination);
    const normalizedSource = sourceFs.pathUtils.normalize(source);
    const operations = [];
    const lutimes = [];
    await destinationFs.mkdirpPromise(destination);
    await copyImpl(operations, lutimes, destinationFs, normalizedDestination, sourceFs, normalizedSource, opts);
    for (const operation of operations)
        await operation();
    const updateTime = typeof destinationFs.lutimesPromise === `function`
        ? destinationFs.lutimesPromise.bind(destinationFs)
        : destinationFs.utimesPromise.bind(destinationFs);
    for (const [p, atime, mtime] of lutimes) {
        await updateTime(p, atime, mtime);
    }
}
exports.copyPromise = copyPromise;
async function copyImpl(operations, lutimes, destinationFs, destination, sourceFs, source, opts) {
    const destinationStat = await maybeLStat(destinationFs, destination);
    const sourceStat = await sourceFs.lstatPromise(source);
    if (opts.stableTime)
        lutimes.push([destination, defaultTime, defaultTime]);
    else
        lutimes.push([destination, sourceStat.atime, sourceStat.mtime]);
    switch (true) {
        case sourceStat.isDirectory():
            {
                await copyFolder(operations, lutimes, destinationFs, destination, destinationStat, sourceFs, source, sourceStat, opts);
            }
            break;
        case sourceStat.isFile():
            {
                await copyFile(operations, lutimes, destinationFs, destination, destinationStat, sourceFs, source, sourceStat, opts);
            }
            break;
        case sourceStat.isSymbolicLink():
            {
                await copySymlink(operations, lutimes, destinationFs, destination, destinationStat, sourceFs, source, sourceStat, opts);
            }
            break;
        default:
            {
                throw new Error(`Unsupported file type (${sourceStat.mode})`);
            }
            break;
    }
    operations.push(async () => destinationFs.chmodPromise(destination, sourceStat.mode & 0o777));
}
async function maybeLStat(baseFs, p) {
    try {
        return await baseFs.lstatPromise(p);
    }
    catch (e) {
        return null;
    }
}
async function copyFolder(operations, lutimes, destinationFs, destination, destinationStat, sourceFs, source, sourceStat, opts) {
    if (destinationStat !== null && !destinationStat.isDirectory()) {
        if (opts.overwrite) {
            operations.push(async () => destinationFs.removePromise(destination));
            destinationStat = null;
        }
        else {
            return;
        }
    }
    if (destinationStat === null)
        operations.push(async () => destinationFs.mkdirPromise(destination, { mode: sourceStat.mode }));
    const entries = await sourceFs.readdirPromise(source);
    if (opts.stableSort) {
        for (const entry of entries.sort()) {
            await copyImpl(operations, lutimes, destinationFs, destinationFs.pathUtils.join(destination, entry), sourceFs, sourceFs.pathUtils.join(source, entry), opts);
        }
    }
    else {
        await Promise.all(entries.map(async (entry) => {
            await copyImpl(operations, lutimes, destinationFs, destinationFs.pathUtils.join(destination, entry), sourceFs, sourceFs.pathUtils.join(source, entry), opts);
        }));
    }
}
async function copyFile(operations, lutimes, destinationFs, destination, destinationStat, sourceFs, source, sourceStat, opts) {
    if (destinationStat !== null) {
        if (opts.overwrite) {
            operations.push(async () => destinationFs.removePromise(destination));
            destinationStat = null;
        }
        else {
            return;
        }
    }
    if (destinationFs === sourceFs) {
        operations.push(async () => destinationFs.copyFilePromise(source, destination, fs_1.default.constants.COPYFILE_FICLONE));
    }
    else {
        operations.push(async () => destinationFs.writeFilePromise(destination, await sourceFs.readFilePromise(source)));
    }
}
async function copySymlink(operations, lutimes, destinationFs, destination, destinationStat, sourceFs, source, sourceStat, opts) {
    if (destinationStat !== null) {
        if (opts.overwrite) {
            operations.push(async () => destinationFs.removePromise(destination));
            destinationStat = null;
        }
        else {
            return;
        }
    }
    const target = await sourceFs.readlinkPromise(source);
    operations.push(async () => destinationFs.symlinkPromise(path_1.convertPath(destinationFs.pathUtils, target), destination));
}
