"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.xfs = exports.extendFs = exports.patchFs = void 0;
const tslib_1 = require("tslib");
const os_1 = tslib_1.__importDefault(require("os"));
const util_1 = require("util");
const NodeFS_1 = require("./NodeFS");
const path_1 = require("./path");
var FakeFS_1 = require("./FakeFS");
Object.defineProperty(exports, "normalizeLineEndings", { enumerable: true, get: function () { return FakeFS_1.normalizeLineEndings; } });
var ZipFS_1 = require("./ZipFS");
Object.defineProperty(exports, "DEFAULT_COMPRESSION_LEVEL", { enumerable: true, get: function () { return ZipFS_1.DEFAULT_COMPRESSION_LEVEL; } });
var path_2 = require("./path");
Object.defineProperty(exports, "PortablePath", { enumerable: true, get: function () { return path_2.PortablePath; } });
Object.defineProperty(exports, "Filename", { enumerable: true, get: function () { return path_2.Filename; } });
var path_3 = require("./path");
Object.defineProperty(exports, "npath", { enumerable: true, get: function () { return path_3.npath; } });
Object.defineProperty(exports, "ppath", { enumerable: true, get: function () { return path_3.ppath; } });
Object.defineProperty(exports, "toFilename", { enumerable: true, get: function () { return path_3.toFilename; } });
var AliasFS_1 = require("./AliasFS");
Object.defineProperty(exports, "AliasFS", { enumerable: true, get: function () { return AliasFS_1.AliasFS; } });
var FakeFS_2 = require("./FakeFS");
Object.defineProperty(exports, "FakeFS", { enumerable: true, get: function () { return FakeFS_2.FakeFS; } });
var CwdFS_1 = require("./CwdFS");
Object.defineProperty(exports, "CwdFS", { enumerable: true, get: function () { return CwdFS_1.CwdFS; } });
var JailFS_1 = require("./JailFS");
Object.defineProperty(exports, "JailFS", { enumerable: true, get: function () { return JailFS_1.JailFS; } });
var LazyFS_1 = require("./LazyFS");
Object.defineProperty(exports, "LazyFS", { enumerable: true, get: function () { return LazyFS_1.LazyFS; } });
var NoFS_1 = require("./NoFS");
Object.defineProperty(exports, "NoFS", { enumerable: true, get: function () { return NoFS_1.NoFS; } });
var NodeFS_2 = require("./NodeFS");
Object.defineProperty(exports, "NodeFS", { enumerable: true, get: function () { return NodeFS_2.NodeFS; } });
var PosixFS_1 = require("./PosixFS");
Object.defineProperty(exports, "PosixFS", { enumerable: true, get: function () { return PosixFS_1.PosixFS; } });
var ProxiedFS_1 = require("./ProxiedFS");
Object.defineProperty(exports, "ProxiedFS", { enumerable: true, get: function () { return ProxiedFS_1.ProxiedFS; } });
var VirtualFS_1 = require("./VirtualFS");
Object.defineProperty(exports, "VirtualFS", { enumerable: true, get: function () { return VirtualFS_1.VirtualFS; } });
var ZipFS_2 = require("./ZipFS");
Object.defineProperty(exports, "ZipFS", { enumerable: true, get: function () { return ZipFS_2.ZipFS; } });
var ZipOpenFS_1 = require("./ZipOpenFS");
Object.defineProperty(exports, "ZipOpenFS", { enumerable: true, get: function () { return ZipOpenFS_1.ZipOpenFS; } });
function getTempName(prefix) {
    const tmpdir = path_1.npath.toPortablePath(os_1.default.tmpdir());
    const hash = Math.ceil(Math.random() * 0x100000000).toString(16).padStart(8, `0`);
    return path_1.ppath.join(tmpdir, `${prefix}${hash}`);
}
function patchFs(patchedFs, fakeFs) {
    const SYNC_IMPLEMENTATIONS = new Set([
        `accessSync`,
        `appendFileSync`,
        `createReadStream`,
        `chmodSync`,
        `closeSync`,
        `copyFileSync`,
        `lstatSync`,
        `lutimesSync`,
        `mkdirSync`,
        `openSync`,
        `readSync`,
        `readlinkSync`,
        `readFileSync`,
        `readdirSync`,
        `readlinkSync`,
        `realpathSync`,
        `renameSync`,
        `rmdirSync`,
        `statSync`,
        `symlinkSync`,
        `unlinkSync`,
        `utimesSync`,
        `watch`,
        `writeFileSync`,
        `writeSync`,
    ]);
    const ASYNC_IMPLEMENTATIONS = new Set([
        `accessPromise`,
        `appendFilePromise`,
        `chmodPromise`,
        `closePromise`,
        `copyFilePromise`,
        `lstatPromise`,
        `lutimesPromise`,
        `mkdirPromise`,
        `openPromise`,
        `readdirPromise`,
        `realpathPromise`,
        `readFilePromise`,
        `readdirPromise`,
        `readlinkPromise`,
        `renamePromise`,
        `rmdirPromise`,
        `statPromise`,
        `symlinkPromise`,
        `unlinkPromise`,
        `utimesPromise`,
        `writeFilePromise`,
        `writeSync`,
    ]);
    const setupFn = (target, name, replacement) => {
        const orig = target[name];
        target[name] = replacement;
        if (typeof orig[util_1.promisify.custom] !== `undefined`) {
            replacement[util_1.promisify.custom] = orig[util_1.promisify.custom];
        }
    };
    /** Callback implementations */
    {
        setupFn(patchedFs, `exists`, (p, ...args) => {
            const hasCallback = typeof args[args.length - 1] === `function`;
            const callback = hasCallback ? args.pop() : () => { };
            process.nextTick(() => {
                fakeFs.existsPromise(p).then(exists => {
                    callback(exists);
                }, () => {
                    callback(false);
                });
            });
        });
        setupFn(patchedFs, `read`, (p, buffer, ...args) => {
            const hasCallback = typeof args[args.length - 1] === `function`;
            const callback = hasCallback ? args.pop() : () => { };
            process.nextTick(() => {
                fakeFs.readPromise(p, buffer, ...args).then(bytesRead => {
                    callback(null, bytesRead, buffer);
                }, error => {
                    callback(error);
                });
            });
        });
        for (const fnName of ASYNC_IMPLEMENTATIONS) {
            const origName = fnName.replace(/Promise$/, ``);
            if (typeof patchedFs[origName] === `undefined`)
                continue;
            const fakeImpl = fakeFs[fnName];
            if (typeof fakeImpl === `undefined`)
                continue;
            const wrapper = (...args) => {
                const hasCallback = typeof args[args.length - 1] === `function`;
                const callback = hasCallback ? args.pop() : () => { };
                process.nextTick(() => {
                    fakeImpl.apply(fakeFs, args).then((result) => {
                        callback(null, result);
                    }, (error) => {
                        callback(error);
                    });
                });
            };
            setupFn(patchedFs, origName, wrapper);
        }
        patchedFs.realpath.native = patchedFs.realpath;
    }
    /** Sync implementations */
    {
        setupFn(patchedFs, `existsSync`, (p) => {
            try {
                return fakeFs.existsSync(p);
            }
            catch (error) {
                return false;
            }
        });
        for (const fnName of SYNC_IMPLEMENTATIONS) {
            const origName = fnName;
            if (typeof patchedFs[origName] === `undefined`)
                continue;
            const fakeImpl = fakeFs[fnName];
            if (typeof fakeImpl === `undefined`)
                continue;
            setupFn(patchedFs, origName, fakeImpl.bind(fakeFs));
        }
        patchedFs.realpathSync.native = patchedFs.realpathSync;
    }
    /** Promise implementations */
    {
        // `fs.promises` is a getter that returns a reference to require(`fs/promises`),
        // so we can just patch `fs.promises` and both will be updated
        const patchedFsPromises = patchedFs.promises;
        if (typeof patchedFsPromises !== `undefined`) {
            // `fs.promises.exists` doesn't exist
            for (const fnName of ASYNC_IMPLEMENTATIONS) {
                const origName = fnName.replace(/Promise$/, ``);
                if (typeof patchedFsPromises[origName] === `undefined`)
                    continue;
                const fakeImpl = fakeFs[fnName];
                if (typeof fakeImpl === `undefined`)
                    continue;
                setupFn(patchedFsPromises, origName, fakeImpl.bind(fakeFs));
            }
            // `fs.promises.realpath` doesn't have a `native` property
        }
    }
}
exports.patchFs = patchFs;
function extendFs(realFs, fakeFs) {
    const patchedFs = Object.create(realFs);
    patchFs(patchedFs, fakeFs);
    return patchedFs;
}
exports.extendFs = extendFs;
const tmpdirs = new Set();
let cleanExitRegistered = false;
function registerCleanExit() {
    if (!cleanExitRegistered)
        cleanExitRegistered = true;
    else
        return;
    const cleanExit = () => {
        process.off(`exit`, cleanExit);
        for (const p of tmpdirs) {
            tmpdirs.delete(p);
            try {
                exports.xfs.removeSync(p);
            }
            catch (_a) {
                // Too bad if there's an error
            }
        }
    };
    process.on(`exit`, cleanExit);
}
exports.xfs = Object.assign(new NodeFS_1.NodeFS(), {
    detachTemp(p) {
        tmpdirs.delete(p);
    },
    mktempSync(cb) {
        registerCleanExit();
        while (true) {
            const p = getTempName(`xfs-`);
            try {
                this.mkdirSync(p);
            }
            catch (error) {
                if (error.code === `EEXIST`) {
                    continue;
                }
                else {
                    throw error;
                }
            }
            const realP = this.realpathSync(p);
            tmpdirs.add(realP);
            if (typeof cb !== `undefined`) {
                try {
                    return cb(realP);
                }
                finally {
                    if (tmpdirs.has(realP)) {
                        tmpdirs.delete(realP);
                        try {
                            this.removeSync(realP);
                        }
                        catch (_a) {
                            // Too bad if there's an error
                        }
                    }
                }
            }
            else {
                return p;
            }
        }
    },
    async mktempPromise(cb) {
        registerCleanExit();
        while (true) {
            const p = getTempName(`xfs-`);
            try {
                await this.mkdirPromise(p);
            }
            catch (error) {
                if (error.code === `EEXIST`) {
                    continue;
                }
                else {
                    throw error;
                }
            }
            const realP = await this.realpathPromise(p);
            tmpdirs.add(realP);
            if (typeof cb !== `undefined`) {
                try {
                    return await cb(realP);
                }
                finally {
                    if (tmpdirs.has(realP)) {
                        tmpdirs.delete(realP);
                        try {
                            await this.removePromise(realP);
                        }
                        catch (_a) {
                            // Too bad if there's an error
                        }
                    }
                }
            }
            else {
                return realP;
            }
        }
    },
});
