// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import assert from 'assert';

/**
 * "Normalizes" complete or partial GeoJSON data into iterable list of features
 * Can accept GeoJSON geometry or "Feature", "FeatureCollection" in addition
 * to plain arrays and iterables.
 * Works by extracting the feature array or wrapping single objects in an array,
 * so that subsequent code can simply iterate over features.
 *
 * @param {object} geojson - geojson data
 * @param {Object|Array} data - geojson object (FeatureCollection, Feature or
 *  Geometry) or array of features
 * @return {Array|"iteratable"} - iterable list of features
 */
export function getGeojsonFeatures(geojson) {
  // If array, assume this is a list of features
  if (Array.isArray(geojson)) {
    return geojson;
  }

  assert(geojson.type, 'GeoJSON does not have type');

  switch (geojson.type) {
    case 'GeometryCollection':
      assert(Array.isArray(geojson.geometries), 'GeoJSON does not have geometries array');
      return geojson.geometries.map(function (geometry) {
        return { geometry: geometry };
      });
    case 'Feature':
      // Wrap the feature in a 'Features' array
      return [geojson];
    case 'FeatureCollection':
      // Just return the 'Features' array from the collection
      assert(Array.isArray(geojson.features), 'GeoJSON does not have features array');
      return geojson.features;
    default:
      // Assume it's a geometry, we'll check type in separateGeojsonFeatures
      // Wrap the geometry object in a 'Feature' object and wrap in an array
      return [{ geometry: geojson }];
  }
}

// Linearize
export function separateGeojsonFeatures(features) {
  var pointFeatures = [];
  var lineFeatures = [];
  var polygonFeatures = [];
  var polygonOutlineFeatures = [];

  features.forEach(function (feature) {
    assert(feature && feature.geometry, 'GeoJSON does not have geometry');

    var _feature$geometry = feature.geometry,
        type = _feature$geometry.type,
        coordinates = _feature$geometry.coordinates,
        properties = feature.properties;

    checkCoordinates(type, coordinates);

    switch (type) {
      case 'Point':
        pointFeatures.push(feature);
        break;
      case 'MultiPoint':
        // TODO - split multipoints
        coordinates.forEach(function (point) {
          pointFeatures.push({ geometry: { coordinates: point }, properties: properties, feature: feature });
        });
        break;
      case 'LineString':
        lineFeatures.push(feature);
        break;
      case 'MultiLineString':
        // Break multilinestrings into multiple lines with same properties
        coordinates.forEach(function (path) {
          lineFeatures.push({ geometry: { coordinates: path }, properties: properties, feature: feature });
        });
        break;
      case 'Polygon':
        polygonFeatures.push(feature);
        // Break polygon into multiple lines with same properties
        coordinates.forEach(function (path) {
          polygonOutlineFeatures.push({ geometry: { coordinates: path }, properties: properties, feature: feature });
        });
        break;
      case 'MultiPolygon':
        // Break multipolygons into multiple polygons with same properties
        coordinates.forEach(function (polygon) {
          polygonFeatures.push({ geometry: { coordinates: polygon }, properties: properties, feature: feature });
          // Break polygon into multiple lines with same properties
          polygon.forEach(function (path) {
            polygonOutlineFeatures.push({ geometry: { coordinates: path }, properties: properties, feature: feature });
          });
        });
        break;
      default:
    }
  });

  return {
    pointFeatures: pointFeatures,
    lineFeatures: lineFeatures,
    polygonFeatures: polygonFeatures,
    polygonOutlineFeatures: polygonOutlineFeatures
  };
}

/**
 * Simple GeoJSON validation util. For perf reasons we do not validate against the full spec,
 * only the following:
   - geometry.type is supported
   - geometry.coordinate has correct nesting level
 */
var COORDINATE_NEST_LEVEL = {
  Point: 1,
  MultiPoint: 2,
  LineString: 2,
  MultiLineString: 3,
  Polygon: 3,
  MultiPolygon: 4
};

function checkCoordinates(type, coordinates) {
  var nestLevel = COORDINATE_NEST_LEVEL[type];

  assert(nestLevel, 'Unknown GeoJSON type ' + type);

  while (coordinates && --nestLevel > 0) {
    coordinates = coordinates[0];
  }
  assert(coordinates && Number.isFinite(coordinates[0]), type + ' coordinates are malformed');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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