// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

export default "#define SHADER_NAME hexagon-cell-layer-vertex-shader-64\n\nattribute vec3 positions;\nattribute vec3 normals;\n\nattribute vec3 instancePositions;\nattribute vec2 instancePositions64xyLow;\nattribute vec4 instanceColors;\nattribute vec3 instancePickingColors;\n\n// Custom uniforms\nuniform float opacity;\nuniform float radius;\nuniform float angle;\nuniform float extruded;\nuniform float coverage;\nuniform float elevationScale;\n\n// Result\nvarying vec4 vColor;\n\n// A magic number to scale elevation so that 1 unit approximate to 1 meter.\n#define ELEVATION_SCALE 0.8\n\nvoid main(void) {\n\n  // rotate primitive position and normal\n  mat2 rotationMatrix = mat2(cos(angle), -sin(angle), sin(angle), cos(angle));\n\n  vec2 rPos = rotationMatrix * positions.xz;\n  vec2 rNorm = rotationMatrix * normals.xz; // the hexagon cells has y axis as the vertical axis\n\n  vec3 rotatedPositions = vec3(rPos.x, positions.y, rPos.y);\n  vec3 rotatedNormals = vec3(rNorm.x, rNorm.y, normals.y);\n\n  // calculate elevation, if 3d not enabled set to 0\n  // cylindar gemoetry height are between -0.5 to 0.5, transform it to between 0, 1\n  float elevation = 0.0;\n\n  if (extruded > 0.5) {\n    elevation = project_scale(instancePositions.z * (positions.y + 0.5) *\n      ELEVATION_SCALE * elevationScale);\n  }\n\n  // if alpha == 0.0 or z < 0.0, do not render element\n  float noRender = float(instanceColors.a == 0.0 || instancePositions.z < 0.0);\n  float dotRadius = radius * mix(coverage, 0.0, noRender);\n\n  // project center of hexagon\n\n  vec4 instancePositions64xy = vec4(\n    instancePositions.x, instancePositions64xyLow.x,\n    instancePositions.y, instancePositions64xyLow.y);\n\n  vec2 projected_coord_xy[2];\n  project_position_fp64(instancePositions64xy, projected_coord_xy);\n\n  vec2 vertex_pos_localspace[4];\n  vec4_fp64(vec4(rotatedPositions.xz * dotRadius, 0.0, 1.0), vertex_pos_localspace);\n\n  vec2 vertex_pos_modelspace[4];\n  vertex_pos_modelspace[0] = sum_fp64(vertex_pos_localspace[0], projected_coord_xy[0]);\n  vertex_pos_modelspace[1] = sum_fp64(vertex_pos_localspace[1], projected_coord_xy[1]);\n  vertex_pos_modelspace[2] = sum_fp64(vertex_pos_localspace[2], vec2(elevation, 0.0));\n  vertex_pos_modelspace[3] = vec2(1.0, 0.0);\n\n  vec4 position_worldspace = vec4(\n    vertex_pos_modelspace[0].x, vertex_pos_modelspace[1].x,\n    vertex_pos_modelspace[2].x, vertex_pos_modelspace[3].x);\n\n  gl_Position = project_to_clipspace_fp64(vertex_pos_modelspace);\n\n  // Light calculations\n  // Worldspace is the linear space after Mercator projection\n\n  vec3 normals_worldspace = rotatedNormals;\n\n  float lightWeight = 1.0;\n\n  if (extruded > 0.5) {\n    lightWeight = getLightWeight(\n      position_worldspace.xyz, // the w component is always 1.0\n      normals_worldspace\n    );\n  }\n\n  vec3 lightWeightedColor = lightWeight * instanceColors.rgb;\n\n  // Color: Either opacity-multiplied instance color, or picking color\n  vColor = vec4(lightWeightedColor, opacity * instanceColors.a) / 255.0;\n\n  // Set color to be rendered to picking fbo (also used to check for selection highlight).\n  picking_setPickingColor(instancePickingColors);\n}\n";
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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