// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

export default "\n#define SHADER_NAME path-layer-vertex-shader\n\nattribute vec3 positions;\n\nattribute vec3 instanceStartPositions;\nattribute vec3 instanceEndPositions;\nattribute vec4 instanceStartEndPositions64xyLow;\nattribute vec3 instanceLeftDeltas;\nattribute vec3 instanceRightDeltas;\nattribute float instanceStrokeWidths;\nattribute vec4 instanceColors;\nattribute vec3 instancePickingColors;\nattribute vec2 instanceDashArrays;\n\nuniform float widthScale;\nuniform float widthMinPixels;\nuniform float widthMaxPixels;\nuniform float jointType;\nuniform float miterLimit;\n\nuniform float opacity;\n\nvarying vec4 vColor;\nvarying vec2 vCornerOffset;\nvarying float vMiterLength;\nvarying vec2 vDashArray;\nvarying float vPathPosition;\nvarying float vPathLength;\n\nconst float EPSILON = 0.001;\nconst float PIXEL_EPSILON = 0.1;\n\nfloat flipIfTrue(bool flag) {\n  return -(float(flag) * 2. - 1.);\n}\n\nvec3 lineJoin(vec2 prevPoint64[2], vec2 currPoint64[2], vec2 nextPoint64[2]) {\n\n  float width = clamp(project_scale(instanceStrokeWidths * widthScale),\n    widthMinPixels, widthMaxPixels) / 2.0;\n\n  vec2 deltaA64[2];\n  vec2 deltaB64[2];\n\n  vec2_sub_fp64(currPoint64, prevPoint64, deltaA64);\n  vec2_sub_fp64(nextPoint64, currPoint64, deltaB64);\n\n  vec2 lengthA64 = vec2_length_fp64(deltaA64);\n  vec2 lengthB64 = vec2_length_fp64(deltaB64);\n\n  vec2 deltaA = vec2(deltaA64[0].x, deltaA64[1].x);\n  vec2 deltaB = vec2(deltaB64[0].x, deltaB64[1].x);\n\n  float lenA = lengthA64.x;\n  float lenB = lengthB64.x;\n\n  vec2 offsetVec;\n  float offsetScale;\n  float offsetDirection;\n\n  // when two points are closer than PIXEL_EPSILON in pixels,\n  // assume they are the same point to avoid precision issue\n  lenA = lenA > PIXEL_EPSILON ? lenA : 0.0;\n  lenB = lenB > PIXEL_EPSILON ? lenB : 0.0;\n  vec2 dirA = lenA > 0. ? deltaA / lenA : vec2(0.0, 0.0);\n  vec2 dirB = lenB > 0. ? deltaB / lenB : vec2(0.0, 0.0);\n\n  vec2 perpA = vec2(-dirA.y, dirA.x);\n  vec2 perpB = vec2(-dirB.y, dirB.x);\n\n  // tangent of the corner\n  vec2 tangent = vec2(dirA + dirB);\n  tangent = length(tangent) > 0. ? normalize(tangent) : perpA;\n  // direction of the corner\n  vec2 miterVec = vec2(-tangent.y, tangent.x);\n  // width offset from current position\n  vec2 perp = mix(perpB, perpA, positions.x);\n  float L = mix(lenB, lenA, positions.x);\n\n  // cap super sharp angles\n  float sinHalfA = abs(dot(miterVec, perp));\n  float cosHalfA = abs(dot(dirA, miterVec));\n  bool turnsRight = dirA.x * dirB.y > dirA.y * dirB.x;\n\n  // relative position to the corner:\n  // -1: inside (smaller side of the angle)\n  // 0: center\n  // 1: outside (bigger side of the angle)\n  float cornerPosition = mix(\n    flipIfTrue(turnsRight == (positions.y > 0.0)),\n    0.0,\n    positions.z\n  );\n\n  offsetScale = 1.0 / max(sinHalfA, EPSILON);\n\n  // do not bevel if line segment is too short\n  cornerPosition *= float(cornerPosition <= 0.0 || sinHalfA < min(lenA, lenB) / width * cosHalfA);\n  // trim if inside corner extends further than the line segment\n  offsetScale = mix(\n    offsetScale,\n    min(offsetScale, L / width / max(cosHalfA, EPSILON)),\n    float(cornerPosition < 0.0)\n  );\n\n  vMiterLength = mix(\n    offsetScale * cornerPosition,\n    mix(offsetScale, 0.0, cornerPosition),\n    step(0.0, cornerPosition)\n  ) - sinHalfA * jointType;\n  offsetDirection = mix(\n    positions.y,\n    mix(\n      flipIfTrue(turnsRight),\n      positions.y * flipIfTrue(turnsRight == (positions.x == 1.)),\n      cornerPosition\n    ),\n    step(0.0, cornerPosition)\n  );\n  offsetVec = mix(miterVec, -tangent, step(0.5, cornerPosition));\n  offsetScale = mix(offsetScale, 1.0 / max(cosHalfA, 0.001), step(0.5, cornerPosition));\n\n  // special treatment for start cap and end cap\n  // using a small number as the limit for determining if the lenA or lenB is 0\n  float isStartCap = step(lenA, 1.0e-5);\n  float isEndCap = step(lenB, 1.0e-5);\n  float isCap = max(isStartCap, isEndCap);\n\n  // 0: center, 1: side\n  cornerPosition = isCap * (1.0 - positions.z);\n\n  // start of path: use next - curr\n  offsetVec = mix(offsetVec, mix(dirB, perpB, cornerPosition), isStartCap);\n  // end of path: use curr - prev\n  offsetVec = mix(offsetVec, mix(dirA, perpA, cornerPosition), isEndCap);\n\n  // extend out a triangle to envelope the round cap\n  offsetScale = mix(\n    offsetScale,\n    mix(4.0 * jointType, 1.0, cornerPosition),\n    isCap\n  );\n  vMiterLength = mix(vMiterLength, 1.0 - cornerPosition, isCap);\n\n  offsetDirection = mix(\n    offsetDirection,\n    mix(flipIfTrue(isStartCap > 0.), positions.y, cornerPosition),\n    isCap\n  );\n\n  vCornerOffset = offsetVec * offsetDirection * offsetScale;\n\n  // Generate variables for dash calculation\n  vDashArray = instanceDashArrays;\n  vPathLength = L / width;\n  float isEnd = positions.x;\n  vec2 offsetFromStartOfPath = mix(vCornerOffset, vCornerOffset + deltaA / width, isEnd);\n  vec2 dir = mix(dirB, dirA, isEnd);\n  vPathPosition = dot(offsetFromStartOfPath, dir);\n\n  return vec3(vCornerOffset * width, 0.0);\n}\n\nvoid main() {\n  vColor = vec4(instanceColors.rgb, instanceColors.a * opacity) / 255.;\n\n  // Set color to be rendered to picking fbo (also used to check for selection highlight).\n  picking_setPickingColor(instancePickingColors);\n\n  float isEnd = positions.x;\n\n  // Calculate current position\n  vec3 currPosition = mix(instanceStartPositions, instanceEndPositions, isEnd);\n  currPosition = project_position(currPosition);\n\n  // Calculate current position 64bit\n\n  vec2 instanceStartPositions64[2];\n  instanceStartPositions64[0] = vec2(instanceStartPositions.x, instanceStartEndPositions64xyLow.x);\n  instanceStartPositions64[1] = vec2(instanceStartPositions.y, instanceStartEndPositions64xyLow.y);\n\n  vec2 instanceEndPositions64[2];\n  instanceEndPositions64[0] = vec2(instanceEndPositions.x, instanceStartEndPositions64xyLow.z);\n  instanceEndPositions64[1] = vec2(instanceEndPositions.y, instanceStartEndPositions64xyLow.w);\n\n  vec2 tempCurrPosition64[2];\n  vec2_mix_fp64(instanceStartPositions64, instanceEndPositions64, isEnd, tempCurrPosition64);\n\n  vec4 currPosition64 = vec4(tempCurrPosition64[0].xy, tempCurrPosition64[1].xy);\n\n  vec2 projected_curr_position[2];\n  project_position_fp64(currPosition64, projected_curr_position);\n  float projected_curr_position_z = project_scale(currPosition.z);\n\n  // Calculate previous position\n\n  vec3 prevPosition = mix(-instanceLeftDeltas, vec3(0.0), isEnd);\n\n  // Calculate prev position 64bit\n\n  vec2 tempPrevPosition64[2];\n  tempPrevPosition64[0] = sum_fp64(vec2(prevPosition.x, 0.0), instanceStartPositions64[0]);\n  tempPrevPosition64[1] = sum_fp64(vec2(prevPosition.y, 0.0), instanceStartPositions64[1]);\n\n  vec4 prevPosition64 = vec4(tempPrevPosition64[0].xy, tempPrevPosition64[1].xy);\n\n  vec2 projected_prev_position[2];\n  project_position_fp64(prevPosition64, projected_prev_position);\n\n  // Calculate next positions\n  vec3 nextPosition = mix(vec3(0.0), instanceRightDeltas, isEnd);\n\n  // Calculate next position 64bit\n\n  vec2 tempNextPosition64[2];\n  tempNextPosition64[0] = sum_fp64(vec2(nextPosition.x, 0.0), instanceEndPositions64[0]);\n  tempNextPosition64[1] = sum_fp64(vec2(nextPosition.y, 0.0), instanceEndPositions64[1]);\n\n  vec4 nextPosition64 = vec4(tempNextPosition64[0].xy, tempNextPosition64[1].xy);\n\n  vec2 projected_next_position[2];\n  project_position_fp64(nextPosition64, projected_next_position);\n\n  vec3 pos = lineJoin(projected_prev_position, projected_curr_position, projected_next_position);\n  vec2 vertex_pos_modelspace[4];\n\n  vertex_pos_modelspace[0] = sum_fp64(vec2(pos.x, 0.0), projected_curr_position[0]);\n  vertex_pos_modelspace[1] = sum_fp64(vec2(pos.y, 0.0), projected_curr_position[1]);\n  vertex_pos_modelspace[2] = vec2(pos.z + projected_curr_position_z, 0.0);\n  vertex_pos_modelspace[3] = vec2(1.0, 0.0);\n\n  gl_Position = project_to_clipspace_fp64(vertex_pos_modelspace);\n}\n";
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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