// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

export default "#define SHADER_NAME path-layer-vertex-shader-64\n\nattribute vec3 positions;\n\nattribute vec3 instanceStartPositions;\nattribute vec3 instanceEndPositions;\nattribute vec3 instanceLeftDeltas;\nattribute vec3 instanceRightDeltas;\nattribute float instanceStrokeWidths;\nattribute vec4 instanceColors;\nattribute vec3 instancePickingColors;\nattribute vec2 instanceDashArrays;\n\nuniform float widthScale;\nuniform float widthMinPixels;\nuniform float widthMaxPixels;\nuniform float jointType;\nuniform float miterLimit;\n\nuniform float opacity;\n\nvarying vec4 vColor;\nvarying vec2 vCornerOffset;\nvarying float vMiterLength;\nvarying vec2 vDashArray;\nvarying float vPathPosition;\nvarying float vPathLength;\n\nconst float EPSILON = 0.001;\nconst float PIXEL_EPSILON = 0.1;\n\nfloat flipIfTrue(bool flag) {\n  return -(float(flag) * 2. - 1.);\n}\n\n// calculate line join positions\nvec3 lineJoin(\n  vec3 prevPoint, vec3 currPoint, vec3 nextPoint,\n  float relativePosition, bool isEnd, bool isJoint,\n  float width\n) {\n  vec2 deltaA = currPoint.xy - prevPoint.xy;\n  vec2 deltaB = nextPoint.xy - currPoint.xy;\n\n  float lenA = length(deltaA);\n  float lenB = length(deltaB);\n\n  // when two points are closer than PIXEL_EPSILON in pixels,\n  // assume they are the same point to avoid precision issue\n  lenA = lenA > PIXEL_EPSILON ? lenA : 0.0;\n  lenB = lenB > PIXEL_EPSILON ? lenB : 0.0;\n\n  vec2 dirA = lenA > 0. ? normalize(deltaA) : vec2(0.0, 0.0);\n  vec2 dirB = lenB > 0. ? normalize(deltaB) : vec2(0.0, 0.0);\n\n  vec2 perpA = vec2(-dirA.y, dirA.x);\n  vec2 perpB = vec2(-dirB.y, dirB.x);\n\n  // tangent of the corner\n  vec2 tangent = vec2(dirA + dirB);\n  tangent = length(tangent) > 0. ? normalize(tangent) : perpA;\n  // direction of the corner\n  vec2 miterVec = vec2(-tangent.y, tangent.x);\n  // width offset from current position\n  vec2 perp = isEnd ? perpA : perpB;\n  float L = isEnd ? lenA : lenB;\n\n  // cap super sharp angles\n  float sinHalfA = abs(dot(miterVec, perp));\n  float cosHalfA = abs(dot(dirA, miterVec));\n\n  bool turnsRight = dirA.x * dirB.y > dirA.y * dirB.x;\n\n  float offsetScale = 1.0 / max(sinHalfA, EPSILON);\n\n  float cornerPosition = isJoint ?\n    0.0 :\n    flipIfTrue(turnsRight == (relativePosition > 0.0));\n\n  // do not bevel if line segment is too short\n  cornerPosition *=\n    float(cornerPosition <= 0.0 || sinHalfA < min(lenA, lenB) / width * cosHalfA);\n\n  // trim if inside corner extends further than the line segment\n  if (cornerPosition < 0.0) {\n    offsetScale = min(offsetScale, L / width / max(cosHalfA, EPSILON));\n  }\n\n  vMiterLength = cornerPosition >= 0.0 ?\n    mix(offsetScale, 0.0, cornerPosition) :\n    offsetScale * cornerPosition;\n  vMiterLength -= sinHalfA * jointType;\n\n  float offsetDirection = mix(\n    positions.y,\n    mix(\n      flipIfTrue(turnsRight),\n      positions.y * flipIfTrue(turnsRight == (positions.x == 1.)),\n      cornerPosition\n    ),\n    step(0.0, cornerPosition)\n  );\n\n  vec2 offsetVec = mix(miterVec, -tangent, step(0.5, cornerPosition));\n  offsetScale = mix(offsetScale, 1.0 / max(cosHalfA, 0.001), step(0.5, cornerPosition));\n\n  // special treatment for start cap and end cap\n  // TODO - This has an issue. len is always positive because it is length.\n  // Step returns zero if -lenA<0, so practically this is a comparison of\n  // lenA with zero, with lots of problems because of the -lenA. Can we use EPSILON?\n  bool isStartCap = step(0.0, -lenA) > 0.5;\n  bool isEndCap = step(0.0, -lenB) > 0.5;\n  bool isCap = isStartCap || isEndCap;\n\n  // 0: center, 1: side\n  cornerPosition = isCap ? (1.0 - positions.z) : 0.;\n\n  // start of path: use next - curr\n  if (isStartCap) {\n    offsetVec = mix(dirB, perpB, cornerPosition);\n  }\n\n  // end of path: use curr - prev\n  if (isEndCap) {\n    offsetVec = mix(dirA, perpA, cornerPosition);\n  }\n\n  // extend out a triangle to envelope the round cap\n  if (isCap) {\n    offsetScale = mix(4.0 * jointType, 1.0, cornerPosition);\n    vMiterLength = 1.0 - cornerPosition;\n    offsetDirection = mix(flipIfTrue(isStartCap), positions.y, cornerPosition);\n  }\n\n  vCornerOffset = offsetVec * offsetDirection * offsetScale;\n\n  // Generate variables for dash calculation\n  vDashArray = instanceDashArrays;\n  vPathLength = L / width;\n  // vec2 offsetFromStartOfPath = isEnd ? vCornerOffset + deltaA / width : vCornerOffset;\n  vec2 offsetFromStartOfPath = vCornerOffset;\n  if (isEnd) {\n    offsetFromStartOfPath += deltaA / width;\n  }\n  vec2 dir = isEnd ? dirA : dirB;\n  vPathPosition = dot(offsetFromStartOfPath, dir);\n\n  return currPoint + vec3(vCornerOffset * width, 0.0);\n}\n\n// calculate line join positions\n// extract params from attributes and uniforms\nvec3 lineJoin(vec3 prevPoint, vec3 currPoint, vec3 nextPoint) {\n\n  // relative position to the corner:\n  // -1: inside (smaller side of the angle)\n  // 0: center\n  // 1: outside (bigger side of the angle)\n\n  float relativePosition = positions.y;\n  bool isEnd = positions.x > EPSILON;\n  bool isJoint = positions.z > EPSILON;\n\n  float width = clamp(project_scale(instanceStrokeWidths * widthScale),\n    widthMinPixels, widthMaxPixels) / 2.0;\n\n  return lineJoin(\n    prevPoint, currPoint, nextPoint,\n    relativePosition, isEnd, isJoint,\n    width\n  );\n}\n\nvoid main() {\n  vColor = vec4(instanceColors.rgb, instanceColors.a * opacity) / 255.;\n\n  // Set color to be rendered to picking fbo (also used to check for selection highlight).\n  picking_setPickingColor(instancePickingColors);\n\n  float isEnd = positions.x;\n\n  vec3 prevPosition = mix(-instanceLeftDeltas, vec3(0.0), isEnd) + instanceStartPositions;\n  prevPosition = project_position(prevPosition);\n\n  vec3 currPosition = mix(instanceStartPositions, instanceEndPositions, isEnd);\n  currPosition = project_position(currPosition);\n\n  vec3 nextPosition = mix(vec3(0.0), instanceRightDeltas, isEnd) + instanceEndPositions;\n  nextPosition = project_position(nextPosition);\n\n  vec3 pos = lineJoin(prevPosition, currPosition, nextPosition);\n\n  gl_Position = project_to_clipspace(vec4(pos, 1.0));\n}\n";
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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