var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

// Handles tesselation of polygons with holes
// - 2D surfaces
// - 2D outlines
// - 3D surfaces (top and sides only)
// - 3D wireframes (not yet)
import * as Polygon from './polygon';
import earcut from 'earcut';
import { experimental } from '../../core';
var fp64LowPart = experimental.fp64LowPart,
    flattenVertices = experimental.flattenVertices,
    fillArray = experimental.fillArray;

// Maybe deck.gl or luma.gl needs to export this

function getPickingColor(index) {
  return [(index + 1) % 256, Math.floor((index + 1) / 256) % 256, Math.floor((index + 1) / 256 / 256) % 256];
}

var DEFAULT_COLOR = [0, 0, 0, 255]; // Black

// This class is set up to allow querying one attribute at a time
// the way the AttributeManager expects it
export var PolygonTesselator = function () {
  function PolygonTesselator(_ref) {
    var polygons = _ref.polygons,
        _ref$fp = _ref.fp64,
        fp64 = _ref$fp === undefined ? false : _ref$fp;

    _classCallCheck(this, PolygonTesselator);

    // Normalize all polygons
    this.polygons = polygons.map(function (polygon) {
      return Polygon.normalize(polygon);
    });
    // Count all polygon vertices
    this.pointCount = getPointCount(this.polygons);
    this.fp64 = fp64;
  }

  _createClass(PolygonTesselator, [{
    key: 'indices',
    value: function indices() {
      var polygons = this.polygons,
          indexCount = this.indexCount;

      return calculateIndices({ polygons: polygons, indexCount: indexCount });
    }
  }, {
    key: 'positions',
    value: function positions() {
      var polygons = this.polygons,
          pointCount = this.pointCount;

      return calculatePositions({ polygons: polygons, pointCount: pointCount, fp64: this.fp64 });
    }
  }, {
    key: 'normals',
    value: function normals() {
      var polygons = this.polygons,
          pointCount = this.pointCount;

      return calculateNormals({ polygons: polygons, pointCount: pointCount });
    }
  }, {
    key: 'colors',
    value: function colors() {
      var _ref2 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref2$getColor = _ref2.getColor,
          getColor = _ref2$getColor === undefined ? function (x) {
        return DEFAULT_COLOR;
      } : _ref2$getColor;

      var polygons = this.polygons,
          pointCount = this.pointCount;

      return calculateColors({ polygons: polygons, pointCount: pointCount, getColor: getColor });
    }
  }, {
    key: 'pickingColors',
    value: function pickingColors() {
      var polygons = this.polygons,
          pointCount = this.pointCount;

      return calculatePickingColors({ polygons: polygons, pointCount: pointCount });
    }

    // getAttribute({size, accessor}) {
    //   const {polygons, pointCount} = this;
    //   return calculateAttribute({polygons, pointCount, size, accessor});
    // }

  }]);

  return PolygonTesselator;
}();

// Count number of points in a list of complex polygons
function getPointCount(polygons) {
  return polygons.reduce(function (points, polygon) {
    return points + Polygon.getVertexCount(polygon);
  }, 0);
}

// COunt number of triangles in a list of complex polygons
function getTriangleCount(polygons) {
  return polygons.reduce(function (triangles, polygon) {
    return triangles + Polygon.getTriangleCount(polygon);
  }, 0);
}

// Returns the offsets of each complex polygon in the combined array of all polygons
function getPolygonOffsets(polygons) {
  var offsets = new Array(polygons.length + 1);
  offsets[0] = 0;
  var offset = 0;
  polygons.forEach(function (polygon, i) {
    offset += Polygon.getVertexCount(polygon);
    offsets[i + 1] = offset;
  });
  return offsets;
}

// Returns the offset of each hole polygon in the flattened array for that polygon
function getHoleIndices(complexPolygon) {
  var holeIndices = null;
  if (complexPolygon.length > 1) {
    var polygonStartIndex = 0;
    holeIndices = [];
    complexPolygon.forEach(function (polygon) {
      polygonStartIndex += polygon.length;
      holeIndices.push(polygonStartIndex);
    });
    // Last element points to end of the flat array, remove it
    holeIndices.pop();
  }
  return holeIndices;
}

function calculateIndices(_ref3) {
  var polygons = _ref3.polygons,
      _ref3$IndexType = _ref3.IndexType,
      IndexType = _ref3$IndexType === undefined ? Uint32Array : _ref3$IndexType;

  // Calculate length of index array (3 * number of triangles)
  var indexCount = 3 * getTriangleCount(polygons);
  var offsets = getPolygonOffsets(polygons);

  // Allocate the attribute
  // TODO it's not the index count but the vertex count that must be checked
  if (IndexType === Uint16Array && indexCount > 65535) {
    throw new Error("Vertex count exceeds browser's limit");
  }
  var attribute = new IndexType(indexCount);

  // 1. get triangulated indices for the internal areas
  // 2. offset them by the number of indices in previous polygons
  var i = 0;
  polygons.forEach(function (polygon, polygonIndex) {
    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
      for (var _iterator = calculateSurfaceIndices(polygon)[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
        var index = _step.value;

        attribute[i++] = index + offsets[polygonIndex];
      }
    } catch (err) {
      _didIteratorError = true;
      _iteratorError = err;
    } finally {
      try {
        if (!_iteratorNormalCompletion && _iterator.return) {
          _iterator.return();
        }
      } finally {
        if (_didIteratorError) {
          throw _iteratorError;
        }
      }
    }
  });

  return attribute;
}

/*
 * Get vertex indices for drawing complexPolygon mesh
 * @private
 * @param {[Number,Number,Number][][]} complexPolygon
 * @returns {[Number]} indices
 */
function calculateSurfaceIndices(complexPolygon) {
  // Prepare an array of hole indices as expected by earcut
  var holeIndices = getHoleIndices(complexPolygon);
  // Flatten the polygon as expected by earcut
  var verts = flattenVertices(complexPolygon);
  // Let earcut triangulate the polygon
  return earcut(verts, holeIndices, 3);
}

function calculatePositions(_ref4) {
  var polygons = _ref4.polygons,
      pointCount = _ref4.pointCount,
      fp64 = _ref4.fp64;

  // Flatten out all the vertices of all the sub subPolygons
  var attribute = new Float32Array(pointCount * 3);
  var attributeLow = void 0;
  if (fp64) {
    // We only need x, y component
    attributeLow = new Float32Array(pointCount * 2);
  }
  var i = 0;
  var j = 0;
  var _iteratorNormalCompletion2 = true;
  var _didIteratorError2 = false;
  var _iteratorError2 = undefined;

  try {
    for (var _iterator2 = polygons[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
      var polygon = _step2.value;

      // eslint-disable-next-line
      Polygon.forEachVertex(polygon, function (vertex) {
        var x = vertex[0];
        var y = vertex[1];
        var z = vertex[2] || 0;
        attribute[i++] = x;
        attribute[i++] = y;
        attribute[i++] = z;
        if (fp64) {
          attributeLow[j++] = fp64LowPart(x);
          attributeLow[j++] = fp64LowPart(y);
        }
      });
    }
  } catch (err) {
    _didIteratorError2 = true;
    _iteratorError2 = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion2 && _iterator2.return) {
        _iterator2.return();
      }
    } finally {
      if (_didIteratorError2) {
        throw _iteratorError2;
      }
    }
  }

  return { positions: attribute, positions64xyLow: attributeLow };
}

function calculateNormals(_ref5) {
  var polygons = _ref5.polygons,
      pointCount = _ref5.pointCount;

  // TODO - use generic vertex attribute?
  var attribute = new Float32Array(pointCount * 3);
  // normals is not used in flat polygon shader
  // fillArray({target: attribute, source: [0, 0, 1], start: 0, count: pointCount});
  return attribute;
}

function calculateColors(_ref6) {
  var polygons = _ref6.polygons,
      pointCount = _ref6.pointCount,
      getColor = _ref6.getColor;

  var attribute = new Uint8ClampedArray(pointCount * 4);
  var i = 0;
  polygons.forEach(function (complexPolygon, polygonIndex) {
    // Calculate polygon color
    var color = getColor(polygonIndex);
    color[3] = Number.isFinite(color[3]) ? color[3] : 255;

    var vertexCount = Polygon.getVertexCount(complexPolygon);
    fillArray({ target: attribute, source: color, start: i, count: vertexCount });
    i += color.length * vertexCount;
  });
  return attribute;
}

function calculatePickingColors(_ref7) {
  var polygons = _ref7.polygons,
      pointCount = _ref7.pointCount;

  var attribute = new Uint8ClampedArray(pointCount * 3);
  var i = 0;
  polygons.forEach(function (complexPolygon, polygonIndex) {
    var color = getPickingColor(polygonIndex);
    var vertexCount = Polygon.getVertexCount(complexPolygon);
    fillArray({ target: attribute, source: color, start: i, count: vertexCount });
    i += color.length * vertexCount;
  });
  return attribute;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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