var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

/* eslint-disable guard-for-in */
import Stats from './stats';
import log from '../utils/log';
import { GL } from 'luma.gl';
import assert from 'assert';

import AttributeTransitionManager from './attribute-transition-manager';

var LOG_START_END_PRIORITY = 1;
var LOG_DETAIL_PRIORITY = 2;

function noop() {}

/* eslint-disable complexity */
export function glArrayFromType(glType) {
  var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref$clamped = _ref.clamped,
      clamped = _ref$clamped === undefined ? true : _ref$clamped;

  // Sorted in some order of likelihood to reduce amount of comparisons
  switch (glType) {
    case GL.FLOAT:
      return Float32Array;
    case GL.UNSIGNED_SHORT:
    case GL.UNSIGNED_SHORT_5_6_5:
    case GL.UNSIGNED_SHORT_4_4_4_4:
    case GL.UNSIGNED_SHORT_5_5_5_1:
      return Uint16Array;
    case GL.UNSIGNED_INT:
      return Uint32Array;
    case GL.UNSIGNED_BYTE:
      return clamped ? Uint8ClampedArray : Uint8Array;
    case GL.BYTE:
      return Int8Array;
    case GL.SHORT:
      return Int16Array;
    case GL.INT:
      return Int32Array;
    default:
      throw new Error('Failed to deduce type from array');
  }
}
/* eslint-enable complexity */

// Default loggers
var logFunctions = {
  savedMessages: null,
  timeStart: null,

  onLog: function onLog(_ref2) {
    var level = _ref2.level,
        message = _ref2.message;

    log.log(level, message);
  },
  onUpdateStart: function onUpdateStart(_ref3) {
    var level = _ref3.level,
        id = _ref3.id,
        numInstances = _ref3.numInstances;

    logFunctions.savedMessages = [];
    logFunctions.timeStart = new Date();
  },
  onUpdate: function onUpdate(_ref4) {
    var level = _ref4.level,
        message = _ref4.message;

    if (logFunctions.savedMessages) {
      logFunctions.savedMessages.push(message);
    }
  },
  onUpdateEnd: function onUpdateEnd(_ref5) {
    var level = _ref5.level,
        id = _ref5.id,
        numInstances = _ref5.numInstances;

    var timeMs = Math.round(new Date() - logFunctions.timeStart);
    var time = timeMs + 'ms';
    log.group(level, 'Updated attributes for ' + numInstances + ' instances in ' + id + ' in ' + time, {
      collapsed: true
    });
    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
      for (var _iterator = logFunctions.savedMessages[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
        var message = _step.value;

        log.log(level, message);
      }
    } catch (err) {
      _didIteratorError = true;
      _iteratorError = err;
    } finally {
      try {
        if (!_iteratorNormalCompletion && _iterator.return) {
          _iterator.return();
        }
      } finally {
        if (_didIteratorError) {
          throw _iteratorError;
        }
      }
    }

    log.groupEnd(level, 'Updated attributes for ' + numInstances + ' instances in ' + id + ' in ' + time);
    logFunctions.savedMessages = null;
  }
};

var AttributeManager = function () {
  _createClass(AttributeManager, null, [{
    key: 'setDefaultLogFunctions',

    /**
     * Sets log functions to help trace or time attribute updates.
     * Default logging uses deck logger.
     *
     * `onLog` is called for each attribute.
     *
     * To enable detailed control of timming and e.g. hierarchical logging,
     * hooks are also provided for update start and end.
     *
     * @param {Object} [opts]
     * @param {String} [opts.onLog=] - called to print
     * @param {String} [opts.onUpdateStart=] - called before update() starts
     * @param {String} [opts.onUpdateEnd=] - called after update() ends
     */
    value: function setDefaultLogFunctions() {
      var _ref6 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          onLog = _ref6.onLog,
          onUpdateStart = _ref6.onUpdateStart,
          onUpdate = _ref6.onUpdate,
          onUpdateEnd = _ref6.onUpdateEnd;

      if (onLog !== undefined) {
        logFunctions.onLog = onLog || noop;
      }
      if (onUpdateStart !== undefined) {
        logFunctions.onUpdateStart = onUpdateStart || noop;
      }
      if (onUpdate !== undefined) {
        logFunctions.onUpdate = onUpdate || noop;
      }
      if (onUpdateEnd !== undefined) {
        logFunctions.onUpdateEnd = onUpdateEnd || noop;
      }
    }

    /**
     * @classdesc
     * Automated attribute generation and management. Suitable when a set of
     * vertex shader attributes are generated by iteration over a data array,
     * and updates to these attributes are needed either when the data itself
     * changes, or when other data relevant to the calculations change.
     *
     * - First the application registers descriptions of its dynamic vertex
     *   attributes using AttributeManager.add().
     * - Then, when any change that affects attributes is detected by the
     *   application, the app will call AttributeManager.invalidate().
     * - Finally before it renders, it calls AttributeManager.update() to
     *   ensure that attributes are automatically rebuilt if anything has been
     *   invalidated.
     *
     * The application provided update functions describe how attributes
     * should be updated from a data array and are expected to traverse
     * that data array (or iterable) and fill in the attribute's typed array.
     *
     * Note that the attribute manager intentionally does not do advanced
     * change detection, but instead makes it easy to build such detection
     * by offering the ability to "invalidate" each attribute separately.
     *
     * Summary:
     * - keeps track of valid state for each attribute
     * - auto reallocates attributes when needed
     * - auto updates attributes with registered updater functions
     * - allows overriding with application supplied buffers
     *
     * Limitations:
     * - There are currently no provisions for only invalidating a range of
     *   indices in an attribute.
     *
     * @class
     * @param {Object} [props]
     * @param {String} [props.id] - identifier (for debugging)
     */

  }]);

  function AttributeManager(gl) {
    var _ref7 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        _ref7$id = _ref7.id,
        id = _ref7$id === undefined ? 'attribute-manager' : _ref7$id;

    _classCallCheck(this, AttributeManager);

    this.id = id;
    this.gl = gl;

    this.attributes = {};
    this.updateTriggers = {};
    this.accessors = {};
    this.allocedInstances = -1;
    this.needsRedraw = true;

    this.userData = {};
    this.stats = new Stats({ id: 'attr' });

    this.attributeTransitionManger = new AttributeTransitionManager(gl, {
      id: id + '-transitions'
    });

    // For debugging sanity, prevent uninitialized members
    Object.seal(this);
  }

  /**
   * Adds attributes
   * Takes a map of attribute descriptor objects
   * - keys are attribute names
   * - values are objects with attribute fields
   *
   * attribute.size - number of elements per object
   * attribute.updater - number of elements
   * attribute.instanced=0 - is this is an instanced attribute (a.k.a. divisor)
   * attribute.noAlloc=false - if this attribute should not be allocated
   *
   * @example
   * attributeManager.add({
   *   positions: {size: 2, update: calculatePositions}
   *   colors: {size: 3, update: calculateColors}
   * });
   *
   * @param {Object} attributes - attribute map (see above)
   * @param {Object} updaters - separate map of update functions (deprecated)
   */


  _createClass(AttributeManager, [{
    key: 'add',
    value: function add(attributes) {
      var updaters = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      this._add(attributes, updaters);
    }

    /**
     * Removes attributes
     * Takes an array of attribute names and delete them from
     * the attribute map if they exists
     *
     * @example
     * attributeManager.remove(['position']);
     *
     * @param {Object} attributeNameArray - attribute name array (see above)
     */

  }, {
    key: 'remove',
    value: function remove(attributeNameArray) {
      for (var i = 0; i < attributeNameArray.length; i++) {
        var name = attributeNameArray[i];
        if (this.attributes[name] !== undefined) {
          delete this.attributes[name];
        }
      }
    }

    /* Marks an attribute for update
     * @param {string} triggerName: attribute or accessor name
     */

  }, {
    key: 'invalidate',
    value: function invalidate(triggerName) {
      var invalidatedAttributes = this._invalidateTrigger(triggerName);

      // For performance tuning
      logFunctions.onLog({
        level: LOG_DETAIL_PRIORITY,
        message: 'invalidated attributes ' + invalidatedAttributes + ' (' + triggerName + ') for ' + this.id,
        id: this.identifier
      });
    }
  }, {
    key: 'invalidateAll',
    value: function invalidateAll() {
      for (var attributeName in this.attributes) {
        this.attributes[attributeName].needsUpdate = true;
      }

      // For performance tuning
      logFunctions.onLog({
        level: LOG_DETAIL_PRIORITY,
        message: 'invalidated all attributes for ' + this.id,
        id: this.identifier
      });
    }
  }, {
    key: '_invalidateTrigger',
    value: function _invalidateTrigger(triggerName) {
      var attributes = this.attributes,
          updateTriggers = this.updateTriggers;

      var invalidatedAttributes = updateTriggers[triggerName];

      if (!invalidatedAttributes) {
        var message = 'invalidating non-existent trigger ' + triggerName + ' for ' + this.id + '\n';
        message += 'Valid triggers: ' + Object.keys(attributes).join(', ');
        log.warn(message, invalidatedAttributes);
      } else {
        invalidatedAttributes.forEach(function (name) {
          var attribute = attributes[name];
          if (attribute) {
            attribute.needsUpdate = true;
          }
        });
      }
      return invalidatedAttributes;
    }

    /**
     * Ensure all attribute buffers are updated from props or data.
     *
     * Note: Any preallocated buffers in "buffers" matching registered attribute
     * names will be used. No update will happen in this case.
     * Note: Calls onUpdateStart and onUpdateEnd log callbacks before and after.
     *
     * @param {Object} opts - options
     * @param {Object} opts.data - data (iterable object)
     * @param {Object} opts.numInstances - count of data
     * @param {Object} opts.buffers = {} - pre-allocated buffers
     * @param {Object} opts.props - passed to updaters
     * @param {Object} opts.context - Used as "this" context for updaters
     */

  }, {
    key: 'update',
    value: function update() {
      var _ref8 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          data = _ref8.data,
          numInstances = _ref8.numInstances,
          transitions = _ref8.transitions,
          _ref8$props = _ref8.props,
          props = _ref8$props === undefined ? {} : _ref8$props,
          _ref8$buffers = _ref8.buffers,
          buffers = _ref8$buffers === undefined ? {} : _ref8$buffers,
          _ref8$context = _ref8.context,
          context = _ref8$context === undefined ? {} : _ref8$context,
          _ref8$ignoreUnknownAt = _ref8.ignoreUnknownAttributes,
          ignoreUnknownAttributes = _ref8$ignoreUnknownAt === undefined ? false : _ref8$ignoreUnknownAt;

      // First apply any application provided buffers
      this._checkExternalBuffers({ buffers: buffers, ignoreUnknownAttributes: ignoreUnknownAttributes });
      this._setExternalBuffers(buffers);

      // Only initiate alloc/update (and logging) if actually needed
      if (this._analyzeBuffers({ numInstances: numInstances })) {
        logFunctions.onUpdateStart({ level: LOG_START_END_PRIORITY, id: this.id, numInstances: numInstances });
        this.stats.timeStart();
        this._updateBuffers({ numInstances: numInstances, data: data, props: props, context: context });
        this.stats.timeEnd();
        logFunctions.onUpdateEnd({ level: LOG_START_END_PRIORITY, id: this.id, numInstances: numInstances });
      }

      this.attributeTransitionManger.update(this.attributes, transitions);
    }

    /**
     * Returns all attribute descriptors
     * Note: Format matches luma.gl Model/Program.setAttributes()
     * @return {Object} attributes - descriptors
     */

  }, {
    key: 'getAttributes',
    value: function getAttributes() {
      return this.attributes;
    }

    /**
     * Returns changed attribute descriptors
     * This indicates which WebGLBuggers need to be updated
     * @return {Object} attributes - descriptors
     */

  }, {
    key: 'getChangedAttributes',
    value: function getChangedAttributes(_ref9) {
      var _ref9$transition = _ref9.transition,
          transition = _ref9$transition === undefined ? false : _ref9$transition,
          _ref9$clearChangedFla = _ref9.clearChangedFlags,
          clearChangedFlags = _ref9$clearChangedFla === undefined ? false : _ref9$clearChangedFla;
      var attributes = this.attributes,
          attributeTransitionManger = this.attributeTransitionManger;


      if (transition) {
        return attributeTransitionManger.getAttributes();
      }

      var changedAttributes = {};
      for (var attributeName in attributes) {
        var attribute = attributes[attributeName];
        if (attribute.changed) {
          attribute.changed = attribute.changed && !clearChangedFlags;

          // Only return non-transition attributes
          if (!attributeTransitionManger.hasAttribute(attributeName)) {
            changedAttributes[attributeName] = attribute;
          }
        }
      }
      return changedAttributes;
    }

    /**
     * Returns the redraw flag, optionally clearing it.
     * Redraw flag will be set if any attributes attributes changed since
     * flag was last cleared.
     *
     * @param {Object} [opts]
     * @param {String} [opts.clearRedrawFlags=false] - whether to clear the flag
     * @return {false|String} - reason a redraw is needed.
     */

  }, {
    key: 'getNeedsRedraw',
    value: function getNeedsRedraw() {
      var _ref10 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref10$clearRedrawFla = _ref10.clearRedrawFlags,
          clearRedrawFlags = _ref10$clearRedrawFla === undefined ? false : _ref10$clearRedrawFla;

      var redraw = this.needsRedraw;
      this.needsRedraw = this.needsRedraw && !clearRedrawFlags;
      return redraw && this.id;
    }

    /**
     * Sets the redraw flag.
     * @param {Boolean} redraw=true
     * @return {AttributeManager} - for chaining
     */

  }, {
    key: 'setNeedsRedraw',
    value: function setNeedsRedraw() {
      var redraw = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : true;

      this.needsRedraw = true;
      return this;
    }

    // DEPRECATED METHODS

    /**
     * Adds attributes
     * @param {Object} attributes - attribute map (see above)
     * @param {Object} updaters - separate map of update functions (deprecated)
     */

  }, {
    key: 'addInstanced',
    value: function addInstanced(attributes) {
      var updaters = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      this._add(attributes, updaters, { instanced: 1 });
    }

    // PROTECTED METHODS - Only to be used by collaborating classes, not by apps

    /**
     * Returns object containing all accessors as keys, with non-null values
     * @return {Object} - accessors object
     */

  }, {
    key: 'getAccessors',
    value: function getAccessors() {
      return this.updateTriggers;
    }

    // PRIVATE METHODS

    // Used to register an attribute

  }, {
    key: '_add',
    value: function _add(attributes) {
      var updaters = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      var _extraProps = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

      var newAttributes = {};

      for (var attributeName in attributes) {
        // support for separate update function map
        // For now, just copy any attributes from that map into the main map
        // TODO - Attribute maps are a deprecated feature, remove
        if (attributeName in updaters) {
          attributes[attributeName] = Object.assign({}, attributes[attributeName], updaters[attributeName]);
        }

        var attribute = attributes[attributeName];

        var isGeneric = attribute.isGeneric || false;
        var isIndexed = attribute.isIndexed || attribute.elements;
        var size = attribute.elements && 1 || attribute.size;
        var value = attribute.value || null;

        // Initialize the attribute descriptor, with WebGL and metadata fields
        var attributeData = Object.assign({
          // Ensure that fields are present before Object.seal()
          target: undefined,
          userData: {} // Reserved for application
        },
        // Metadata
        attribute, {
          // State
          isExternalBuffer: false,
          needsAlloc: false,
          needsUpdate: false,
          changed: false,

          // Luma fields
          isGeneric: isGeneric,
          isIndexed: isIndexed,
          size: size,
          value: value
        }, _extraProps);
        // Sanity - no app fields on our attributes. Use userData instead.
        Object.seal(attributeData);

        // Check all fields and generate helpful error messages
        this._validateAttributeDefinition(attributeName, attributeData);

        // Add to both attributes list (for registration with model)
        newAttributes[attributeName] = attributeData;
      }

      Object.assign(this.attributes, newAttributes);

      this._mapUpdateTriggersToAttributes();
    }

    // build updateTrigger name to attribute name mapping

  }, {
    key: '_mapUpdateTriggersToAttributes',
    value: function _mapUpdateTriggersToAttributes() {
      var _this = this;

      var triggers = {};

      var _loop = function _loop(attributeName) {
        var attribute = _this.attributes[attributeName];
        var accessor = attribute.accessor;

        // Backards compatibility: allow attribute name to be used as update trigger key

        triggers[attributeName] = [attributeName];

        // use accessor name as update trigger key
        if (typeof accessor === 'string') {
          accessor = [accessor];
        }
        if (Array.isArray(accessor)) {
          accessor.forEach(function (accessorName) {
            if (!triggers[accessorName]) {
              triggers[accessorName] = [];
            }
            triggers[accessorName].push(attributeName);
          });
        }
      };

      for (var attributeName in this.attributes) {
        _loop(attributeName);
      }

      this.updateTriggers = triggers;
    }
  }, {
    key: '_validateAttributeDefinition',
    value: function _validateAttributeDefinition(attributeName, attribute) {
      assert(attribute.size >= 1 && attribute.size <= 4, 'Attribute definition for ' + attributeName + ' invalid size');

      // Check that either 'accessor' or 'update' is a valid function
      var hasUpdater = attribute.noAlloc || typeof attribute.update === 'function' || typeof attribute.accessor === 'string';
      if (!hasUpdater) {
        throw new Error('Attribute ' + attributeName + ' missing update or accessor');
      }
    }

    // Checks that any attribute buffers in props are valid
    // Note: This is just to help app catch mistakes

  }, {
    key: '_checkExternalBuffers',
    value: function _checkExternalBuffers() {
      var _ref11 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref11$buffers = _ref11.buffers,
          buffers = _ref11$buffers === undefined ? {} : _ref11$buffers,
          _ref11$ignoreUnknownA = _ref11.ignoreUnknownAttributes,
          ignoreUnknownAttributes = _ref11$ignoreUnknownA === undefined ? false : _ref11$ignoreUnknownA;

      var attributes = this.attributes;

      for (var attributeName in buffers) {
        var attribute = attributes[attributeName];
        if (!attribute && !ignoreUnknownAttributes) {
          throw new Error('Unknown attribute prop ' + attributeName);
        }
        // const buffer = buffers[attributeName];
        // TODO - check buffer type
      }
    }

    // Set the buffers for the supplied attributes
    // Update attribute buffers from any attributes in props
    // Detach any previously set buffers, marking all
    // Attributes for auto allocation
    /* eslint-disable max-statements */

  }, {
    key: '_setExternalBuffers',
    value: function _setExternalBuffers(bufferMap) {
      var attributes = this.attributes,
          numInstances = this.numInstances;

      // Copy the refs of any supplied buffers in the props

      for (var attributeName in attributes) {
        var attribute = attributes[attributeName];
        var buffer = bufferMap[attributeName];
        attribute.isExternalBuffer = false;
        if (buffer) {
          var ArrayType = glArrayFromType(attribute.type || GL.FLOAT);
          if (!(buffer instanceof ArrayType)) {
            throw new Error('Attribute ' + attributeName + ' must be of type ' + ArrayType.name);
          }
          if (attribute.auto && buffer.length <= numInstances * attribute.size) {
            throw new Error('Attribute prop array must match length and size');
          }

          attribute.isExternalBuffer = true;
          attribute.needsUpdate = false;
          if (attribute.value !== buffer) {
            attribute.value = buffer;
            attribute.changed = true;
            this.needsRedraw = true;
          }
        }
      }
    }
    /* eslint-enable max-statements */

    /* Checks that typed arrays for attributes are big enough
     * sets alloc flag if not
     * @return {Boolean} whether any updates are needed
     */

  }, {
    key: '_analyzeBuffers',
    value: function _analyzeBuffers(_ref12) {
      var numInstances = _ref12.numInstances;
      var attributes = this.attributes;

      assert(numInstances !== undefined, 'numInstances not defined');

      // Track whether any allocations or updates are needed
      var needsUpdate = false;

      for (var attributeName in attributes) {
        var attribute = attributes[attributeName];
        if (!attribute.isExternalBuffer) {
          // Do we need to reallocate the attribute's typed array?
          var needsAlloc = attribute.value === null || attribute.value.length / attribute.size < numInstances;
          if (needsAlloc && (attribute.update || attribute.accessor)) {
            attribute.needsAlloc = true;
            needsUpdate = true;
          }
          if (attribute.needsUpdate) {
            needsUpdate = true;
          }
        }
      }

      return needsUpdate;
    }

    /**
     * @private
     * Calls update on any buffers that need update
     * TODO? - If app supplied all attributes, no need to iterate over data
     *
     * @param {Object} opts - options
     * @param {Object} opts.data - data (iterable object)
     * @param {Object} opts.numInstances - count of data
     * @param {Object} opts.buffers = {} - pre-allocated buffers
     * @param {Object} opts.props - passed to updaters
     * @param {Object} opts.context - Used as "this" context for updaters
     */
    /* eslint-disable max-statements, complexity */

  }, {
    key: '_updateBuffers',
    value: function _updateBuffers(_ref13) {
      var numInstances = _ref13.numInstances,
          data = _ref13.data,
          props = _ref13.props,
          context = _ref13.context;
      var attributes = this.attributes;

      // Allocate at least one element to ensure a valid buffer

      var allocCount = Math.max(numInstances, 1);

      for (var attributeName in attributes) {
        var attribute = attributes[attributeName];

        // Allocate a new typed array if needed
        if (attribute.needsAlloc) {
          var ArrayType = glArrayFromType(attribute.type || GL.FLOAT);
          attribute.value = new ArrayType(attribute.size * allocCount);
          logFunctions.onUpdate({
            level: LOG_DETAIL_PRIORITY,
            message: attributeName + ' allocated ' + allocCount,
            id: this.id
          });
          attribute.needsAlloc = false;
          attribute.needsUpdate = true;
        }
      }

      for (var _attributeName in attributes) {
        var _attribute = attributes[_attributeName];
        // Call updater function if needed
        if (_attribute.needsUpdate) {
          this._updateBuffer({ attribute: _attribute, attributeName: _attributeName, numInstances: numInstances, data: data, props: props, context: context });
        }
      }

      this.allocedInstances = allocCount;
    }
  }, {
    key: '_updateBuffer',
    value: function _updateBuffer(_ref14) {
      var attribute = _ref14.attribute,
          attributeName = _ref14.attributeName,
          numInstances = _ref14.numInstances,
          data = _ref14.data,
          props = _ref14.props,
          context = _ref14.context;
      var update = attribute.update,
          accessor = attribute.accessor;


      var timeStart = new Date();
      if (update) {
        // Custom updater - typically for non-instanced layers
        update.call(context, attribute, { data: data, props: props, numInstances: numInstances });
        this._checkAttributeArray(attribute, attributeName);
      } else if (accessor) {
        // Standard updater
        this._updateBufferViaStandardAccessor({ attribute: attribute, data: data, props: props });
        this._checkAttributeArray(attribute, attributeName);
      } else {
        logFunctions.onUpdate({
          level: LOG_DETAIL_PRIORITY,
          message: attributeName + ' missing update function',
          id: this.id
        });
      }
      var timeMs = Math.round(new Date() - timeStart);
      var time = timeMs + 'ms';
      logFunctions.onUpdate({
        level: LOG_DETAIL_PRIORITY,
        message: attributeName + ' updated ' + numInstances + ' ' + time,
        id: this.id
      });

      attribute.needsUpdate = false;
      attribute.changed = true;
      this.needsRedraw = true;
    }
    /* eslint-enable max-statements */

  }, {
    key: '_updateBufferViaStandardAccessor',
    value: function _updateBufferViaStandardAccessor(_ref15) {
      var attribute = _ref15.attribute,
          data = _ref15.data,
          props = _ref15.props;
      var accessor = attribute.accessor,
          value = attribute.value,
          size = attribute.size;

      var accessorFunc = props[accessor];

      assert(typeof accessorFunc === 'function', 'accessor "' + accessor + '" is not a function');

      var _attribute$defaultVal = attribute.defaultValue,
          defaultValue = _attribute$defaultVal === undefined ? [0, 0, 0, 0] : _attribute$defaultVal;

      defaultValue = Array.isArray(defaultValue) ? defaultValue : [defaultValue];
      var i = 0;
      var _iteratorNormalCompletion2 = true;
      var _didIteratorError2 = false;
      var _iteratorError2 = undefined;

      try {
        for (var _iterator2 = data[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
          var object = _step2.value;

          var objectValue = accessorFunc(object);
          objectValue = Array.isArray(objectValue) ? objectValue : [objectValue];
          /* eslint-disable no-fallthrough, default-case */
          switch (size) {
            case 4:
              value[i + 3] = Number.isFinite(objectValue[3]) ? objectValue[3] : defaultValue[3];
            case 3:
              value[i + 2] = Number.isFinite(objectValue[2]) ? objectValue[2] : defaultValue[2];
            case 2:
              value[i + 1] = Number.isFinite(objectValue[1]) ? objectValue[1] : defaultValue[1];
            case 1:
              value[i + 0] = Number.isFinite(objectValue[0]) ? objectValue[0] : defaultValue[0];
          }
          i += size;
        }
      } catch (err) {
        _didIteratorError2 = true;
        _iteratorError2 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion2 && _iterator2.return) {
            _iterator2.return();
          }
        } finally {
          if (_didIteratorError2) {
            throw _iteratorError2;
          }
        }
      }
    }
  }, {
    key: '_checkAttributeArray',
    value: function _checkAttributeArray(attribute, attributeName) {
      var value = attribute.value;

      if (value && value.length >= 4) {
        var valid = Number.isFinite(value[0]) && Number.isFinite(value[1]) && Number.isFinite(value[2]) && Number.isFinite(value[3]);
        if (!valid) {
          throw new Error('Illegal attribute generated for ' + attributeName);
        }
      }
    }

    /**
     * Update attribute transition to the current timestamp
     * Returns `true` if any transition is in progress
     */

  }, {
    key: 'updateTransition',
    value: function updateTransition() {
      var attributeTransitionManger = this.attributeTransitionManger;

      var transitionUpdated = attributeTransitionManger.setCurrentTime(Date.now());
      this.needsRedraw = this.needsRedraw || transitionUpdated;
      return transitionUpdated;
    }
  }]);

  return AttributeManager;
}();

export default AttributeManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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