// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

/* global window */
import { GL, withParameters, setParameters } from 'luma.gl';
import log from '../utils/log';
import assert from 'assert';

var LOG_PRIORITY_DRAW = 2;

var renderCount = 0;

// TODO - Exported for pick-layers.js - Move to util?
export var getPixelRatio = function getPixelRatio(_ref) {
  var useDevicePixels = _ref.useDevicePixels;

  assert(typeof useDevicePixels === 'boolean', 'Invalid useDevicePixels');
  return useDevicePixels && typeof window !== 'undefined' ? window.devicePixelRatio : 1;
};

// Convert viewport top-left CSS coordinates to bottom up WebGL coordinates
var getGLViewport = function getGLViewport(gl, _ref2) {
  var viewport = _ref2.viewport,
      pixelRatio = _ref2.pixelRatio;

  // TODO - dummy default for node
  var height = gl.canvas ? gl.canvas.clientHeight : 100;
  // Convert viewport top-left CSS coordinates to bottom up WebGL coordinates
  var dimensions = viewport;
  return [dimensions.x * pixelRatio, (height - dimensions.y - dimensions.height) * pixelRatio, dimensions.width * pixelRatio, dimensions.height * pixelRatio];
};

// Helper functions

function clearCanvas(gl, _ref3) {
  var useDevicePixels = _ref3.useDevicePixels;

  // const pixelRatio = getPixelRatio({useDevicePixels});
  var width = gl.drawingBufferWidth;
  var height = gl.drawingBufferHeight;
  // clear depth and color buffers, restoring transparency
  withParameters(gl, { viewport: [0, 0, width, height] }, function () {
    gl.clear(GL.COLOR_BUFFER_BIT | GL.DEPTH_BUFFER_BIT);
  });
}

// Draw a list of layers in a list of viewports
export function drawLayers(gl, _ref4) {
  var layers = _ref4.layers,
      viewports = _ref4.viewports,
      onViewportActive = _ref4.onViewportActive,
      useDevicePixels = _ref4.useDevicePixels,
      _ref4$drawPickingColo = _ref4.drawPickingColors,
      drawPickingColors = _ref4$drawPickingColo === undefined ? false : _ref4$drawPickingColo,
      _ref4$deviceRect = _ref4.deviceRect,
      deviceRect = _ref4$deviceRect === undefined ? null : _ref4$deviceRect,
      _ref4$parameters = _ref4.parameters,
      parameters = _ref4$parameters === undefined ? {} : _ref4$parameters,
      _ref4$layerFilter = _ref4.layerFilter,
      layerFilter = _ref4$layerFilter === undefined ? null : _ref4$layerFilter,
      _ref4$pass = _ref4.pass,
      pass = _ref4$pass === undefined ? 'draw' : _ref4$pass,
      _ref4$redrawReason = _ref4.redrawReason,
      redrawReason = _ref4$redrawReason === undefined ? '' : _ref4$redrawReason;

  clearCanvas(gl, { useDevicePixels: useDevicePixels });

  // effectManager.preDraw();

  viewports.forEach(function (viewportOrDescriptor, i) {
    var viewport = getViewportFromDescriptor(viewportOrDescriptor);

    // Update context to point to this viewport
    onViewportActive(viewport);

    // render this viewport
    drawLayersInViewport(gl, {
      layers: layers,
      viewport: viewport,
      useDevicePixels: useDevicePixels,
      drawPickingColors: drawPickingColors,
      deviceRect: deviceRect,
      parameters: parameters,
      layerFilter: layerFilter,
      pass: pass,
      redrawReason: redrawReason
    });
  });

  // effectManager.draw();
}

// Draws list of layers and viewports into the picking buffer
// Note: does not sample the buffer, that has to be done by the caller
export function drawPickingBuffer(gl, _ref5) {
  var layers = _ref5.layers,
      viewports = _ref5.viewports,
      onViewportActive = _ref5.onViewportActive,
      useDevicePixels = _ref5.useDevicePixels,
      pickingFBO = _ref5.pickingFBO,
      _ref5$deviceRect = _ref5.deviceRect,
      x = _ref5$deviceRect.x,
      y = _ref5$deviceRect.y,
      width = _ref5$deviceRect.width,
      height = _ref5$deviceRect.height,
      _ref5$layerFilter = _ref5.layerFilter,
      layerFilter = _ref5$layerFilter === undefined ? null : _ref5$layerFilter,
      _ref5$redrawReason = _ref5.redrawReason,
      redrawReason = _ref5$redrawReason === undefined ? '' : _ref5$redrawReason;

  // Make sure we clear scissor test and fbo bindings in case of exceptions
  // We are only interested in one pixel, no need to render anything else
  // Note that the callback here is called synchronously.
  // Set blend mode for picking
  // always overwrite existing pixel with [r,g,b,layerIndex]
  return withParameters(gl, {
    framebuffer: pickingFBO,
    scissorTest: true,
    scissor: [x, y, width, height],
    clearColor: [0, 0, 0, 0]
  }, function () {
    drawLayers(gl, {
      layers: layers,
      viewports: viewports,
      onViewportActive: onViewportActive,
      useDevicePixels: useDevicePixels,
      drawPickingColors: true,
      layerFilter: layerFilter,
      pass: 'picking',
      redrawReason: redrawReason,
      parameters: {
        blend: true,
        blendFunc: [gl.ONE, gl.ZERO, gl.CONSTANT_ALPHA, gl.ZERO],
        blendEquation: gl.FUNC_ADD,
        blendColor: [0, 0, 0, 0]
      }
    });
  });
}

// Draws a list of layers in one viewport
// TODO - when picking we could completely skip rendering viewports that dont
// intersect with the picking rect
function drawLayersInViewport(gl, _ref6) {
  var layers = _ref6.layers,
      viewport = _ref6.viewport,
      useDevicePixels = _ref6.useDevicePixels,
      _ref6$drawPickingColo = _ref6.drawPickingColors,
      drawPickingColors = _ref6$drawPickingColo === undefined ? false : _ref6$drawPickingColo,
      _ref6$deviceRect = _ref6.deviceRect,
      deviceRect = _ref6$deviceRect === undefined ? null : _ref6$deviceRect,
      _ref6$parameters = _ref6.parameters,
      parameters = _ref6$parameters === undefined ? {} : _ref6$parameters,
      layerFilter = _ref6.layerFilter,
      _ref6$pass = _ref6.pass,
      pass = _ref6$pass === undefined ? 'draw' : _ref6$pass,
      _ref6$redrawReason = _ref6.redrawReason,
      redrawReason = _ref6$redrawReason === undefined ? '' : _ref6$redrawReason;

  var pixelRatio = getPixelRatio({ useDevicePixels: useDevicePixels });
  var glViewport = getGLViewport(gl, { viewport: viewport, pixelRatio: pixelRatio });

  // render layers in normal colors
  var renderStats = {
    totalCount: layers.length,
    visibleCount: 0,
    compositeCount: 0,
    pickableCount: 0
  };

  // const {x, y, width, height} = deviceRect || [];

  setParameters(gl, parameters || {});

  // render layers in normal colors
  layers.forEach(function (layer, layerIndex) {
    // Check if we should draw layer
    var shouldDrawLayer = layer.props.visible;
    if (drawPickingColors) {
      shouldDrawLayer = shouldDrawLayer && layer.props.pickable;
    }
    if (shouldDrawLayer && layerFilter) {
      shouldDrawLayer = layerFilter({ layer: layer, viewport: viewport, isPicking: drawPickingColors });
    }

    // Calculate stats
    if (shouldDrawLayer && layer.props.pickable) {
      renderStats.pickableCount++;
    }
    if (layer.isComposite) {
      renderStats.compositeCount++;
    }

    // Draw the layer
    if (shouldDrawLayer) {
      if (!layer.isComposite) {
        renderStats.visibleCount++;
      }

      drawLayerInViewport({ gl: gl, layer: layer, layerIndex: layerIndex, drawPickingColors: drawPickingColors, glViewport: glViewport, parameters: parameters });
    }
  });

  renderCount++;

  logRenderStats({ renderStats: renderStats, pass: pass, redrawReason: redrawReason });
}

function drawLayerInViewport(_ref7) {
  var gl = _ref7.gl,
      layer = _ref7.layer,
      layerIndex = _ref7.layerIndex,
      drawPickingColors = _ref7.drawPickingColors,
      glViewport = _ref7.glViewport,
      parameters = _ref7.parameters;

  var moduleParameters = Object.assign({}, layer.props, {
    viewport: layer.context.viewport,
    pickingActive: drawPickingColors ? 1 : 0
  });

  var uniforms = Object.assign({}, layer.context.uniforms, { layerIndex: layerIndex });

  // All parameter resolving is done here instead of the layer
  // Blend parameters must not be overriden
  var layerParameters = Object.assign({}, layer.props.parameters || {}, parameters);

  Object.assign(layerParameters, {
    viewport: glViewport
  });

  if (drawPickingColors) {
    Object.assign(layerParameters, {
      blendColor: [0, 0, 0, (layerIndex + 1) / 255]
    });
  } else {
    Object.assign(moduleParameters, getObjectHighlightParameters(layer));
  }

  layer.drawLayer({
    moduleParameters: moduleParameters,
    uniforms: uniforms,
    parameters: layerParameters
  });
}

function logRenderStats(_ref8) {
  var renderStats = _ref8.renderStats,
      pass = _ref8.pass,
      redrawReason = _ref8.redrawReason;

  if (log.priority >= LOG_PRIORITY_DRAW) {
    var totalCount = renderStats.totalCount,
        visibleCount = renderStats.visibleCount,
        compositeCount = renderStats.compositeCount,
        pickableCount = renderStats.pickableCount;

    var primitiveCount = totalCount - compositeCount;
    var hiddenCount = primitiveCount - visibleCount;

    var message = '';
    message += 'RENDER #' + renderCount + ' ' + visibleCount + ' (of ' + totalCount + ' layers) to ' + pass + ' because ' + redrawReason + ' ';
    if (log.priority > LOG_PRIORITY_DRAW) {
      message += '(' + hiddenCount + ' hidden, ' + compositeCount + ' composite ' + pickableCount + ' unpickable)';
    }

    log.log(LOG_PRIORITY_DRAW, message);
  }
}

// Get a viewport from a viewport descriptor (which can be a plain viewport)
function getViewportFromDescriptor(viewportOrDescriptor) {
  return viewportOrDescriptor.viewport ? viewportOrDescriptor.viewport : viewportOrDescriptor;
}

/**
 * Returns the picking color of currenlty selected object of the given 'layer'.
 * @return {Array} - the picking color or null if layers selected object is invalid.
 */
function getObjectHighlightParameters(layer) {
  // TODO - inefficient to update settings every render?
  // TODO: Add warning if 'highlightedObjectIndex' is > numberOfInstances of the model.

  // Update picking module settings if highlightedObjectIndex is set.
  // This will overwrite any settings from auto highlighting.
  if (Number.isInteger(layer.props.highlightedObjectIndex)) {
    var pickingSelectedColor = layer.props.highlightedObjectIndex >= 0 ? layer.encodePickingColor(layer.props.highlightedObjectIndex) : null;

    return {
      pickingSelectedColor: pickingSelectedColor
    };
  }
  return null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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