// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import { drawPickingBuffer, getPixelRatio } from './draw-layers';
import log from '../utils/log';
import assert from 'assert';

var NO_PICKED_OBJECT = {
  pickedColor: null,
  pickedLayer: null,
  pickedObjectIndex: -1
};

/* eslint-disable max-depth, max-statements */
// Pick the closest object at the given (x,y) coordinate
export function pickObject(gl, _ref) {
  var layers = _ref.layers,
      viewports = _ref.viewports,
      x = _ref.x,
      y = _ref.y,
      radius = _ref.radius,
      layerFilter = _ref.layerFilter,
      mode = _ref.mode,
      onViewportActive = _ref.onViewportActive,
      pickingFBO = _ref.pickingFBO,
      lastPickedInfo = _ref.lastPickedInfo,
      useDevicePixels = _ref.useDevicePixels;

  // Convert from canvas top-left to WebGL bottom-left coordinates
  // And compensate for pixelRatio
  var pixelRatio = getPixelRatio({ useDevicePixels: useDevicePixels });
  var deviceX = Math.round(x * pixelRatio);
  var deviceY = Math.round(gl.canvas.height - y * pixelRatio);
  var deviceRadius = Math.round(radius * pixelRatio);

  var deviceRect = getPickingRect({
    deviceX: deviceX,
    deviceY: deviceY,
    deviceRadius: deviceRadius,
    deviceWidth: pickingFBO.width,
    deviceHeight: pickingFBO.height
  });

  var pickedColors = deviceRect && drawAndSamplePickingBuffer(gl, {
    layers: layers,
    viewports: viewports,
    onViewportActive: onViewportActive,
    useDevicePixels: useDevicePixels,
    pickingFBO: pickingFBO,
    deviceRect: deviceRect,
    layerFilter: layerFilter,
    redrawReason: mode
  });

  var pickInfo = pickedColors && getClosestFromPickingBuffer(gl, {
    pickedColors: pickedColors,
    layers: layers,
    deviceX: deviceX,
    deviceY: deviceY,
    deviceRadius: deviceRadius,
    deviceRect: deviceRect
  }) || NO_PICKED_OBJECT;

  return processPickInfo({
    pickInfo: pickInfo,
    lastPickedInfo: lastPickedInfo,
    mode: mode,
    layers: layers,
    viewports: viewports,
    x: x,
    y: y,
    deviceX: deviceX,
    deviceY: deviceY,
    pixelRatio: pixelRatio
  });
}

// Pick all objects within the given bounding box
export function pickVisibleObjects(gl, _ref2) {
  var layers = _ref2.layers,
      viewports = _ref2.viewports,
      x = _ref2.x,
      y = _ref2.y,
      width = _ref2.width,
      height = _ref2.height,
      mode = _ref2.mode,
      layerFilter = _ref2.layerFilter,
      onViewportActive = _ref2.onViewportActive,
      pickingFBO = _ref2.pickingFBO,
      useDevicePixels = _ref2.useDevicePixels;

  // Convert from canvas top-left to WebGL bottom-left coordinates
  // And compensate for pixelRatio
  var pixelRatio = getPixelRatio({ useDevicePixels: useDevicePixels });

  var deviceLeft = Math.round(x * pixelRatio);
  var deviceBottom = Math.round(gl.canvas.height - y * pixelRatio);
  var deviceRight = Math.round((x + width) * pixelRatio);
  var deviceTop = Math.round(gl.canvas.height - (y + height) * pixelRatio);

  var deviceRect = {
    x: deviceLeft,
    y: deviceTop,
    width: deviceRight - deviceLeft,
    height: deviceBottom - deviceTop
  };

  var pickedColors = drawAndSamplePickingBuffer(gl, {
    layers: layers,
    viewports: viewports,
    onViewportActive: onViewportActive,
    pickingFBO: pickingFBO,
    useDevicePixels: useDevicePixels,
    deviceRect: deviceRect,
    layerFilter: layerFilter,
    redrawReason: mode
  });

  var pickInfos = getUniquesFromPickingBuffer(gl, { pickedColors: pickedColors, layers: layers });

  // Only return unique infos, identified by info.object
  var uniqueInfos = new Map();

  pickInfos.forEach(function (pickInfo) {
    var info = {
      color: pickInfo.pickedColor,
      layer: null,
      index: pickInfo.pickedObjectIndex,
      picked: true,
      x: x,
      y: y,
      width: width,
      height: height,
      pixelRatio: pixelRatio
    };

    info = getLayerPickingInfo({ layer: pickInfo.pickedLayer, info: info, mode: mode });
    if (!uniqueInfos.has(info.object)) {
      uniqueInfos.set(info.object, info);
    }
  });

  return Array.from(uniqueInfos.values());
}

// HELPER METHODS

// returns pickedColor or null if no pickable layers found.
function drawAndSamplePickingBuffer(gl, _ref3) {
  var layers = _ref3.layers,
      viewports = _ref3.viewports,
      onViewportActive = _ref3.onViewportActive,
      useDevicePixels = _ref3.useDevicePixels,
      pickingFBO = _ref3.pickingFBO,
      deviceRect = _ref3.deviceRect,
      layerFilter = _ref3.layerFilter,
      redrawReason = _ref3.redrawReason;

  assert(deviceRect);
  assert(Number.isFinite(deviceRect.width) && deviceRect.width > 0, '`width` must be > 0');
  assert(Number.isFinite(deviceRect.height) && deviceRect.height > 0, '`height` must be > 0');

  var pickableLayers = layers.filter(function (layer) {
    return layer.isPickable();
  });
  if (pickableLayers.length < 1) {
    return null;
  }

  drawPickingBuffer(gl, {
    layers: layers,
    viewports: viewports,
    onViewportActive: onViewportActive,
    useDevicePixels: useDevicePixels,
    pickingFBO: pickingFBO,
    deviceRect: deviceRect,
    layerFilter: layerFilter,
    redrawReason: redrawReason
  });

  // Read from an already rendered picking buffer
  // Returns an Uint8ClampedArray of picked pixels
  var x = deviceRect.x,
      y = deviceRect.y,
      width = deviceRect.width,
      height = deviceRect.height;

  var pickedColors = new Uint8Array(width * height * 4);
  pickingFBO.readPixels({ x: x, y: y, width: width, height: height, pixelArray: pickedColors });
  return pickedColors;
}

// Indentifies which viewport, if any corresponds to x and y
// Returns first viewport if no match
// TODO - need to determine which viewport we are in
// TODO - document concept of "primary viewport" that matches all coords?
// TODO - static method on Viewport class?
function getViewportFromCoordinates(_ref4) {
  var viewports = _ref4.viewports;

  var viewport = viewports[0];
  return viewport;
}

// Calculate a picking rect centered on deviceX and deviceY and clipped to device
// Returns null if pixel is outside of device
function getPickingRect(_ref5) {
  var deviceX = _ref5.deviceX,
      deviceY = _ref5.deviceY,
      deviceRadius = _ref5.deviceRadius,
      deviceWidth = _ref5.deviceWidth,
      deviceHeight = _ref5.deviceHeight;

  var valid = deviceX >= 0 && deviceY >= 0 && deviceX < deviceWidth && deviceY < deviceHeight;

  // x, y out of bounds.
  if (!valid) {
    return null;
  }

  // Create a box of size `radius * 2 + 1` centered at [deviceX, deviceY]
  var x = Math.max(0, deviceX - deviceRadius);
  var y = Math.max(0, deviceY - deviceRadius);
  var width = Math.min(deviceWidth, deviceX + deviceRadius) - x + 1;
  var height = Math.min(deviceHeight, deviceY + deviceRadius) - y + 1;

  return { x: x, y: y, width: width, height: height };
}

// TODO - break this monster function into 3+ parts
function processPickInfo(_ref6) {
  var pickInfo = _ref6.pickInfo,
      lastPickedInfo = _ref6.lastPickedInfo,
      mode = _ref6.mode,
      layers = _ref6.layers,
      viewports = _ref6.viewports,
      x = _ref6.x,
      y = _ref6.y,
      deviceX = _ref6.deviceX,
      deviceY = _ref6.deviceY,
      pixelRatio = _ref6.pixelRatio;
  var pickedColor = pickInfo.pickedColor,
      pickedLayer = pickInfo.pickedLayer,
      pickedObjectIndex = pickInfo.pickedObjectIndex;


  var affectedLayers = pickedLayer ? [pickedLayer] : [];

  if (mode === 'hover') {
    // only invoke onHover events if picked object has changed
    var lastPickedObjectIndex = lastPickedInfo.index;
    var lastPickedLayerId = lastPickedInfo.layerId;
    var pickedLayerId = pickedLayer && pickedLayer.props.id;

    // proceed only if picked object changed
    if (pickedLayerId !== lastPickedLayerId || pickedObjectIndex !== lastPickedObjectIndex) {
      if (pickedLayerId !== lastPickedLayerId) {
        // We cannot store a ref to lastPickedLayer in the context because
        // the state of an outdated layer is no longer valid
        // and the props may have changed
        var lastPickedLayer = layers.find(function (layer) {
          return layer.props.id === lastPickedLayerId;
        });
        if (lastPickedLayer) {
          // Let leave event fire before enter event
          affectedLayers.unshift(lastPickedLayer);
        }
      }

      // Update layer manager context
      lastPickedInfo.layerId = pickedLayerId;
      lastPickedInfo.index = pickedObjectIndex;
    }
  }

  var viewport = getViewportFromCoordinates({ viewports: viewports }); // TODO - add coords

  var baseInfo = {
    color: null,
    layer: null,
    index: -1,
    picked: false,
    x: x,
    y: y,
    pixel: [x, y],
    lngLat: viewport.unproject([x, y]),
    devicePixel: [deviceX, deviceY],
    pixelRatio: pixelRatio
  };

  // Use a Map to store all picking infos.
  // The following two forEach loops are the result of
  // https://github.com/uber/deck.gl/issues/443
  // Please be very careful when changing this pattern
  var infos = new Map();

  affectedLayers.forEach(function (layer) {
    var info = Object.assign({}, baseInfo);

    if (layer === pickedLayer) {
      info.color = pickedColor;
      info.index = pickedObjectIndex;
      info.picked = true;
    }

    info = getLayerPickingInfo({ layer: layer, info: info, mode: mode });

    // This guarantees that there will be only one copy of info for
    // one composite layer
    if (info) {
      infos.set(info.layer.id, info);
    }

    var pickingSelectedColor = layer.props.autoHighlight && pickedLayer === layer ? pickedColor : null;

    var pickingParameters = {
      pickingSelectedColor: pickingSelectedColor
    };

    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
      for (var _iterator = layer.getModels()[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
        var model = _step.value;

        model.updateModuleSettings(pickingParameters);
      }
    } catch (err) {
      _didIteratorError = true;
      _iteratorError = err;
    } finally {
      try {
        if (!_iteratorNormalCompletion && _iterator.return) {
          _iterator.return();
        }
      } finally {
        if (_didIteratorError) {
          throw _iteratorError;
        }
      }
    }
  });

  var unhandledPickInfos = callLayerPickingCallbacks(infos, mode);

  return unhandledPickInfos;
}

// Per-layer event handlers (e.g. onClick, onHover) are provided by the
// user and out of deck.gl's control. It's very much possible that
// the user calls React lifecycle methods in these function, such as
// ReactComponent.setState(). React lifecycle methods sometimes induce
// a re-render and re-generation of props of deck.gl and its layers,
// which invalidates all layers currently passed to this very function.

// Therefore, per-layer event handlers must be invoked at the end
// of the picking operation. NO operation that relies on the states of current
// layers should be called after this code.
function callLayerPickingCallbacks(infos, mode) {
  var unhandledPickInfos = [];

  infos.forEach(function (info) {
    var handled = false;
    switch (mode) {
      case 'click':
        handled = info.layer.props.onClick(info);
        break;
      case 'hover':
        handled = info.layer.props.onHover(info);
        break;
      case 'query':
        break;
      default:
        throw new Error('unknown pick type');
    }

    if (!handled) {
      unhandledPickInfos.push(info);
    }
  });

  return unhandledPickInfos;
}

/**
 * Pick at a specified pixel with a tolerance radius
 * Returns the closest object to the pixel in shape `{pickedColor, pickedLayer, pickedObjectIndex}`
 */
export function getClosestFromPickingBuffer(gl, _ref7) {
  var pickedColors = _ref7.pickedColors,
      layers = _ref7.layers,
      deviceX = _ref7.deviceX,
      deviceY = _ref7.deviceY,
      deviceRadius = _ref7.deviceRadius,
      deviceRect = _ref7.deviceRect;

  assert(pickedColors);

  // Traverse all pixels in picking results and find the one closest to the supplied
  // [deviceX, deviceY]
  var x = deviceRect.x,
      y = deviceRect.y,
      width = deviceRect.width,
      height = deviceRect.height;

  var minSquareDistanceToCenter = deviceRadius * deviceRadius;
  var closestPixelIndex = -1;
  var i = 0;

  for (var row = 0; row < height; row++) {
    var dy = row + y - deviceY;
    var dy2 = dy * dy;

    if (dy2 > minSquareDistanceToCenter) {
      // skip this row
      i += 4 * width;
    } else {
      for (var col = 0; col < width; col++) {
        // Decode picked layer from color
        var pickedLayerIndex = pickedColors[i + 3] - 1;

        if (pickedLayerIndex >= 0) {
          var dx = col + x - deviceX;
          var d2 = dx * dx + dy2;

          if (d2 <= minSquareDistanceToCenter) {
            minSquareDistanceToCenter = d2;
            closestPixelIndex = i;
          }
        }
        i += 4;
      }
    }
  }

  if (closestPixelIndex >= 0) {
    // Decode picked object index from color
    var _pickedLayerIndex = pickedColors[closestPixelIndex + 3] - 1;
    var pickedColor = pickedColors.slice(closestPixelIndex, closestPixelIndex + 4);
    var pickedLayer = layers[_pickedLayerIndex];
    if (pickedLayer) {
      var pickedObjectIndex = pickedLayer.decodePickingColor(pickedColor);
      return { pickedColor: pickedColor, pickedLayer: pickedLayer, pickedObjectIndex: pickedObjectIndex };
    }
    log.error('Picked non-existent layer. Is picking buffer corrupt?');
  }

  return NO_PICKED_OBJECT;
}
/* eslint-enable max-depth, max-statements */

/**
 * Examines a picking buffer for unique colors
 * Returns array of unique objects in shape `{x, y, pickedColor, pickedLayer, pickedObjectIndex}`
 */
function getUniquesFromPickingBuffer(gl, _ref8) {
  var pickedColors = _ref8.pickedColors,
      layers = _ref8.layers;

  var uniqueColors = new Map();

  // Traverse all pixels in picking results and get unique colors
  if (pickedColors) {
    for (var i = 0; i < pickedColors.length; i += 4) {
      // Decode picked layer from color
      var pickedLayerIndex = pickedColors[i + 3] - 1;

      if (pickedLayerIndex >= 0) {
        var pickedColor = pickedColors.slice(i, i + 4);
        var colorKey = pickedColor.join(',');
        // eslint-disable-next-line
        if (!uniqueColors.has(colorKey)) {
          var pickedLayer = layers[pickedLayerIndex];
          // eslint-disable-next-line
          if (pickedLayer) {
            uniqueColors.set(colorKey, {
              pickedColor: pickedColor,
              pickedLayer: pickedLayer,
              pickedObjectIndex: pickedLayer.decodePickingColor(pickedColor)
            });
          } else {
            log.error('Picked non-existent layer. Is picking buffer corrupt?');
          }
        }
      }
    }
  }

  return Array.from(uniqueColors.values());
}

// Walk up the layer composite chain to populate the info object
function getLayerPickingInfo(_ref9) {
  var layer = _ref9.layer,
      info = _ref9.info,
      mode = _ref9.mode;

  while (layer && info) {
    // For a composite layer, sourceLayer will point to the sublayer
    // where the event originates from.
    // It provides additional context for the composite layer's
    // getPickingInfo() method to populate the info object
    var sourceLayer = info.layer || layer;
    info.layer = layer;
    // layer.pickLayer() function requires a non-null ```layer.state```
    // object to funtion properly. So the layer refereced here
    // must be the "current" layer, not an "out-dated" / "invalidated" layer
    info = layer.pickLayer({ info: info, mode: mode, sourceLayer: sourceLayer });
    layer = layer.parentLayer;
  }
  return info;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9jb3JlL2xpYi9waWNrLWxheWVycy5qcyJdLCJuYW1lcyI6WyJkcmF3UGlja2luZ0J1ZmZlciIsImdldFBpeGVsUmF0aW8iLCJsb2ciLCJhc3NlcnQiLCJOT19QSUNLRURfT0JKRUNUIiwicGlja2VkQ29sb3IiLCJwaWNrZWRMYXllciIsInBpY2tlZE9iamVjdEluZGV4IiwicGlja09iamVjdCIsImdsIiwibGF5ZXJzIiwidmlld3BvcnRzIiwieCIsInkiLCJyYWRpdXMiLCJsYXllckZpbHRlciIsIm1vZGUiLCJvblZpZXdwb3J0QWN0aXZlIiwicGlja2luZ0ZCTyIsImxhc3RQaWNrZWRJbmZvIiwidXNlRGV2aWNlUGl4ZWxzIiwicGl4ZWxSYXRpbyIsImRldmljZVgiLCJNYXRoIiwicm91bmQiLCJkZXZpY2VZIiwiY2FudmFzIiwiaGVpZ2h0IiwiZGV2aWNlUmFkaXVzIiwiZGV2aWNlUmVjdCIsImdldFBpY2tpbmdSZWN0IiwiZGV2aWNlV2lkdGgiLCJ3aWR0aCIsImRldmljZUhlaWdodCIsInBpY2tlZENvbG9ycyIsImRyYXdBbmRTYW1wbGVQaWNraW5nQnVmZmVyIiwicmVkcmF3UmVhc29uIiwicGlja0luZm8iLCJnZXRDbG9zZXN0RnJvbVBpY2tpbmdCdWZmZXIiLCJwcm9jZXNzUGlja0luZm8iLCJwaWNrVmlzaWJsZU9iamVjdHMiLCJkZXZpY2VMZWZ0IiwiZGV2aWNlQm90dG9tIiwiZGV2aWNlUmlnaHQiLCJkZXZpY2VUb3AiLCJwaWNrSW5mb3MiLCJnZXRVbmlxdWVzRnJvbVBpY2tpbmdCdWZmZXIiLCJ1bmlxdWVJbmZvcyIsIk1hcCIsImZvckVhY2giLCJpbmZvIiwiY29sb3IiLCJsYXllciIsImluZGV4IiwicGlja2VkIiwiZ2V0TGF5ZXJQaWNraW5nSW5mbyIsImhhcyIsIm9iamVjdCIsInNldCIsIkFycmF5IiwiZnJvbSIsInZhbHVlcyIsIk51bWJlciIsImlzRmluaXRlIiwicGlja2FibGVMYXllcnMiLCJmaWx0ZXIiLCJpc1BpY2thYmxlIiwibGVuZ3RoIiwiVWludDhBcnJheSIsInJlYWRQaXhlbHMiLCJwaXhlbEFycmF5IiwiZ2V0Vmlld3BvcnRGcm9tQ29vcmRpbmF0ZXMiLCJ2aWV3cG9ydCIsInZhbGlkIiwibWF4IiwibWluIiwiYWZmZWN0ZWRMYXllcnMiLCJsYXN0UGlja2VkT2JqZWN0SW5kZXgiLCJsYXN0UGlja2VkTGF5ZXJJZCIsImxheWVySWQiLCJwaWNrZWRMYXllcklkIiwicHJvcHMiLCJpZCIsImxhc3RQaWNrZWRMYXllciIsImZpbmQiLCJ1bnNoaWZ0IiwiYmFzZUluZm8iLCJwaXhlbCIsImxuZ0xhdCIsInVucHJvamVjdCIsImRldmljZVBpeGVsIiwiaW5mb3MiLCJPYmplY3QiLCJhc3NpZ24iLCJwaWNraW5nU2VsZWN0ZWRDb2xvciIsImF1dG9IaWdobGlnaHQiLCJwaWNraW5nUGFyYW1ldGVycyIsImdldE1vZGVscyIsIm1vZGVsIiwidXBkYXRlTW9kdWxlU2V0dGluZ3MiLCJ1bmhhbmRsZWRQaWNrSW5mb3MiLCJjYWxsTGF5ZXJQaWNraW5nQ2FsbGJhY2tzIiwiaGFuZGxlZCIsIm9uQ2xpY2siLCJvbkhvdmVyIiwiRXJyb3IiLCJwdXNoIiwibWluU3F1YXJlRGlzdGFuY2VUb0NlbnRlciIsImNsb3Nlc3RQaXhlbEluZGV4IiwiaSIsInJvdyIsImR5IiwiZHkyIiwiY29sIiwicGlja2VkTGF5ZXJJbmRleCIsImR4IiwiZDIiLCJzbGljZSIsImRlY29kZVBpY2tpbmdDb2xvciIsImVycm9yIiwidW5pcXVlQ29sb3JzIiwiY29sb3JLZXkiLCJqb2luIiwic291cmNlTGF5ZXIiLCJwaWNrTGF5ZXIiLCJwYXJlbnRMYXllciJdLCJtYXBwaW5ncyI6IkFBQUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBRUEsU0FBUUEsaUJBQVIsRUFBMkJDLGFBQTNCLFFBQStDLGVBQS9DO0FBQ0EsT0FBT0MsR0FBUCxNQUFnQixjQUFoQjtBQUNBLE9BQU9DLE1BQVAsTUFBbUIsUUFBbkI7O0FBRUEsSUFBTUMsbUJBQW1CO0FBQ3ZCQyxlQUFhLElBRFU7QUFFdkJDLGVBQWEsSUFGVTtBQUd2QkMscUJBQW1CLENBQUM7QUFIRyxDQUF6Qjs7QUFNQTtBQUNBO0FBQ0EsT0FBTyxTQUFTQyxVQUFULENBQ0xDLEVBREssUUFlTDtBQUFBLE1BWkVDLE1BWUYsUUFaRUEsTUFZRjtBQUFBLE1BWEVDLFNBV0YsUUFYRUEsU0FXRjtBQUFBLE1BVkVDLENBVUYsUUFWRUEsQ0FVRjtBQUFBLE1BVEVDLENBU0YsUUFURUEsQ0FTRjtBQUFBLE1BUkVDLE1BUUYsUUFSRUEsTUFRRjtBQUFBLE1BUEVDLFdBT0YsUUFQRUEsV0FPRjtBQUFBLE1BTkVDLElBTUYsUUFORUEsSUFNRjtBQUFBLE1BTEVDLGdCQUtGLFFBTEVBLGdCQUtGO0FBQUEsTUFKRUMsVUFJRixRQUpFQSxVQUlGO0FBQUEsTUFIRUMsY0FHRixRQUhFQSxjQUdGO0FBQUEsTUFGRUMsZUFFRixRQUZFQSxlQUVGOztBQUNBO0FBQ0E7QUFDQSxNQUFNQyxhQUFhcEIsY0FBYyxFQUFDbUIsZ0NBQUQsRUFBZCxDQUFuQjtBQUNBLE1BQU1FLFVBQVVDLEtBQUtDLEtBQUwsQ0FBV1osSUFBSVMsVUFBZixDQUFoQjtBQUNBLE1BQU1JLFVBQVVGLEtBQUtDLEtBQUwsQ0FBV2YsR0FBR2lCLE1BQUgsQ0FBVUMsTUFBVixHQUFtQmQsSUFBSVEsVUFBbEMsQ0FBaEI7QUFDQSxNQUFNTyxlQUFlTCxLQUFLQyxLQUFMLENBQVdWLFNBQVNPLFVBQXBCLENBQXJCOztBQUVBLE1BQU1RLGFBQWFDLGVBQWU7QUFDaENSLG9CQURnQztBQUVoQ0csb0JBRmdDO0FBR2hDRyw4QkFIZ0M7QUFJaENHLGlCQUFhYixXQUFXYyxLQUpRO0FBS2hDQyxrQkFBY2YsV0FBV1M7QUFMTyxHQUFmLENBQW5COztBQVFBLE1BQU1PLGVBQ0pMLGNBQ0FNLDJCQUEyQjFCLEVBQTNCLEVBQStCO0FBQzdCQyxrQkFENkI7QUFFN0JDLHdCQUY2QjtBQUc3Qk0sc0NBSDZCO0FBSTdCRyxvQ0FKNkI7QUFLN0JGLDBCQUw2QjtBQU03QlcsMEJBTjZCO0FBTzdCZCw0QkFQNkI7QUFRN0JxQixrQkFBY3BCO0FBUmUsR0FBL0IsQ0FGRjs7QUFhQSxNQUFNcUIsV0FDSEgsZ0JBQ0NJLDRCQUE0QjdCLEVBQTVCLEVBQWdDO0FBQzlCeUIsOEJBRDhCO0FBRTlCeEIsa0JBRjhCO0FBRzlCWSxvQkFIOEI7QUFJOUJHLG9CQUo4QjtBQUs5QkcsOEJBTDhCO0FBTTlCQztBQU44QixHQUFoQyxDQURGLElBU0F6QixnQkFWRjs7QUFZQSxTQUFPbUMsZ0JBQWdCO0FBQ3JCRixzQkFEcUI7QUFFckJsQixrQ0FGcUI7QUFHckJILGNBSHFCO0FBSXJCTixrQkFKcUI7QUFLckJDLHdCQUxxQjtBQU1yQkMsUUFOcUI7QUFPckJDLFFBUHFCO0FBUXJCUyxvQkFScUI7QUFTckJHLG9CQVRxQjtBQVVyQko7QUFWcUIsR0FBaEIsQ0FBUDtBQVlEOztBQUVEO0FBQ0EsT0FBTyxTQUFTbUIsa0JBQVQsQ0FDTC9CLEVBREssU0FlTDtBQUFBLE1BWkVDLE1BWUYsU0FaRUEsTUFZRjtBQUFBLE1BWEVDLFNBV0YsU0FYRUEsU0FXRjtBQUFBLE1BVkVDLENBVUYsU0FWRUEsQ0FVRjtBQUFBLE1BVEVDLENBU0YsU0FURUEsQ0FTRjtBQUFBLE1BUkVtQixLQVFGLFNBUkVBLEtBUUY7QUFBQSxNQVBFTCxNQU9GLFNBUEVBLE1BT0Y7QUFBQSxNQU5FWCxJQU1GLFNBTkVBLElBTUY7QUFBQSxNQUxFRCxXQUtGLFNBTEVBLFdBS0Y7QUFBQSxNQUpFRSxnQkFJRixTQUpFQSxnQkFJRjtBQUFBLE1BSEVDLFVBR0YsU0FIRUEsVUFHRjtBQUFBLE1BRkVFLGVBRUYsU0FGRUEsZUFFRjs7QUFDQTtBQUNBO0FBQ0EsTUFBTUMsYUFBYXBCLGNBQWMsRUFBQ21CLGdDQUFELEVBQWQsQ0FBbkI7O0FBRUEsTUFBTXFCLGFBQWFsQixLQUFLQyxLQUFMLENBQVdaLElBQUlTLFVBQWYsQ0FBbkI7QUFDQSxNQUFNcUIsZUFBZW5CLEtBQUtDLEtBQUwsQ0FBV2YsR0FBR2lCLE1BQUgsQ0FBVUMsTUFBVixHQUFtQmQsSUFBSVEsVUFBbEMsQ0FBckI7QUFDQSxNQUFNc0IsY0FBY3BCLEtBQUtDLEtBQUwsQ0FBVyxDQUFDWixJQUFJb0IsS0FBTCxJQUFjWCxVQUF6QixDQUFwQjtBQUNBLE1BQU11QixZQUFZckIsS0FBS0MsS0FBTCxDQUFXZixHQUFHaUIsTUFBSCxDQUFVQyxNQUFWLEdBQW1CLENBQUNkLElBQUljLE1BQUwsSUFBZU4sVUFBN0MsQ0FBbEI7O0FBRUEsTUFBTVEsYUFBYTtBQUNqQmpCLE9BQUc2QixVQURjO0FBRWpCNUIsT0FBRytCLFNBRmM7QUFHakJaLFdBQU9XLGNBQWNGLFVBSEo7QUFJakJkLFlBQVFlLGVBQWVFO0FBSk4sR0FBbkI7O0FBT0EsTUFBTVYsZUFBZUMsMkJBQTJCMUIsRUFBM0IsRUFBK0I7QUFDbERDLGtCQURrRDtBQUVsREMsd0JBRmtEO0FBR2xETSxzQ0FIa0Q7QUFJbERDLDBCQUprRDtBQUtsREUsb0NBTGtEO0FBTWxEUywwQkFOa0Q7QUFPbERkLDRCQVBrRDtBQVFsRHFCLGtCQUFjcEI7QUFSb0MsR0FBL0IsQ0FBckI7O0FBV0EsTUFBTTZCLFlBQVlDLDRCQUE0QnJDLEVBQTVCLEVBQWdDLEVBQUN5QiwwQkFBRCxFQUFleEIsY0FBZixFQUFoQyxDQUFsQjs7QUFFQTtBQUNBLE1BQU1xQyxjQUFjLElBQUlDLEdBQUosRUFBcEI7O0FBRUFILFlBQVVJLE9BQVYsQ0FBa0Isb0JBQVk7QUFDNUIsUUFBSUMsT0FBTztBQUNUQyxhQUFPZCxTQUFTaEMsV0FEUDtBQUVUK0MsYUFBTyxJQUZFO0FBR1RDLGFBQU9oQixTQUFTOUIsaUJBSFA7QUFJVCtDLGNBQVEsSUFKQztBQUtUMUMsVUFMUztBQU1UQyxVQU5TO0FBT1RtQixrQkFQUztBQVFUTCxvQkFSUztBQVNUTjtBQVRTLEtBQVg7O0FBWUE2QixXQUFPSyxvQkFBb0IsRUFBQ0gsT0FBT2YsU0FBUy9CLFdBQWpCLEVBQThCNEMsVUFBOUIsRUFBb0NsQyxVQUFwQyxFQUFwQixDQUFQO0FBQ0EsUUFBSSxDQUFDK0IsWUFBWVMsR0FBWixDQUFnQk4sS0FBS08sTUFBckIsQ0FBTCxFQUFtQztBQUNqQ1Ysa0JBQVlXLEdBQVosQ0FBZ0JSLEtBQUtPLE1BQXJCLEVBQTZCUCxJQUE3QjtBQUNEO0FBQ0YsR0FqQkQ7O0FBbUJBLFNBQU9TLE1BQU1DLElBQU4sQ0FBV2IsWUFBWWMsTUFBWixFQUFYLENBQVA7QUFDRDs7QUFFRDs7QUFFQTtBQUNBLFNBQVMxQiwwQkFBVCxDQUNFMUIsRUFERixTQVlFO0FBQUEsTUFURUMsTUFTRixTQVRFQSxNQVNGO0FBQUEsTUFSRUMsU0FRRixTQVJFQSxTQVFGO0FBQUEsTUFQRU0sZ0JBT0YsU0FQRUEsZ0JBT0Y7QUFBQSxNQU5FRyxlQU1GLFNBTkVBLGVBTUY7QUFBQSxNQUxFRixVQUtGLFNBTEVBLFVBS0Y7QUFBQSxNQUpFVyxVQUlGLFNBSkVBLFVBSUY7QUFBQSxNQUhFZCxXQUdGLFNBSEVBLFdBR0Y7QUFBQSxNQUZFcUIsWUFFRixTQUZFQSxZQUVGOztBQUNBakMsU0FBTzBCLFVBQVA7QUFDQTFCLFNBQU8yRCxPQUFPQyxRQUFQLENBQWdCbEMsV0FBV0csS0FBM0IsS0FBcUNILFdBQVdHLEtBQVgsR0FBbUIsQ0FBL0QsRUFBa0UscUJBQWxFO0FBQ0E3QixTQUFPMkQsT0FBT0MsUUFBUCxDQUFnQmxDLFdBQVdGLE1BQTNCLEtBQXNDRSxXQUFXRixNQUFYLEdBQW9CLENBQWpFLEVBQW9FLHNCQUFwRTs7QUFFQSxNQUFNcUMsaUJBQWlCdEQsT0FBT3VELE1BQVAsQ0FBYztBQUFBLFdBQVNiLE1BQU1jLFVBQU4sRUFBVDtBQUFBLEdBQWQsQ0FBdkI7QUFDQSxNQUFJRixlQUFlRyxNQUFmLEdBQXdCLENBQTVCLEVBQStCO0FBQzdCLFdBQU8sSUFBUDtBQUNEOztBQUVEbkUsb0JBQWtCUyxFQUFsQixFQUFzQjtBQUNwQkMsa0JBRG9CO0FBRXBCQyx3QkFGb0I7QUFHcEJNLHNDQUhvQjtBQUlwQkcsb0NBSm9CO0FBS3BCRiwwQkFMb0I7QUFNcEJXLDBCQU5vQjtBQU9wQmQsNEJBUG9CO0FBUXBCcUI7QUFSb0IsR0FBdEI7O0FBV0E7QUFDQTtBQXRCQSxNQXVCT3hCLENBdkJQLEdBdUI4QmlCLFVBdkI5QixDQXVCT2pCLENBdkJQO0FBQUEsTUF1QlVDLENBdkJWLEdBdUI4QmdCLFVBdkI5QixDQXVCVWhCLENBdkJWO0FBQUEsTUF1QmFtQixLQXZCYixHQXVCOEJILFVBdkI5QixDQXVCYUcsS0F2QmI7QUFBQSxNQXVCb0JMLE1BdkJwQixHQXVCOEJFLFVBdkI5QixDQXVCb0JGLE1BdkJwQjs7QUF3QkEsTUFBTU8sZUFBZSxJQUFJa0MsVUFBSixDQUFlcEMsUUFBUUwsTUFBUixHQUFpQixDQUFoQyxDQUFyQjtBQUNBVCxhQUFXbUQsVUFBWCxDQUFzQixFQUFDekQsSUFBRCxFQUFJQyxJQUFKLEVBQU9tQixZQUFQLEVBQWNMLGNBQWQsRUFBc0IyQyxZQUFZcEMsWUFBbEMsRUFBdEI7QUFDQSxTQUFPQSxZQUFQO0FBQ0Q7O0FBRUQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLFNBQVNxQywwQkFBVCxRQUFpRDtBQUFBLE1BQVo1RCxTQUFZLFNBQVpBLFNBQVk7O0FBQy9DLE1BQU02RCxXQUFXN0QsVUFBVSxDQUFWLENBQWpCO0FBQ0EsU0FBTzZELFFBQVA7QUFDRDs7QUFFRDtBQUNBO0FBQ0EsU0FBUzFDLGNBQVQsUUFBcUY7QUFBQSxNQUE1RFIsT0FBNEQsU0FBNURBLE9BQTREO0FBQUEsTUFBbkRHLE9BQW1ELFNBQW5EQSxPQUFtRDtBQUFBLE1BQTFDRyxZQUEwQyxTQUExQ0EsWUFBMEM7QUFBQSxNQUE1QkcsV0FBNEIsU0FBNUJBLFdBQTRCO0FBQUEsTUFBZkUsWUFBZSxTQUFmQSxZQUFlOztBQUNuRixNQUFNd0MsUUFBUW5ELFdBQVcsQ0FBWCxJQUFnQkcsV0FBVyxDQUEzQixJQUFnQ0gsVUFBVVMsV0FBMUMsSUFBeUROLFVBQVVRLFlBQWpGOztBQUVBO0FBQ0EsTUFBSSxDQUFDd0MsS0FBTCxFQUFZO0FBQ1YsV0FBTyxJQUFQO0FBQ0Q7O0FBRUQ7QUFDQSxNQUFNN0QsSUFBSVcsS0FBS21ELEdBQUwsQ0FBUyxDQUFULEVBQVlwRCxVQUFVTSxZQUF0QixDQUFWO0FBQ0EsTUFBTWYsSUFBSVUsS0FBS21ELEdBQUwsQ0FBUyxDQUFULEVBQVlqRCxVQUFVRyxZQUF0QixDQUFWO0FBQ0EsTUFBTUksUUFBUVQsS0FBS29ELEdBQUwsQ0FBUzVDLFdBQVQsRUFBc0JULFVBQVVNLFlBQWhDLElBQWdEaEIsQ0FBaEQsR0FBb0QsQ0FBbEU7QUFDQSxNQUFNZSxTQUFTSixLQUFLb0QsR0FBTCxDQUFTMUMsWUFBVCxFQUF1QlIsVUFBVUcsWUFBakMsSUFBaURmLENBQWpELEdBQXFELENBQXBFOztBQUVBLFNBQU8sRUFBQ0QsSUFBRCxFQUFJQyxJQUFKLEVBQU9tQixZQUFQLEVBQWNMLGNBQWQsRUFBUDtBQUNEOztBQUVEO0FBQ0EsU0FBU1ksZUFBVCxRQVdHO0FBQUEsTUFWREYsUUFVQyxTQVZEQSxRQVVDO0FBQUEsTUFURGxCLGNBU0MsU0FUREEsY0FTQztBQUFBLE1BUkRILElBUUMsU0FSREEsSUFRQztBQUFBLE1BUEROLE1BT0MsU0FQREEsTUFPQztBQUFBLE1BTkRDLFNBTUMsU0FOREEsU0FNQztBQUFBLE1BTERDLENBS0MsU0FMREEsQ0FLQztBQUFBLE1BSkRDLENBSUMsU0FKREEsQ0FJQztBQUFBLE1BSERTLE9BR0MsU0FIREEsT0FHQztBQUFBLE1BRkRHLE9BRUMsU0FGREEsT0FFQztBQUFBLE1BRERKLFVBQ0MsU0FEREEsVUFDQztBQUFBLE1BQ01oQixXQUROLEdBQ3FEZ0MsUUFEckQsQ0FDTWhDLFdBRE47QUFBQSxNQUNtQkMsV0FEbkIsR0FDcUQrQixRQURyRCxDQUNtQi9CLFdBRG5CO0FBQUEsTUFDZ0NDLGlCQURoQyxHQUNxRDhCLFFBRHJELENBQ2dDOUIsaUJBRGhDOzs7QUFHRCxNQUFNcUUsaUJBQWlCdEUsY0FBYyxDQUFDQSxXQUFELENBQWQsR0FBOEIsRUFBckQ7O0FBRUEsTUFBSVUsU0FBUyxPQUFiLEVBQXNCO0FBQ3BCO0FBQ0EsUUFBTTZELHdCQUF3QjFELGVBQWVrQyxLQUE3QztBQUNBLFFBQU15QixvQkFBb0IzRCxlQUFlNEQsT0FBekM7QUFDQSxRQUFNQyxnQkFBZ0IxRSxlQUFlQSxZQUFZMkUsS0FBWixDQUFrQkMsRUFBdkQ7O0FBRUE7QUFDQSxRQUFJRixrQkFBa0JGLGlCQUFsQixJQUF1Q3ZFLHNCQUFzQnNFLHFCQUFqRSxFQUF3RjtBQUN0RixVQUFJRyxrQkFBa0JGLGlCQUF0QixFQUF5QztBQUN2QztBQUNBO0FBQ0E7QUFDQSxZQUFNSyxrQkFBa0J6RSxPQUFPMEUsSUFBUCxDQUFZO0FBQUEsaUJBQVNoQyxNQUFNNkIsS0FBTixDQUFZQyxFQUFaLEtBQW1CSixpQkFBNUI7QUFBQSxTQUFaLENBQXhCO0FBQ0EsWUFBSUssZUFBSixFQUFxQjtBQUNuQjtBQUNBUCx5QkFBZVMsT0FBZixDQUF1QkYsZUFBdkI7QUFDRDtBQUNGOztBQUVEO0FBQ0FoRSxxQkFBZTRELE9BQWYsR0FBeUJDLGFBQXpCO0FBQ0E3RCxxQkFBZWtDLEtBQWYsR0FBdUI5QyxpQkFBdkI7QUFDRDtBQUNGOztBQUVELE1BQU1pRSxXQUFXRCwyQkFBMkIsRUFBQzVELG9CQUFELEVBQTNCLENBQWpCLENBOUJDLENBOEJ5RDs7QUFFMUQsTUFBTTJFLFdBQVc7QUFDZm5DLFdBQU8sSUFEUTtBQUVmQyxXQUFPLElBRlE7QUFHZkMsV0FBTyxDQUFDLENBSE87QUFJZkMsWUFBUSxLQUpPO0FBS2YxQyxRQUxlO0FBTWZDLFFBTmU7QUFPZjBFLFdBQU8sQ0FBQzNFLENBQUQsRUFBSUMsQ0FBSixDQVBRO0FBUWYyRSxZQUFRaEIsU0FBU2lCLFNBQVQsQ0FBbUIsQ0FBQzdFLENBQUQsRUFBSUMsQ0FBSixDQUFuQixDQVJPO0FBU2Y2RSxpQkFBYSxDQUFDcEUsT0FBRCxFQUFVRyxPQUFWLENBVEU7QUFVZko7QUFWZSxHQUFqQjs7QUFhQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLE1BQU1zRSxRQUFRLElBQUkzQyxHQUFKLEVBQWQ7O0FBRUE0QixpQkFBZTNCLE9BQWYsQ0FBdUIsaUJBQVM7QUFDOUIsUUFBSUMsT0FBTzBDLE9BQU9DLE1BQVAsQ0FBYyxFQUFkLEVBQWtCUCxRQUFsQixDQUFYOztBQUVBLFFBQUlsQyxVQUFVOUMsV0FBZCxFQUEyQjtBQUN6QjRDLFdBQUtDLEtBQUwsR0FBYTlDLFdBQWI7QUFDQTZDLFdBQUtHLEtBQUwsR0FBYTlDLGlCQUFiO0FBQ0EyQyxXQUFLSSxNQUFMLEdBQWMsSUFBZDtBQUNEOztBQUVESixXQUFPSyxvQkFBb0IsRUFBQ0gsWUFBRCxFQUFRRixVQUFSLEVBQWNsQyxVQUFkLEVBQXBCLENBQVA7O0FBRUE7QUFDQTtBQUNBLFFBQUlrQyxJQUFKLEVBQVU7QUFDUnlDLFlBQU1qQyxHQUFOLENBQVVSLEtBQUtFLEtBQUwsQ0FBVzhCLEVBQXJCLEVBQXlCaEMsSUFBekI7QUFDRDs7QUFFRCxRQUFNNEMsdUJBQ0oxQyxNQUFNNkIsS0FBTixDQUFZYyxhQUFaLElBQTZCekYsZ0JBQWdCOEMsS0FBN0MsR0FBcUQvQyxXQUFyRCxHQUFtRSxJQURyRTs7QUFHQSxRQUFNMkYsb0JBQW9CO0FBQ3hCRjtBQUR3QixLQUExQjs7QUFwQjhCO0FBQUE7QUFBQTs7QUFBQTtBQXdCOUIsMkJBQW9CMUMsTUFBTTZDLFNBQU4sRUFBcEIsOEhBQXVDO0FBQUEsWUFBNUJDLEtBQTRCOztBQUNyQ0EsY0FBTUMsb0JBQU4sQ0FBMkJILGlCQUEzQjtBQUNEO0FBMUI2QjtBQUFBO0FBQUE7QUFBQTtBQUFBO0FBQUE7QUFBQTtBQUFBO0FBQUE7QUFBQTtBQUFBO0FBQUE7QUFBQTtBQUFBO0FBMkIvQixHQTNCRDs7QUE2QkEsTUFBTUkscUJBQXFCQywwQkFBMEJWLEtBQTFCLEVBQWlDM0UsSUFBakMsQ0FBM0I7O0FBRUEsU0FBT29GLGtCQUFQO0FBQ0Q7O0FBRUQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUVBO0FBQ0E7QUFDQTtBQUNBLFNBQVNDLHlCQUFULENBQW1DVixLQUFuQyxFQUEwQzNFLElBQTFDLEVBQWdEO0FBQzlDLE1BQU1vRixxQkFBcUIsRUFBM0I7O0FBRUFULFFBQU0xQyxPQUFOLENBQWMsZ0JBQVE7QUFDcEIsUUFBSXFELFVBQVUsS0FBZDtBQUNBLFlBQVF0RixJQUFSO0FBQ0UsV0FBSyxPQUFMO0FBQ0VzRixrQkFBVXBELEtBQUtFLEtBQUwsQ0FBVzZCLEtBQVgsQ0FBaUJzQixPQUFqQixDQUF5QnJELElBQXpCLENBQVY7QUFDQTtBQUNGLFdBQUssT0FBTDtBQUNFb0Qsa0JBQVVwRCxLQUFLRSxLQUFMLENBQVc2QixLQUFYLENBQWlCdUIsT0FBakIsQ0FBeUJ0RCxJQUF6QixDQUFWO0FBQ0E7QUFDRixXQUFLLE9BQUw7QUFDRTtBQUNGO0FBQ0UsY0FBTSxJQUFJdUQsS0FBSixDQUFVLG1CQUFWLENBQU47QUFWSjs7QUFhQSxRQUFJLENBQUNILE9BQUwsRUFBYztBQUNaRix5QkFBbUJNLElBQW5CLENBQXdCeEQsSUFBeEI7QUFDRDtBQUNGLEdBbEJEOztBQW9CQSxTQUFPa0Qsa0JBQVA7QUFDRDs7QUFFRDs7OztBQUlBLE9BQU8sU0FBUzlELDJCQUFULENBQ0w3QixFQURLLFNBR0w7QUFBQSxNQURDeUIsWUFDRCxTQURDQSxZQUNEO0FBQUEsTUFEZXhCLE1BQ2YsU0FEZUEsTUFDZjtBQUFBLE1BRHVCWSxPQUN2QixTQUR1QkEsT0FDdkI7QUFBQSxNQURnQ0csT0FDaEMsU0FEZ0NBLE9BQ2hDO0FBQUEsTUFEeUNHLFlBQ3pDLFNBRHlDQSxZQUN6QztBQUFBLE1BRHVEQyxVQUN2RCxTQUR1REEsVUFDdkQ7O0FBQ0ExQixTQUFPK0IsWUFBUDs7QUFFQTtBQUNBO0FBSkEsTUFLT3RCLENBTFAsR0FLOEJpQixVQUw5QixDQUtPakIsQ0FMUDtBQUFBLE1BS1VDLENBTFYsR0FLOEJnQixVQUw5QixDQUtVaEIsQ0FMVjtBQUFBLE1BS2FtQixLQUxiLEdBSzhCSCxVQUw5QixDQUthRyxLQUxiO0FBQUEsTUFLb0JMLE1BTHBCLEdBSzhCRSxVQUw5QixDQUtvQkYsTUFMcEI7O0FBTUEsTUFBSWdGLDRCQUE0Qi9FLGVBQWVBLFlBQS9DO0FBQ0EsTUFBSWdGLG9CQUFvQixDQUFDLENBQXpCO0FBQ0EsTUFBSUMsSUFBSSxDQUFSOztBQUVBLE9BQUssSUFBSUMsTUFBTSxDQUFmLEVBQWtCQSxNQUFNbkYsTUFBeEIsRUFBZ0NtRixLQUFoQyxFQUF1QztBQUNyQyxRQUFNQyxLQUFLRCxNQUFNakcsQ0FBTixHQUFVWSxPQUFyQjtBQUNBLFFBQU11RixNQUFNRCxLQUFLQSxFQUFqQjs7QUFFQSxRQUFJQyxNQUFNTCx5QkFBVixFQUFxQztBQUNuQztBQUNBRSxXQUFLLElBQUk3RSxLQUFUO0FBQ0QsS0FIRCxNQUdPO0FBQ0wsV0FBSyxJQUFJaUYsTUFBTSxDQUFmLEVBQWtCQSxNQUFNakYsS0FBeEIsRUFBK0JpRixLQUEvQixFQUFzQztBQUNwQztBQUNBLFlBQU1DLG1CQUFtQmhGLGFBQWEyRSxJQUFJLENBQWpCLElBQXNCLENBQS9DOztBQUVBLFlBQUlLLG9CQUFvQixDQUF4QixFQUEyQjtBQUN6QixjQUFNQyxLQUFLRixNQUFNckcsQ0FBTixHQUFVVSxPQUFyQjtBQUNBLGNBQU04RixLQUFLRCxLQUFLQSxFQUFMLEdBQVVILEdBQXJCOztBQUVBLGNBQUlJLE1BQU1ULHlCQUFWLEVBQXFDO0FBQ25DQSx3Q0FBNEJTLEVBQTVCO0FBQ0FSLGdDQUFvQkMsQ0FBcEI7QUFDRDtBQUNGO0FBQ0RBLGFBQUssQ0FBTDtBQUNEO0FBQ0Y7QUFDRjs7QUFFRCxNQUFJRCxxQkFBcUIsQ0FBekIsRUFBNEI7QUFDMUI7QUFDQSxRQUFNTSxvQkFBbUJoRixhQUFhMEUsb0JBQW9CLENBQWpDLElBQXNDLENBQS9EO0FBQ0EsUUFBTXZHLGNBQWM2QixhQUFhbUYsS0FBYixDQUFtQlQsaUJBQW5CLEVBQXNDQSxvQkFBb0IsQ0FBMUQsQ0FBcEI7QUFDQSxRQUFNdEcsY0FBY0ksT0FBT3dHLGlCQUFQLENBQXBCO0FBQ0EsUUFBSTVHLFdBQUosRUFBaUI7QUFDZixVQUFNQyxvQkFBb0JELFlBQVlnSCxrQkFBWixDQUErQmpILFdBQS9CLENBQTFCO0FBQ0EsYUFBTyxFQUFDQSx3QkFBRCxFQUFjQyx3QkFBZCxFQUEyQkMsb0NBQTNCLEVBQVA7QUFDRDtBQUNETCxRQUFJcUgsS0FBSixDQUFVLHVEQUFWO0FBQ0Q7O0FBRUQsU0FBT25ILGdCQUFQO0FBQ0Q7QUFDRDs7QUFFQTs7OztBQUlBLFNBQVMwQywyQkFBVCxDQUFxQ3JDLEVBQXJDLFNBQWlFO0FBQUEsTUFBdkJ5QixZQUF1QixTQUF2QkEsWUFBdUI7QUFBQSxNQUFUeEIsTUFBUyxTQUFUQSxNQUFTOztBQUMvRCxNQUFNOEcsZUFBZSxJQUFJeEUsR0FBSixFQUFyQjs7QUFFQTtBQUNBLE1BQUlkLFlBQUosRUFBa0I7QUFDaEIsU0FBSyxJQUFJMkUsSUFBSSxDQUFiLEVBQWdCQSxJQUFJM0UsYUFBYWlDLE1BQWpDLEVBQXlDMEMsS0FBSyxDQUE5QyxFQUFpRDtBQUMvQztBQUNBLFVBQU1LLG1CQUFtQmhGLGFBQWEyRSxJQUFJLENBQWpCLElBQXNCLENBQS9DOztBQUVBLFVBQUlLLG9CQUFvQixDQUF4QixFQUEyQjtBQUN6QixZQUFNN0csY0FBYzZCLGFBQWFtRixLQUFiLENBQW1CUixDQUFuQixFQUFzQkEsSUFBSSxDQUExQixDQUFwQjtBQUNBLFlBQU1ZLFdBQVdwSCxZQUFZcUgsSUFBWixDQUFpQixHQUFqQixDQUFqQjtBQUNBO0FBQ0EsWUFBSSxDQUFDRixhQUFhaEUsR0FBYixDQUFpQmlFLFFBQWpCLENBQUwsRUFBaUM7QUFDL0IsY0FBTW5ILGNBQWNJLE9BQU93RyxnQkFBUCxDQUFwQjtBQUNBO0FBQ0EsY0FBSTVHLFdBQUosRUFBaUI7QUFDZmtILHlCQUFhOUQsR0FBYixDQUFpQitELFFBQWpCLEVBQTJCO0FBQ3pCcEgsc0NBRHlCO0FBRXpCQyxzQ0FGeUI7QUFHekJDLGlDQUFtQkQsWUFBWWdILGtCQUFaLENBQStCakgsV0FBL0I7QUFITSxhQUEzQjtBQUtELFdBTkQsTUFNTztBQUNMSCxnQkFBSXFILEtBQUosQ0FBVSx1REFBVjtBQUNEO0FBQ0Y7QUFDRjtBQUNGO0FBQ0Y7O0FBRUQsU0FBTzVELE1BQU1DLElBQU4sQ0FBVzRELGFBQWEzRCxNQUFiLEVBQVgsQ0FBUDtBQUNEOztBQUVEO0FBQ0EsU0FBU04sbUJBQVQsUUFBa0Q7QUFBQSxNQUFwQkgsS0FBb0IsU0FBcEJBLEtBQW9CO0FBQUEsTUFBYkYsSUFBYSxTQUFiQSxJQUFhO0FBQUEsTUFBUGxDLElBQU8sU0FBUEEsSUFBTzs7QUFDaEQsU0FBT29DLFNBQVNGLElBQWhCLEVBQXNCO0FBQ3BCO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsUUFBTXlFLGNBQWN6RSxLQUFLRSxLQUFMLElBQWNBLEtBQWxDO0FBQ0FGLFNBQUtFLEtBQUwsR0FBYUEsS0FBYjtBQUNBO0FBQ0E7QUFDQTtBQUNBRixXQUFPRSxNQUFNd0UsU0FBTixDQUFnQixFQUFDMUUsVUFBRCxFQUFPbEMsVUFBUCxFQUFhMkcsd0JBQWIsRUFBaEIsQ0FBUDtBQUNBdkUsWUFBUUEsTUFBTXlFLFdBQWQ7QUFDRDtBQUNELFNBQU8zRSxJQUFQO0FBQ0QiLCJmaWxlIjoicGljay1sYXllcnMuanMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgKGMpIDIwMTUgLSAyMDE3IFViZXIgVGVjaG5vbG9naWVzLCBJbmMuXG4vL1xuLy8gUGVybWlzc2lvbiBpcyBoZXJlYnkgZ3JhbnRlZCwgZnJlZSBvZiBjaGFyZ2UsIHRvIGFueSBwZXJzb24gb2J0YWluaW5nIGEgY29weVxuLy8gb2YgdGhpcyBzb2Z0d2FyZSBhbmQgYXNzb2NpYXRlZCBkb2N1bWVudGF0aW9uIGZpbGVzICh0aGUgXCJTb2Z0d2FyZVwiKSwgdG8gZGVhbFxuLy8gaW4gdGhlIFNvZnR3YXJlIHdpdGhvdXQgcmVzdHJpY3Rpb24sIGluY2x1ZGluZyB3aXRob3V0IGxpbWl0YXRpb24gdGhlIHJpZ2h0c1xuLy8gdG8gdXNlLCBjb3B5LCBtb2RpZnksIG1lcmdlLCBwdWJsaXNoLCBkaXN0cmlidXRlLCBzdWJsaWNlbnNlLCBhbmQvb3Igc2VsbFxuLy8gY29waWVzIG9mIHRoZSBTb2Z0d2FyZSwgYW5kIHRvIHBlcm1pdCBwZXJzb25zIHRvIHdob20gdGhlIFNvZnR3YXJlIGlzXG4vLyBmdXJuaXNoZWQgdG8gZG8gc28sIHN1YmplY3QgdG8gdGhlIGZvbGxvd2luZyBjb25kaXRpb25zOlxuLy9cbi8vIFRoZSBhYm92ZSBjb3B5cmlnaHQgbm90aWNlIGFuZCB0aGlzIHBlcm1pc3Npb24gbm90aWNlIHNoYWxsIGJlIGluY2x1ZGVkIGluXG4vLyBhbGwgY29waWVzIG9yIHN1YnN0YW50aWFsIHBvcnRpb25zIG9mIHRoZSBTb2Z0d2FyZS5cbi8vXG4vLyBUSEUgU09GVFdBUkUgSVMgUFJPVklERUQgXCJBUyBJU1wiLCBXSVRIT1VUIFdBUlJBTlRZIE9GIEFOWSBLSU5ELCBFWFBSRVNTIE9SXG4vLyBJTVBMSUVELCBJTkNMVURJTkcgQlVUIE5PVCBMSU1JVEVEIFRPIFRIRSBXQVJSQU5USUVTIE9GIE1FUkNIQU5UQUJJTElUWSxcbi8vIEZJVE5FU1MgRk9SIEEgUEFSVElDVUxBUiBQVVJQT1NFIEFORCBOT05JTkZSSU5HRU1FTlQuIElOIE5PIEVWRU5UIFNIQUxMIFRIRVxuLy8gQVVUSE9SUyBPUiBDT1BZUklHSFQgSE9MREVSUyBCRSBMSUFCTEUgRk9SIEFOWSBDTEFJTSwgREFNQUdFUyBPUiBPVEhFUlxuLy8gTElBQklMSVRZLCBXSEVUSEVSIElOIEFOIEFDVElPTiBPRiBDT05UUkFDVCwgVE9SVCBPUiBPVEhFUldJU0UsIEFSSVNJTkcgRlJPTSxcbi8vIE9VVCBPRiBPUiBJTiBDT05ORUNUSU9OIFdJVEggVEhFIFNPRlRXQVJFIE9SIFRIRSBVU0UgT1IgT1RIRVIgREVBTElOR1MgSU5cbi8vIFRIRSBTT0ZUV0FSRS5cblxuaW1wb3J0IHtkcmF3UGlja2luZ0J1ZmZlciwgZ2V0UGl4ZWxSYXRpb30gZnJvbSAnLi9kcmF3LWxheWVycyc7XG5pbXBvcnQgbG9nIGZyb20gJy4uL3V0aWxzL2xvZyc7XG5pbXBvcnQgYXNzZXJ0IGZyb20gJ2Fzc2VydCc7XG5cbmNvbnN0IE5PX1BJQ0tFRF9PQkpFQ1QgPSB7XG4gIHBpY2tlZENvbG9yOiBudWxsLFxuICBwaWNrZWRMYXllcjogbnVsbCxcbiAgcGlja2VkT2JqZWN0SW5kZXg6IC0xXG59O1xuXG4vKiBlc2xpbnQtZGlzYWJsZSBtYXgtZGVwdGgsIG1heC1zdGF0ZW1lbnRzICovXG4vLyBQaWNrIHRoZSBjbG9zZXN0IG9iamVjdCBhdCB0aGUgZ2l2ZW4gKHgseSkgY29vcmRpbmF0ZVxuZXhwb3J0IGZ1bmN0aW9uIHBpY2tPYmplY3QoXG4gIGdsLFxuICB7XG4gICAgbGF5ZXJzLFxuICAgIHZpZXdwb3J0cyxcbiAgICB4LFxuICAgIHksXG4gICAgcmFkaXVzLFxuICAgIGxheWVyRmlsdGVyLFxuICAgIG1vZGUsXG4gICAgb25WaWV3cG9ydEFjdGl2ZSxcbiAgICBwaWNraW5nRkJPLFxuICAgIGxhc3RQaWNrZWRJbmZvLFxuICAgIHVzZURldmljZVBpeGVsc1xuICB9XG4pIHtcbiAgLy8gQ29udmVydCBmcm9tIGNhbnZhcyB0b3AtbGVmdCB0byBXZWJHTCBib3R0b20tbGVmdCBjb29yZGluYXRlc1xuICAvLyBBbmQgY29tcGVuc2F0ZSBmb3IgcGl4ZWxSYXRpb1xuICBjb25zdCBwaXhlbFJhdGlvID0gZ2V0UGl4ZWxSYXRpbyh7dXNlRGV2aWNlUGl4ZWxzfSk7XG4gIGNvbnN0IGRldmljZVggPSBNYXRoLnJvdW5kKHggKiBwaXhlbFJhdGlvKTtcbiAgY29uc3QgZGV2aWNlWSA9IE1hdGgucm91bmQoZ2wuY2FudmFzLmhlaWdodCAtIHkgKiBwaXhlbFJhdGlvKTtcbiAgY29uc3QgZGV2aWNlUmFkaXVzID0gTWF0aC5yb3VuZChyYWRpdXMgKiBwaXhlbFJhdGlvKTtcblxuICBjb25zdCBkZXZpY2VSZWN0ID0gZ2V0UGlja2luZ1JlY3Qoe1xuICAgIGRldmljZVgsXG4gICAgZGV2aWNlWSxcbiAgICBkZXZpY2VSYWRpdXMsXG4gICAgZGV2aWNlV2lkdGg6IHBpY2tpbmdGQk8ud2lkdGgsXG4gICAgZGV2aWNlSGVpZ2h0OiBwaWNraW5nRkJPLmhlaWdodFxuICB9KTtcblxuICBjb25zdCBwaWNrZWRDb2xvcnMgPVxuICAgIGRldmljZVJlY3QgJiZcbiAgICBkcmF3QW5kU2FtcGxlUGlja2luZ0J1ZmZlcihnbCwge1xuICAgICAgbGF5ZXJzLFxuICAgICAgdmlld3BvcnRzLFxuICAgICAgb25WaWV3cG9ydEFjdGl2ZSxcbiAgICAgIHVzZURldmljZVBpeGVscyxcbiAgICAgIHBpY2tpbmdGQk8sXG4gICAgICBkZXZpY2VSZWN0LFxuICAgICAgbGF5ZXJGaWx0ZXIsXG4gICAgICByZWRyYXdSZWFzb246IG1vZGVcbiAgICB9KTtcblxuICBjb25zdCBwaWNrSW5mbyA9XG4gICAgKHBpY2tlZENvbG9ycyAmJlxuICAgICAgZ2V0Q2xvc2VzdEZyb21QaWNraW5nQnVmZmVyKGdsLCB7XG4gICAgICAgIHBpY2tlZENvbG9ycyxcbiAgICAgICAgbGF5ZXJzLFxuICAgICAgICBkZXZpY2VYLFxuICAgICAgICBkZXZpY2VZLFxuICAgICAgICBkZXZpY2VSYWRpdXMsXG4gICAgICAgIGRldmljZVJlY3RcbiAgICAgIH0pKSB8fFxuICAgIE5PX1BJQ0tFRF9PQkpFQ1Q7XG5cbiAgcmV0dXJuIHByb2Nlc3NQaWNrSW5mbyh7XG4gICAgcGlja0luZm8sXG4gICAgbGFzdFBpY2tlZEluZm8sXG4gICAgbW9kZSxcbiAgICBsYXllcnMsXG4gICAgdmlld3BvcnRzLFxuICAgIHgsXG4gICAgeSxcbiAgICBkZXZpY2VYLFxuICAgIGRldmljZVksXG4gICAgcGl4ZWxSYXRpb1xuICB9KTtcbn1cblxuLy8gUGljayBhbGwgb2JqZWN0cyB3aXRoaW4gdGhlIGdpdmVuIGJvdW5kaW5nIGJveFxuZXhwb3J0IGZ1bmN0aW9uIHBpY2tWaXNpYmxlT2JqZWN0cyhcbiAgZ2wsXG4gIHtcbiAgICBsYXllcnMsXG4gICAgdmlld3BvcnRzLFxuICAgIHgsXG4gICAgeSxcbiAgICB3aWR0aCxcbiAgICBoZWlnaHQsXG4gICAgbW9kZSxcbiAgICBsYXllckZpbHRlcixcbiAgICBvblZpZXdwb3J0QWN0aXZlLFxuICAgIHBpY2tpbmdGQk8sXG4gICAgdXNlRGV2aWNlUGl4ZWxzXG4gIH1cbikge1xuICAvLyBDb252ZXJ0IGZyb20gY2FudmFzIHRvcC1sZWZ0IHRvIFdlYkdMIGJvdHRvbS1sZWZ0IGNvb3JkaW5hdGVzXG4gIC8vIEFuZCBjb21wZW5zYXRlIGZvciBwaXhlbFJhdGlvXG4gIGNvbnN0IHBpeGVsUmF0aW8gPSBnZXRQaXhlbFJhdGlvKHt1c2VEZXZpY2VQaXhlbHN9KTtcblxuICBjb25zdCBkZXZpY2VMZWZ0ID0gTWF0aC5yb3VuZCh4ICogcGl4ZWxSYXRpbyk7XG4gIGNvbnN0IGRldmljZUJvdHRvbSA9IE1hdGgucm91bmQoZ2wuY2FudmFzLmhlaWdodCAtIHkgKiBwaXhlbFJhdGlvKTtcbiAgY29uc3QgZGV2aWNlUmlnaHQgPSBNYXRoLnJvdW5kKCh4ICsgd2lkdGgpICogcGl4ZWxSYXRpbyk7XG4gIGNvbnN0IGRldmljZVRvcCA9IE1hdGgucm91bmQoZ2wuY2FudmFzLmhlaWdodCAtICh5ICsgaGVpZ2h0KSAqIHBpeGVsUmF0aW8pO1xuXG4gIGNvbnN0IGRldmljZVJlY3QgPSB7XG4gICAgeDogZGV2aWNlTGVmdCxcbiAgICB5OiBkZXZpY2VUb3AsXG4gICAgd2lkdGg6IGRldmljZVJpZ2h0IC0gZGV2aWNlTGVmdCxcbiAgICBoZWlnaHQ6IGRldmljZUJvdHRvbSAtIGRldmljZVRvcFxuICB9O1xuXG4gIGNvbnN0IHBpY2tlZENvbG9ycyA9IGRyYXdBbmRTYW1wbGVQaWNraW5nQnVmZmVyKGdsLCB7XG4gICAgbGF5ZXJzLFxuICAgIHZpZXdwb3J0cyxcbiAgICBvblZpZXdwb3J0QWN0aXZlLFxuICAgIHBpY2tpbmdGQk8sXG4gICAgdXNlRGV2aWNlUGl4ZWxzLFxuICAgIGRldmljZVJlY3QsXG4gICAgbGF5ZXJGaWx0ZXIsXG4gICAgcmVkcmF3UmVhc29uOiBtb2RlXG4gIH0pO1xuXG4gIGNvbnN0IHBpY2tJbmZvcyA9IGdldFVuaXF1ZXNGcm9tUGlja2luZ0J1ZmZlcihnbCwge3BpY2tlZENvbG9ycywgbGF5ZXJzfSk7XG5cbiAgLy8gT25seSByZXR1cm4gdW5pcXVlIGluZm9zLCBpZGVudGlmaWVkIGJ5IGluZm8ub2JqZWN0XG4gIGNvbnN0IHVuaXF1ZUluZm9zID0gbmV3IE1hcCgpO1xuXG4gIHBpY2tJbmZvcy5mb3JFYWNoKHBpY2tJbmZvID0+IHtcbiAgICBsZXQgaW5mbyA9IHtcbiAgICAgIGNvbG9yOiBwaWNrSW5mby5waWNrZWRDb2xvcixcbiAgICAgIGxheWVyOiBudWxsLFxuICAgICAgaW5kZXg6IHBpY2tJbmZvLnBpY2tlZE9iamVjdEluZGV4LFxuICAgICAgcGlja2VkOiB0cnVlLFxuICAgICAgeCxcbiAgICAgIHksXG4gICAgICB3aWR0aCxcbiAgICAgIGhlaWdodCxcbiAgICAgIHBpeGVsUmF0aW9cbiAgICB9O1xuXG4gICAgaW5mbyA9IGdldExheWVyUGlja2luZ0luZm8oe2xheWVyOiBwaWNrSW5mby5waWNrZWRMYXllciwgaW5mbywgbW9kZX0pO1xuICAgIGlmICghdW5pcXVlSW5mb3MuaGFzKGluZm8ub2JqZWN0KSkge1xuICAgICAgdW5pcXVlSW5mb3Muc2V0KGluZm8ub2JqZWN0LCBpbmZvKTtcbiAgICB9XG4gIH0pO1xuXG4gIHJldHVybiBBcnJheS5mcm9tKHVuaXF1ZUluZm9zLnZhbHVlcygpKTtcbn1cblxuLy8gSEVMUEVSIE1FVEhPRFNcblxuLy8gcmV0dXJucyBwaWNrZWRDb2xvciBvciBudWxsIGlmIG5vIHBpY2thYmxlIGxheWVycyBmb3VuZC5cbmZ1bmN0aW9uIGRyYXdBbmRTYW1wbGVQaWNraW5nQnVmZmVyKFxuICBnbCxcbiAge1xuICAgIGxheWVycyxcbiAgICB2aWV3cG9ydHMsXG4gICAgb25WaWV3cG9ydEFjdGl2ZSxcbiAgICB1c2VEZXZpY2VQaXhlbHMsXG4gICAgcGlja2luZ0ZCTyxcbiAgICBkZXZpY2VSZWN0LFxuICAgIGxheWVyRmlsdGVyLFxuICAgIHJlZHJhd1JlYXNvblxuICB9XG4pIHtcbiAgYXNzZXJ0KGRldmljZVJlY3QpO1xuICBhc3NlcnQoTnVtYmVyLmlzRmluaXRlKGRldmljZVJlY3Qud2lkdGgpICYmIGRldmljZVJlY3Qud2lkdGggPiAwLCAnYHdpZHRoYCBtdXN0IGJlID4gMCcpO1xuICBhc3NlcnQoTnVtYmVyLmlzRmluaXRlKGRldmljZVJlY3QuaGVpZ2h0KSAmJiBkZXZpY2VSZWN0LmhlaWdodCA+IDAsICdgaGVpZ2h0YCBtdXN0IGJlID4gMCcpO1xuXG4gIGNvbnN0IHBpY2thYmxlTGF5ZXJzID0gbGF5ZXJzLmZpbHRlcihsYXllciA9PiBsYXllci5pc1BpY2thYmxlKCkpO1xuICBpZiAocGlja2FibGVMYXllcnMubGVuZ3RoIDwgMSkge1xuICAgIHJldHVybiBudWxsO1xuICB9XG5cbiAgZHJhd1BpY2tpbmdCdWZmZXIoZ2wsIHtcbiAgICBsYXllcnMsXG4gICAgdmlld3BvcnRzLFxuICAgIG9uVmlld3BvcnRBY3RpdmUsXG4gICAgdXNlRGV2aWNlUGl4ZWxzLFxuICAgIHBpY2tpbmdGQk8sXG4gICAgZGV2aWNlUmVjdCxcbiAgICBsYXllckZpbHRlcixcbiAgICByZWRyYXdSZWFzb25cbiAgfSk7XG5cbiAgLy8gUmVhZCBmcm9tIGFuIGFscmVhZHkgcmVuZGVyZWQgcGlja2luZyBidWZmZXJcbiAgLy8gUmV0dXJucyBhbiBVaW50OENsYW1wZWRBcnJheSBvZiBwaWNrZWQgcGl4ZWxzXG4gIGNvbnN0IHt4LCB5LCB3aWR0aCwgaGVpZ2h0fSA9IGRldmljZVJlY3Q7XG4gIGNvbnN0IHBpY2tlZENvbG9ycyA9IG5ldyBVaW50OEFycmF5KHdpZHRoICogaGVpZ2h0ICogNCk7XG4gIHBpY2tpbmdGQk8ucmVhZFBpeGVscyh7eCwgeSwgd2lkdGgsIGhlaWdodCwgcGl4ZWxBcnJheTogcGlja2VkQ29sb3JzfSk7XG4gIHJldHVybiBwaWNrZWRDb2xvcnM7XG59XG5cbi8vIEluZGVudGlmaWVzIHdoaWNoIHZpZXdwb3J0LCBpZiBhbnkgY29ycmVzcG9uZHMgdG8geCBhbmQgeVxuLy8gUmV0dXJucyBmaXJzdCB2aWV3cG9ydCBpZiBubyBtYXRjaFxuLy8gVE9ETyAtIG5lZWQgdG8gZGV0ZXJtaW5lIHdoaWNoIHZpZXdwb3J0IHdlIGFyZSBpblxuLy8gVE9ETyAtIGRvY3VtZW50IGNvbmNlcHQgb2YgXCJwcmltYXJ5IHZpZXdwb3J0XCIgdGhhdCBtYXRjaGVzIGFsbCBjb29yZHM/XG4vLyBUT0RPIC0gc3RhdGljIG1ldGhvZCBvbiBWaWV3cG9ydCBjbGFzcz9cbmZ1bmN0aW9uIGdldFZpZXdwb3J0RnJvbUNvb3JkaW5hdGVzKHt2aWV3cG9ydHN9KSB7XG4gIGNvbnN0IHZpZXdwb3J0ID0gdmlld3BvcnRzWzBdO1xuICByZXR1cm4gdmlld3BvcnQ7XG59XG5cbi8vIENhbGN1bGF0ZSBhIHBpY2tpbmcgcmVjdCBjZW50ZXJlZCBvbiBkZXZpY2VYIGFuZCBkZXZpY2VZIGFuZCBjbGlwcGVkIHRvIGRldmljZVxuLy8gUmV0dXJucyBudWxsIGlmIHBpeGVsIGlzIG91dHNpZGUgb2YgZGV2aWNlXG5mdW5jdGlvbiBnZXRQaWNraW5nUmVjdCh7ZGV2aWNlWCwgZGV2aWNlWSwgZGV2aWNlUmFkaXVzLCBkZXZpY2VXaWR0aCwgZGV2aWNlSGVpZ2h0fSkge1xuICBjb25zdCB2YWxpZCA9IGRldmljZVggPj0gMCAmJiBkZXZpY2VZID49IDAgJiYgZGV2aWNlWCA8IGRldmljZVdpZHRoICYmIGRldmljZVkgPCBkZXZpY2VIZWlnaHQ7XG5cbiAgLy8geCwgeSBvdXQgb2YgYm91bmRzLlxuICBpZiAoIXZhbGlkKSB7XG4gICAgcmV0dXJuIG51bGw7XG4gIH1cblxuICAvLyBDcmVhdGUgYSBib3ggb2Ygc2l6ZSBgcmFkaXVzICogMiArIDFgIGNlbnRlcmVkIGF0IFtkZXZpY2VYLCBkZXZpY2VZXVxuICBjb25zdCB4ID0gTWF0aC5tYXgoMCwgZGV2aWNlWCAtIGRldmljZVJhZGl1cyk7XG4gIGNvbnN0IHkgPSBNYXRoLm1heCgwLCBkZXZpY2VZIC0gZGV2aWNlUmFkaXVzKTtcbiAgY29uc3Qgd2lkdGggPSBNYXRoLm1pbihkZXZpY2VXaWR0aCwgZGV2aWNlWCArIGRldmljZVJhZGl1cykgLSB4ICsgMTtcbiAgY29uc3QgaGVpZ2h0ID0gTWF0aC5taW4oZGV2aWNlSGVpZ2h0LCBkZXZpY2VZICsgZGV2aWNlUmFkaXVzKSAtIHkgKyAxO1xuXG4gIHJldHVybiB7eCwgeSwgd2lkdGgsIGhlaWdodH07XG59XG5cbi8vIFRPRE8gLSBicmVhayB0aGlzIG1vbnN0ZXIgZnVuY3Rpb24gaW50byAzKyBwYXJ0c1xuZnVuY3Rpb24gcHJvY2Vzc1BpY2tJbmZvKHtcbiAgcGlja0luZm8sXG4gIGxhc3RQaWNrZWRJbmZvLFxuICBtb2RlLFxuICBsYXllcnMsXG4gIHZpZXdwb3J0cyxcbiAgeCxcbiAgeSxcbiAgZGV2aWNlWCxcbiAgZGV2aWNlWSxcbiAgcGl4ZWxSYXRpb1xufSkge1xuICBjb25zdCB7cGlja2VkQ29sb3IsIHBpY2tlZExheWVyLCBwaWNrZWRPYmplY3RJbmRleH0gPSBwaWNrSW5mbztcblxuICBjb25zdCBhZmZlY3RlZExheWVycyA9IHBpY2tlZExheWVyID8gW3BpY2tlZExheWVyXSA6IFtdO1xuXG4gIGlmIChtb2RlID09PSAnaG92ZXInKSB7XG4gICAgLy8gb25seSBpbnZva2Ugb25Ib3ZlciBldmVudHMgaWYgcGlja2VkIG9iamVjdCBoYXMgY2hhbmdlZFxuICAgIGNvbnN0IGxhc3RQaWNrZWRPYmplY3RJbmRleCA9IGxhc3RQaWNrZWRJbmZvLmluZGV4O1xuICAgIGNvbnN0IGxhc3RQaWNrZWRMYXllcklkID0gbGFzdFBpY2tlZEluZm8ubGF5ZXJJZDtcbiAgICBjb25zdCBwaWNrZWRMYXllcklkID0gcGlja2VkTGF5ZXIgJiYgcGlja2VkTGF5ZXIucHJvcHMuaWQ7XG5cbiAgICAvLyBwcm9jZWVkIG9ubHkgaWYgcGlja2VkIG9iamVjdCBjaGFuZ2VkXG4gICAgaWYgKHBpY2tlZExheWVySWQgIT09IGxhc3RQaWNrZWRMYXllcklkIHx8IHBpY2tlZE9iamVjdEluZGV4ICE9PSBsYXN0UGlja2VkT2JqZWN0SW5kZXgpIHtcbiAgICAgIGlmIChwaWNrZWRMYXllcklkICE9PSBsYXN0UGlja2VkTGF5ZXJJZCkge1xuICAgICAgICAvLyBXZSBjYW5ub3Qgc3RvcmUgYSByZWYgdG8gbGFzdFBpY2tlZExheWVyIGluIHRoZSBjb250ZXh0IGJlY2F1c2VcbiAgICAgICAgLy8gdGhlIHN0YXRlIG9mIGFuIG91dGRhdGVkIGxheWVyIGlzIG5vIGxvbmdlciB2YWxpZFxuICAgICAgICAvLyBhbmQgdGhlIHByb3BzIG1heSBoYXZlIGNoYW5nZWRcbiAgICAgICAgY29uc3QgbGFzdFBpY2tlZExheWVyID0gbGF5ZXJzLmZpbmQobGF5ZXIgPT4gbGF5ZXIucHJvcHMuaWQgPT09IGxhc3RQaWNrZWRMYXllcklkKTtcbiAgICAgICAgaWYgKGxhc3RQaWNrZWRMYXllcikge1xuICAgICAgICAgIC8vIExldCBsZWF2ZSBldmVudCBmaXJlIGJlZm9yZSBlbnRlciBldmVudFxuICAgICAgICAgIGFmZmVjdGVkTGF5ZXJzLnVuc2hpZnQobGFzdFBpY2tlZExheWVyKTtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICAvLyBVcGRhdGUgbGF5ZXIgbWFuYWdlciBjb250ZXh0XG4gICAgICBsYXN0UGlja2VkSW5mby5sYXllcklkID0gcGlja2VkTGF5ZXJJZDtcbiAgICAgIGxhc3RQaWNrZWRJbmZvLmluZGV4ID0gcGlja2VkT2JqZWN0SW5kZXg7XG4gICAgfVxuICB9XG5cbiAgY29uc3Qgdmlld3BvcnQgPSBnZXRWaWV3cG9ydEZyb21Db29yZGluYXRlcyh7dmlld3BvcnRzfSk7IC8vIFRPRE8gLSBhZGQgY29vcmRzXG5cbiAgY29uc3QgYmFzZUluZm8gPSB7XG4gICAgY29sb3I6IG51bGwsXG4gICAgbGF5ZXI6IG51bGwsXG4gICAgaW5kZXg6IC0xLFxuICAgIHBpY2tlZDogZmFsc2UsXG4gICAgeCxcbiAgICB5LFxuICAgIHBpeGVsOiBbeCwgeV0sXG4gICAgbG5nTGF0OiB2aWV3cG9ydC51bnByb2plY3QoW3gsIHldKSxcbiAgICBkZXZpY2VQaXhlbDogW2RldmljZVgsIGRldmljZVldLFxuICAgIHBpeGVsUmF0aW9cbiAgfTtcblxuICAvLyBVc2UgYSBNYXAgdG8gc3RvcmUgYWxsIHBpY2tpbmcgaW5mb3MuXG4gIC8vIFRoZSBmb2xsb3dpbmcgdHdvIGZvckVhY2ggbG9vcHMgYXJlIHRoZSByZXN1bHQgb2ZcbiAgLy8gaHR0cHM6Ly9naXRodWIuY29tL3ViZXIvZGVjay5nbC9pc3N1ZXMvNDQzXG4gIC8vIFBsZWFzZSBiZSB2ZXJ5IGNhcmVmdWwgd2hlbiBjaGFuZ2luZyB0aGlzIHBhdHRlcm5cbiAgY29uc3QgaW5mb3MgPSBuZXcgTWFwKCk7XG5cbiAgYWZmZWN0ZWRMYXllcnMuZm9yRWFjaChsYXllciA9PiB7XG4gICAgbGV0IGluZm8gPSBPYmplY3QuYXNzaWduKHt9LCBiYXNlSW5mbyk7XG5cbiAgICBpZiAobGF5ZXIgPT09IHBpY2tlZExheWVyKSB7XG4gICAgICBpbmZvLmNvbG9yID0gcGlja2VkQ29sb3I7XG4gICAgICBpbmZvLmluZGV4ID0gcGlja2VkT2JqZWN0SW5kZXg7XG4gICAgICBpbmZvLnBpY2tlZCA9IHRydWU7XG4gICAgfVxuXG4gICAgaW5mbyA9IGdldExheWVyUGlja2luZ0luZm8oe2xheWVyLCBpbmZvLCBtb2RlfSk7XG5cbiAgICAvLyBUaGlzIGd1YXJhbnRlZXMgdGhhdCB0aGVyZSB3aWxsIGJlIG9ubHkgb25lIGNvcHkgb2YgaW5mbyBmb3JcbiAgICAvLyBvbmUgY29tcG9zaXRlIGxheWVyXG4gICAgaWYgKGluZm8pIHtcbiAgICAgIGluZm9zLnNldChpbmZvLmxheWVyLmlkLCBpbmZvKTtcbiAgICB9XG5cbiAgICBjb25zdCBwaWNraW5nU2VsZWN0ZWRDb2xvciA9XG4gICAgICBsYXllci5wcm9wcy5hdXRvSGlnaGxpZ2h0ICYmIHBpY2tlZExheWVyID09PSBsYXllciA/IHBpY2tlZENvbG9yIDogbnVsbDtcblxuICAgIGNvbnN0IHBpY2tpbmdQYXJhbWV0ZXJzID0ge1xuICAgICAgcGlja2luZ1NlbGVjdGVkQ29sb3JcbiAgICB9O1xuXG4gICAgZm9yIChjb25zdCBtb2RlbCBvZiBsYXllci5nZXRNb2RlbHMoKSkge1xuICAgICAgbW9kZWwudXBkYXRlTW9kdWxlU2V0dGluZ3MocGlja2luZ1BhcmFtZXRlcnMpO1xuICAgIH1cbiAgfSk7XG5cbiAgY29uc3QgdW5oYW5kbGVkUGlja0luZm9zID0gY2FsbExheWVyUGlja2luZ0NhbGxiYWNrcyhpbmZvcywgbW9kZSk7XG5cbiAgcmV0dXJuIHVuaGFuZGxlZFBpY2tJbmZvcztcbn1cblxuLy8gUGVyLWxheWVyIGV2ZW50IGhhbmRsZXJzIChlLmcuIG9uQ2xpY2ssIG9uSG92ZXIpIGFyZSBwcm92aWRlZCBieSB0aGVcbi8vIHVzZXIgYW5kIG91dCBvZiBkZWNrLmdsJ3MgY29udHJvbC4gSXQncyB2ZXJ5IG11Y2ggcG9zc2libGUgdGhhdFxuLy8gdGhlIHVzZXIgY2FsbHMgUmVhY3QgbGlmZWN5Y2xlIG1ldGhvZHMgaW4gdGhlc2UgZnVuY3Rpb24sIHN1Y2ggYXNcbi8vIFJlYWN0Q29tcG9uZW50LnNldFN0YXRlKCkuIFJlYWN0IGxpZmVjeWNsZSBtZXRob2RzIHNvbWV0aW1lcyBpbmR1Y2Vcbi8vIGEgcmUtcmVuZGVyIGFuZCByZS1nZW5lcmF0aW9uIG9mIHByb3BzIG9mIGRlY2suZ2wgYW5kIGl0cyBsYXllcnMsXG4vLyB3aGljaCBpbnZhbGlkYXRlcyBhbGwgbGF5ZXJzIGN1cnJlbnRseSBwYXNzZWQgdG8gdGhpcyB2ZXJ5IGZ1bmN0aW9uLlxuXG4vLyBUaGVyZWZvcmUsIHBlci1sYXllciBldmVudCBoYW5kbGVycyBtdXN0IGJlIGludm9rZWQgYXQgdGhlIGVuZFxuLy8gb2YgdGhlIHBpY2tpbmcgb3BlcmF0aW9uLiBOTyBvcGVyYXRpb24gdGhhdCByZWxpZXMgb24gdGhlIHN0YXRlcyBvZiBjdXJyZW50XG4vLyBsYXllcnMgc2hvdWxkIGJlIGNhbGxlZCBhZnRlciB0aGlzIGNvZGUuXG5mdW5jdGlvbiBjYWxsTGF5ZXJQaWNraW5nQ2FsbGJhY2tzKGluZm9zLCBtb2RlKSB7XG4gIGNvbnN0IHVuaGFuZGxlZFBpY2tJbmZvcyA9IFtdO1xuXG4gIGluZm9zLmZvckVhY2goaW5mbyA9PiB7XG4gICAgbGV0IGhhbmRsZWQgPSBmYWxzZTtcbiAgICBzd2l0Y2ggKG1vZGUpIHtcbiAgICAgIGNhc2UgJ2NsaWNrJzpcbiAgICAgICAgaGFuZGxlZCA9IGluZm8ubGF5ZXIucHJvcHMub25DbGljayhpbmZvKTtcbiAgICAgICAgYnJlYWs7XG4gICAgICBjYXNlICdob3Zlcic6XG4gICAgICAgIGhhbmRsZWQgPSBpbmZvLmxheWVyLnByb3BzLm9uSG92ZXIoaW5mbyk7XG4gICAgICAgIGJyZWFrO1xuICAgICAgY2FzZSAncXVlcnknOlxuICAgICAgICBicmVhaztcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcigndW5rbm93biBwaWNrIHR5cGUnKTtcbiAgICB9XG5cbiAgICBpZiAoIWhhbmRsZWQpIHtcbiAgICAgIHVuaGFuZGxlZFBpY2tJbmZvcy5wdXNoKGluZm8pO1xuICAgIH1cbiAgfSk7XG5cbiAgcmV0dXJuIHVuaGFuZGxlZFBpY2tJbmZvcztcbn1cblxuLyoqXG4gKiBQaWNrIGF0IGEgc3BlY2lmaWVkIHBpeGVsIHdpdGggYSB0b2xlcmFuY2UgcmFkaXVzXG4gKiBSZXR1cm5zIHRoZSBjbG9zZXN0IG9iamVjdCB0byB0aGUgcGl4ZWwgaW4gc2hhcGUgYHtwaWNrZWRDb2xvciwgcGlja2VkTGF5ZXIsIHBpY2tlZE9iamVjdEluZGV4fWBcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldENsb3Nlc3RGcm9tUGlja2luZ0J1ZmZlcihcbiAgZ2wsXG4gIHtwaWNrZWRDb2xvcnMsIGxheWVycywgZGV2aWNlWCwgZGV2aWNlWSwgZGV2aWNlUmFkaXVzLCBkZXZpY2VSZWN0fVxuKSB7XG4gIGFzc2VydChwaWNrZWRDb2xvcnMpO1xuXG4gIC8vIFRyYXZlcnNlIGFsbCBwaXhlbHMgaW4gcGlja2luZyByZXN1bHRzIGFuZCBmaW5kIHRoZSBvbmUgY2xvc2VzdCB0byB0aGUgc3VwcGxpZWRcbiAgLy8gW2RldmljZVgsIGRldmljZVldXG4gIGNvbnN0IHt4LCB5LCB3aWR0aCwgaGVpZ2h0fSA9IGRldmljZVJlY3Q7XG4gIGxldCBtaW5TcXVhcmVEaXN0YW5jZVRvQ2VudGVyID0gZGV2aWNlUmFkaXVzICogZGV2aWNlUmFkaXVzO1xuICBsZXQgY2xvc2VzdFBpeGVsSW5kZXggPSAtMTtcbiAgbGV0IGkgPSAwO1xuXG4gIGZvciAobGV0IHJvdyA9IDA7IHJvdyA8IGhlaWdodDsgcm93KyspIHtcbiAgICBjb25zdCBkeSA9IHJvdyArIHkgLSBkZXZpY2VZO1xuICAgIGNvbnN0IGR5MiA9IGR5ICogZHk7XG5cbiAgICBpZiAoZHkyID4gbWluU3F1YXJlRGlzdGFuY2VUb0NlbnRlcikge1xuICAgICAgLy8gc2tpcCB0aGlzIHJvd1xuICAgICAgaSArPSA0ICogd2lkdGg7XG4gICAgfSBlbHNlIHtcbiAgICAgIGZvciAobGV0IGNvbCA9IDA7IGNvbCA8IHdpZHRoOyBjb2wrKykge1xuICAgICAgICAvLyBEZWNvZGUgcGlja2VkIGxheWVyIGZyb20gY29sb3JcbiAgICAgICAgY29uc3QgcGlja2VkTGF5ZXJJbmRleCA9IHBpY2tlZENvbG9yc1tpICsgM10gLSAxO1xuXG4gICAgICAgIGlmIChwaWNrZWRMYXllckluZGV4ID49IDApIHtcbiAgICAgICAgICBjb25zdCBkeCA9IGNvbCArIHggLSBkZXZpY2VYO1xuICAgICAgICAgIGNvbnN0IGQyID0gZHggKiBkeCArIGR5MjtcblxuICAgICAgICAgIGlmIChkMiA8PSBtaW5TcXVhcmVEaXN0YW5jZVRvQ2VudGVyKSB7XG4gICAgICAgICAgICBtaW5TcXVhcmVEaXN0YW5jZVRvQ2VudGVyID0gZDI7XG4gICAgICAgICAgICBjbG9zZXN0UGl4ZWxJbmRleCA9IGk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGkgKz0gNDtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBpZiAoY2xvc2VzdFBpeGVsSW5kZXggPj0gMCkge1xuICAgIC8vIERlY29kZSBwaWNrZWQgb2JqZWN0IGluZGV4IGZyb20gY29sb3JcbiAgICBjb25zdCBwaWNrZWRMYXllckluZGV4ID0gcGlja2VkQ29sb3JzW2Nsb3Nlc3RQaXhlbEluZGV4ICsgM10gLSAxO1xuICAgIGNvbnN0IHBpY2tlZENvbG9yID0gcGlja2VkQ29sb3JzLnNsaWNlKGNsb3Nlc3RQaXhlbEluZGV4LCBjbG9zZXN0UGl4ZWxJbmRleCArIDQpO1xuICAgIGNvbnN0IHBpY2tlZExheWVyID0gbGF5ZXJzW3BpY2tlZExheWVySW5kZXhdO1xuICAgIGlmIChwaWNrZWRMYXllcikge1xuICAgICAgY29uc3QgcGlja2VkT2JqZWN0SW5kZXggPSBwaWNrZWRMYXllci5kZWNvZGVQaWNraW5nQ29sb3IocGlja2VkQ29sb3IpO1xuICAgICAgcmV0dXJuIHtwaWNrZWRDb2xvciwgcGlja2VkTGF5ZXIsIHBpY2tlZE9iamVjdEluZGV4fTtcbiAgICB9XG4gICAgbG9nLmVycm9yKCdQaWNrZWQgbm9uLWV4aXN0ZW50IGxheWVyLiBJcyBwaWNraW5nIGJ1ZmZlciBjb3JydXB0PycpO1xuICB9XG5cbiAgcmV0dXJuIE5PX1BJQ0tFRF9PQkpFQ1Q7XG59XG4vKiBlc2xpbnQtZW5hYmxlIG1heC1kZXB0aCwgbWF4LXN0YXRlbWVudHMgKi9cblxuLyoqXG4gKiBFeGFtaW5lcyBhIHBpY2tpbmcgYnVmZmVyIGZvciB1bmlxdWUgY29sb3JzXG4gKiBSZXR1cm5zIGFycmF5IG9mIHVuaXF1ZSBvYmplY3RzIGluIHNoYXBlIGB7eCwgeSwgcGlja2VkQ29sb3IsIHBpY2tlZExheWVyLCBwaWNrZWRPYmplY3RJbmRleH1gXG4gKi9cbmZ1bmN0aW9uIGdldFVuaXF1ZXNGcm9tUGlja2luZ0J1ZmZlcihnbCwge3BpY2tlZENvbG9ycywgbGF5ZXJzfSkge1xuICBjb25zdCB1bmlxdWVDb2xvcnMgPSBuZXcgTWFwKCk7XG5cbiAgLy8gVHJhdmVyc2UgYWxsIHBpeGVscyBpbiBwaWNraW5nIHJlc3VsdHMgYW5kIGdldCB1bmlxdWUgY29sb3JzXG4gIGlmIChwaWNrZWRDb2xvcnMpIHtcbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IHBpY2tlZENvbG9ycy5sZW5ndGg7IGkgKz0gNCkge1xuICAgICAgLy8gRGVjb2RlIHBpY2tlZCBsYXllciBmcm9tIGNvbG9yXG4gICAgICBjb25zdCBwaWNrZWRMYXllckluZGV4ID0gcGlja2VkQ29sb3JzW2kgKyAzXSAtIDE7XG5cbiAgICAgIGlmIChwaWNrZWRMYXllckluZGV4ID49IDApIHtcbiAgICAgICAgY29uc3QgcGlja2VkQ29sb3IgPSBwaWNrZWRDb2xvcnMuc2xpY2UoaSwgaSArIDQpO1xuICAgICAgICBjb25zdCBjb2xvcktleSA9IHBpY2tlZENvbG9yLmpvaW4oJywnKTtcbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lXG4gICAgICAgIGlmICghdW5pcXVlQ29sb3JzLmhhcyhjb2xvcktleSkpIHtcbiAgICAgICAgICBjb25zdCBwaWNrZWRMYXllciA9IGxheWVyc1twaWNrZWRMYXllckluZGV4XTtcbiAgICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbiAgICAgICAgICBpZiAocGlja2VkTGF5ZXIpIHtcbiAgICAgICAgICAgIHVuaXF1ZUNvbG9ycy5zZXQoY29sb3JLZXksIHtcbiAgICAgICAgICAgICAgcGlja2VkQ29sb3IsXG4gICAgICAgICAgICAgIHBpY2tlZExheWVyLFxuICAgICAgICAgICAgICBwaWNrZWRPYmplY3RJbmRleDogcGlja2VkTGF5ZXIuZGVjb2RlUGlja2luZ0NvbG9yKHBpY2tlZENvbG9yKVxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGxvZy5lcnJvcignUGlja2VkIG5vbi1leGlzdGVudCBsYXllci4gSXMgcGlja2luZyBidWZmZXIgY29ycnVwdD8nKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICByZXR1cm4gQXJyYXkuZnJvbSh1bmlxdWVDb2xvcnMudmFsdWVzKCkpO1xufVxuXG4vLyBXYWxrIHVwIHRoZSBsYXllciBjb21wb3NpdGUgY2hhaW4gdG8gcG9wdWxhdGUgdGhlIGluZm8gb2JqZWN0XG5mdW5jdGlvbiBnZXRMYXllclBpY2tpbmdJbmZvKHtsYXllciwgaW5mbywgbW9kZX0pIHtcbiAgd2hpbGUgKGxheWVyICYmIGluZm8pIHtcbiAgICAvLyBGb3IgYSBjb21wb3NpdGUgbGF5ZXIsIHNvdXJjZUxheWVyIHdpbGwgcG9pbnQgdG8gdGhlIHN1YmxheWVyXG4gICAgLy8gd2hlcmUgdGhlIGV2ZW50IG9yaWdpbmF0ZXMgZnJvbS5cbiAgICAvLyBJdCBwcm92aWRlcyBhZGRpdGlvbmFsIGNvbnRleHQgZm9yIHRoZSBjb21wb3NpdGUgbGF5ZXInc1xuICAgIC8vIGdldFBpY2tpbmdJbmZvKCkgbWV0aG9kIHRvIHBvcHVsYXRlIHRoZSBpbmZvIG9iamVjdFxuICAgIGNvbnN0IHNvdXJjZUxheWVyID0gaW5mby5sYXllciB8fCBsYXllcjtcbiAgICBpbmZvLmxheWVyID0gbGF5ZXI7XG4gICAgLy8gbGF5ZXIucGlja0xheWVyKCkgZnVuY3Rpb24gcmVxdWlyZXMgYSBub24tbnVsbCBgYGBsYXllci5zdGF0ZWBgYFxuICAgIC8vIG9iamVjdCB0byBmdW50aW9uIHByb3Blcmx5LiBTbyB0aGUgbGF5ZXIgcmVmZXJlY2VkIGhlcmVcbiAgICAvLyBtdXN0IGJlIHRoZSBcImN1cnJlbnRcIiBsYXllciwgbm90IGFuIFwib3V0LWRhdGVkXCIgLyBcImludmFsaWRhdGVkXCIgbGF5ZXJcbiAgICBpbmZvID0gbGF5ZXIucGlja0xheWVyKHtpbmZvLCBtb2RlLCBzb3VyY2VMYXllcn0pO1xuICAgIGxheWVyID0gbGF5ZXIucGFyZW50TGF5ZXI7XG4gIH1cbiAgcmV0dXJuIGluZm87XG59XG4iXX0=