"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.colorDelta = colorDelta;
exports.colorDeltaSq = colorDeltaSq;
exports.diffImagePixels = diffImagePixels;
/*
 * Calculate perceived color difference using YIQ NTSC transmission color space
 * Based on 2010 paper by Yuriy Kotsarenko and Fernando Ramos
 * http://www.progmat.uaem.mx:8080/artVol2Num2/Articulo3Vol2Num2.pdf
 */
var DEFAULT_THRESHOLD = 255 * 0.05;

var getY = function getY(r, g, b) {
  return r * 0.29889531 + g * 0.58662247 + b * 0.11448223;
};
var getI = function getI(r, g, b) {
  return r * 0.59597799 - g * 0.2741761 - b * 0.32180189;
};
var getQ = function getQ(r, g, b) {
  return r * 0.21147017 - g * 0.52261711 + b * 0.31114694;
};
var getESq = function getESq(dY, dI, dQ) {
  return 0.5053 * dY * dY + 0.299 * dI * dI + 0.1957 * dQ * dQ;
};

// Get blended r/g/b value after applying alpha
var applyAlpha = function applyAlpha(c, a) {
  return 255 + (c - 255) * a / 255;
};

/**
 * Get dE square at given index from two pixel arrays
 * @param {Uint8ClampedArray} img1 - pixel data of first image
 * @param {Uint8ClampedArray} img2 - pixel data of second image
 * @param {Number} i - pixel index
 */
function colorDelta(img1, img2, index) {
  return Math.sqrt(colorDeltaSq(img1, img2, index));
}

function colorDeltaSq(img1, img2, index) {
  var i = index * 4;
  var a1 = img1[i + 3];
  var a2 = img2[i + 3];

  var r1 = applyAlpha(img1[i + 0], a1);
  var g1 = applyAlpha(img1[i + 1], a1);
  var b1 = applyAlpha(img1[i + 2], a1);

  var r2 = applyAlpha(img2[i + 0], a2);
  var g2 = applyAlpha(img2[i + 1], a2);
  var b2 = applyAlpha(img2[i + 2], a2);

  return getESq(getY(r1, g1, b1) - getY(r2, g2, b2), getI(r1, g1, b1) - getI(r2, g2, b2), getQ(r1, g1, b1) - getQ(r2, g2, b2));
}

// TODO - expects imagedata structs
// may need a helper func to accept different arguments types
function diffImagePixels(data1, data2) {
  var colorDeltaThreshold = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : DEFAULT_THRESHOLD;

  var pixelCount = data1.data.length / 4;
  var maxDeltaSq = colorDeltaThreshold * colorDeltaThreshold;
  var badPixels = 0;
  for (var i = 0; i < pixelCount; i++) {
    var delta = colorDeltaSq(data1.data, data2.data, i);
    if (delta > maxDeltaSq) {
      badPixels++;
    }
  }
  var percentage = 1 - badPixels / pixelCount;
  return percentage;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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