// A browser implementation of the Node.js `fs.readFile` method

import assert from 'assert';

var window = require('global/window');
var File = window.File;

/**
 * File reader function for the browser, intentionally similar
 * to node's fs.readFile API, however returns a Promise rather than
 * callbacks
 *
 * @param {File|Blob} file  HTML File or Blob object to read as string
 * @returns {Promise.string}  Resolves to a string containing file contents
 */
export function readFile(file) {
  return new Promise(function (resolve, reject) {
    try {
      assert(File, 'window.File not defined. Must run under browser.');
      assert(file instanceof File, 'parameter must be a File object');

      var reader = new window.FileReader();

      reader.onerror = function (e) {
        return reject(new Error(getFileErrorMessage(e)));
      };
      reader.onabort = function () {
        return reject(new Error('Read operation was aborted.'));
      };
      reader.onload = function () {
        return resolve(reader.result);
      };

      reader.readAsText(file);
    } catch (error) {
      reject(error);
    }
  });
}

// NOTES ON ERROR HANDLING
//
// Prepared to externalize error message texts
//
// The weird thing about the FileReader API is that the error definitions
// are only available on the error event instance that is passed to the
// handler. Thus we need to create definitions that are avialble outside
// the handler.
//
// https://developer.mozilla.org/en-US/docs/Web/API/FileReader
//
// Side Note: To complicate matters, there are also a DOMError string set on
// filereader object (error property). Not clear how or if these map
// to the event error codes. These strings are not currently used by this api.
//
// https://developer.mozilla.org/en-US/docs/Web/API/DOMError

function getFileErrorMessage(e) {
  // Map event's error codes to static error codes so that we can
  // externalize error code to error message mapping
  switch (e.target.error.code) {
    case e.target.error.NOT_FOUND_ERR:
      return 'File not found';
    case e.target.error.NOT_READABLE_ERR:
      return 'File not readable';
    case e.target.error.ABORT_ERR:
      return 'Aborted';
    case e.target.error.SECURITY_ERR:
      return 'File locked';
    case e.target.error.ENCODING_ERR:
      return 'File too long';
    default:
      return 'Read error';
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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