"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

exports.default = "\n#define SHADER_NAME path-layer-vertex-shader\n\nattribute vec3 positions;\n\nattribute vec3 instanceStartPositions;\nattribute vec3 instanceEndPositions;\nattribute vec4 instanceStartEndPositions64xyLow;\nattribute vec3 instanceLeftDeltas;\nattribute vec3 instanceRightDeltas;\nattribute float instanceStrokeWidths;\nattribute vec4 instanceColors;\nattribute vec3 instancePickingColors;\nattribute vec2 instanceDashArrays;\n\nuniform float widthScale;\nuniform float widthMinPixels;\nuniform float widthMaxPixels;\nuniform float jointType;\nuniform float miterLimit;\n\nuniform float opacity;\n\nvarying vec4 vColor;\nvarying vec2 vCornerOffset;\nvarying float vMiterLength;\nvarying vec2 vDashArray;\nvarying float vPathPosition;\nvarying float vPathLength;\n\nconst float EPSILON = 0.001;\nconst float PIXEL_EPSILON = 0.1;\n\nfloat flipIfTrue(bool flag) {\n  return -(float(flag) * 2. - 1.);\n}\n\nvec3 lineJoin(vec2 prevPoint64[2], vec2 currPoint64[2], vec2 nextPoint64[2]) {\n\n  float width = clamp(project_scale(instanceStrokeWidths * widthScale),\n    widthMinPixels, widthMaxPixels) / 2.0;\n\n  vec2 deltaA64[2];\n  vec2 deltaB64[2];\n\n  vec2_sub_fp64(currPoint64, prevPoint64, deltaA64);\n  vec2_sub_fp64(nextPoint64, currPoint64, deltaB64);\n\n  vec2 lengthA64 = vec2_length_fp64(deltaA64);\n  vec2 lengthB64 = vec2_length_fp64(deltaB64);\n\n  vec2 deltaA = vec2(deltaA64[0].x, deltaA64[1].x);\n  vec2 deltaB = vec2(deltaB64[0].x, deltaB64[1].x);\n\n  float lenA = lengthA64.x;\n  float lenB = lengthB64.x;\n\n  vec2 offsetVec;\n  float offsetScale;\n  float offsetDirection;\n\n  // when two points are closer than PIXEL_EPSILON in pixels,\n  // assume they are the same point to avoid precision issue\n  lenA = lenA > PIXEL_EPSILON ? lenA : 0.0;\n  lenB = lenB > PIXEL_EPSILON ? lenB : 0.0;\n  vec2 dirA = lenA > 0. ? deltaA / lenA : vec2(0.0, 0.0);\n  vec2 dirB = lenB > 0. ? deltaB / lenB : vec2(0.0, 0.0);\n\n  vec2 perpA = vec2(-dirA.y, dirA.x);\n  vec2 perpB = vec2(-dirB.y, dirB.x);\n\n  // tangent of the corner\n  vec2 tangent = vec2(dirA + dirB);\n  tangent = length(tangent) > 0. ? normalize(tangent) : perpA;\n  // direction of the corner\n  vec2 miterVec = vec2(-tangent.y, tangent.x);\n  // width offset from current position\n  vec2 perp = mix(perpB, perpA, positions.x);\n  float L = mix(lenB, lenA, positions.x);\n\n  // cap super sharp angles\n  float sinHalfA = abs(dot(miterVec, perp));\n  float cosHalfA = abs(dot(dirA, miterVec));\n  bool turnsRight = dirA.x * dirB.y > dirA.y * dirB.x;\n\n  // relative position to the corner:\n  // -1: inside (smaller side of the angle)\n  // 0: center\n  // 1: outside (bigger side of the angle)\n  float cornerPosition = mix(\n    flipIfTrue(turnsRight == (positions.y > 0.0)),\n    0.0,\n    positions.z\n  );\n\n  offsetScale = 1.0 / max(sinHalfA, EPSILON);\n\n  // do not bevel if line segment is too short\n  cornerPosition *= float(cornerPosition <= 0.0 || sinHalfA < min(lenA, lenB) / width * cosHalfA);\n  // trim if inside corner extends further than the line segment\n  offsetScale = mix(\n    offsetScale,\n    min(offsetScale, L / width / max(cosHalfA, EPSILON)),\n    float(cornerPosition < 0.0)\n  );\n\n  vMiterLength = mix(\n    offsetScale * cornerPosition,\n    mix(offsetScale, 0.0, cornerPosition),\n    step(0.0, cornerPosition)\n  ) - sinHalfA * jointType;\n  offsetDirection = mix(\n    positions.y,\n    mix(\n      flipIfTrue(turnsRight),\n      positions.y * flipIfTrue(turnsRight == (positions.x == 1.)),\n      cornerPosition\n    ),\n    step(0.0, cornerPosition)\n  );\n  offsetVec = mix(miterVec, -tangent, step(0.5, cornerPosition));\n  offsetScale = mix(offsetScale, 1.0 / max(cosHalfA, 0.001), step(0.5, cornerPosition));\n\n  // special treatment for start cap and end cap\n  // using a small number as the limit for determining if the lenA or lenB is 0\n  float isStartCap = step(lenA, 1.0e-5);\n  float isEndCap = step(lenB, 1.0e-5);\n  float isCap = max(isStartCap, isEndCap);\n\n  // 0: center, 1: side\n  cornerPosition = isCap * (1.0 - positions.z);\n\n  // start of path: use next - curr\n  offsetVec = mix(offsetVec, mix(dirB, perpB, cornerPosition), isStartCap);\n  // end of path: use curr - prev\n  offsetVec = mix(offsetVec, mix(dirA, perpA, cornerPosition), isEndCap);\n\n  // extend out a triangle to envelope the round cap\n  offsetScale = mix(\n    offsetScale,\n    mix(4.0 * jointType, 1.0, cornerPosition),\n    isCap\n  );\n  vMiterLength = mix(vMiterLength, 1.0 - cornerPosition, isCap);\n\n  offsetDirection = mix(\n    offsetDirection,\n    mix(flipIfTrue(isStartCap > 0.), positions.y, cornerPosition),\n    isCap\n  );\n\n  vCornerOffset = offsetVec * offsetDirection * offsetScale;\n\n  // Generate variables for dash calculation\n  vDashArray = instanceDashArrays;\n  vPathLength = L / width;\n  float isEnd = positions.x;\n  vec2 offsetFromStartOfPath = mix(vCornerOffset, vCornerOffset + deltaA / width, isEnd);\n  vec2 dir = mix(dirB, dirA, isEnd);\n  vPathPosition = dot(offsetFromStartOfPath, dir);\n\n  return vec3(vCornerOffset * width, 0.0);\n}\n\nvoid main() {\n  vColor = vec4(instanceColors.rgb, instanceColors.a * opacity) / 255.;\n\n  // Set color to be rendered to picking fbo (also used to check for selection highlight).\n  picking_setPickingColor(instancePickingColors);\n\n  float isEnd = positions.x;\n\n  // Calculate current position\n  vec3 currPosition = mix(instanceStartPositions, instanceEndPositions, isEnd);\n  currPosition = project_position(currPosition);\n\n  // Calculate current position 64bit\n\n  vec2 instanceStartPositions64[2];\n  instanceStartPositions64[0] = vec2(instanceStartPositions.x, instanceStartEndPositions64xyLow.x);\n  instanceStartPositions64[1] = vec2(instanceStartPositions.y, instanceStartEndPositions64xyLow.y);\n\n  vec2 instanceEndPositions64[2];\n  instanceEndPositions64[0] = vec2(instanceEndPositions.x, instanceStartEndPositions64xyLow.z);\n  instanceEndPositions64[1] = vec2(instanceEndPositions.y, instanceStartEndPositions64xyLow.w);\n\n  vec2 tempCurrPosition64[2];\n  vec2_mix_fp64(instanceStartPositions64, instanceEndPositions64, isEnd, tempCurrPosition64);\n\n  vec4 currPosition64 = vec4(tempCurrPosition64[0].xy, tempCurrPosition64[1].xy);\n\n  vec2 projected_curr_position[2];\n  project_position_fp64(currPosition64, projected_curr_position);\n  float projected_curr_position_z = project_scale(currPosition.z);\n\n  // Calculate previous position\n\n  vec3 prevPosition = mix(-instanceLeftDeltas, vec3(0.0), isEnd);\n\n  // Calculate prev position 64bit\n\n  vec2 tempPrevPosition64[2];\n  tempPrevPosition64[0] = sum_fp64(vec2(prevPosition.x, 0.0), instanceStartPositions64[0]);\n  tempPrevPosition64[1] = sum_fp64(vec2(prevPosition.y, 0.0), instanceStartPositions64[1]);\n\n  vec4 prevPosition64 = vec4(tempPrevPosition64[0].xy, tempPrevPosition64[1].xy);\n\n  vec2 projected_prev_position[2];\n  project_position_fp64(prevPosition64, projected_prev_position);\n\n  // Calculate next positions\n  vec3 nextPosition = mix(vec3(0.0), instanceRightDeltas, isEnd);\n\n  // Calculate next position 64bit\n\n  vec2 tempNextPosition64[2];\n  tempNextPosition64[0] = sum_fp64(vec2(nextPosition.x, 0.0), instanceEndPositions64[0]);\n  tempNextPosition64[1] = sum_fp64(vec2(nextPosition.y, 0.0), instanceEndPositions64[1]);\n\n  vec4 nextPosition64 = vec4(tempNextPosition64[0].xy, tempNextPosition64[1].xy);\n\n  vec2 projected_next_position[2];\n  project_position_fp64(nextPosition64, projected_next_position);\n\n  vec3 pos = lineJoin(projected_prev_position, projected_curr_position, projected_next_position);\n  vec2 vertex_pos_modelspace[4];\n\n  vertex_pos_modelspace[0] = sum_fp64(vec2(pos.x, 0.0), projected_curr_position[0]);\n  vertex_pos_modelspace[1] = sum_fp64(vec2(pos.y, 0.0), projected_curr_position[1]);\n  vertex_pos_modelspace[2] = vec2(pos.z + projected_curr_position_z, 0.0);\n  vertex_pos_modelspace[3] = vec2(1.0, 0.0);\n\n  gl_Position = project_to_clipspace_fp64(vertex_pos_modelspace);\n}\n";
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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