'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUniformsFromViewport = getUniformsFromViewport;

var _multiply = require('gl-mat4/multiply');

var _multiply2 = _interopRequireDefault(_multiply);

var _transformMat = require('gl-vec4/transformMat4');

var _transformMat2 = _interopRequireDefault(_transformMat);

var _log = require('../../utils/log');

var _log2 = _interopRequireDefault(_log);

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

var _constants = require('../../lib/constants');

var _memoize = require('../../utils/memoize');

var _memoize2 = _interopRequireDefault(_memoize);

var _viewportMercatorProject = require('viewport-mercator-project');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// To quickly set a vector to zero
var ZERO_VECTOR = [0, 0, 0, 0];
// 4x4 matrix that drops 4th component of vector
// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

/* global window */
var VECTOR_TO_POINT_MATRIX = [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0];
var IDENTITY_MATRIX = [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1];
var DEFAULT_PIXELS_PER_UNIT2 = [0, 0, 0];
var DEFAULT_COORDINATE_ORIGIN = [0, 0, 0];

var getMemoizedViewportUniforms = (0, _memoize2.default)(calculateViewportUniforms);

// The code that utilizes Matrix4 does the same calculation as their mat4 counterparts,
// has lower performance but provides error checking.
// Uncomment when debugging
function calculateMatrixAndOffset(_ref) {
  var viewport = _ref.viewport,
      coordinateSystem = _ref.coordinateSystem,
      coordinateOrigin = _ref.coordinateOrigin,
      coordinateZoom = _ref.coordinateZoom;
  var viewMatrixUncentered = viewport.viewMatrixUncentered;
  var viewMatrix = viewport.viewMatrix;
  var projectionMatrix = viewport.projectionMatrix;
  var viewProjectionMatrix = viewport.viewProjectionMatrix;


  var projectionCenter = void 0;

  switch (coordinateSystem) {
    case _constants.COORDINATE_SYSTEM.IDENTITY:
    case _constants.COORDINATE_SYSTEM.LNGLAT:
      projectionCenter = ZERO_VECTOR;
      break;

    // TODO: make lighting work for meter offset mode
    case _constants.COORDINATE_SYSTEM.LNGLAT_OFFSETS:
    case _constants.COORDINATE_SYSTEM.METER_OFFSETS:
      // Calculate transformed projectionCenter (using 64 bit precision JS)
      // This is the key to offset mode precision
      // (avoids doing this addition in 32 bit precision in GLSL)
      var positionPixels = (0, _viewportMercatorProject.lngLatToWorld)(coordinateOrigin, Math.pow(2, coordinateZoom));
      // projectionCenter = new Matrix4(viewProjectionMatrix)
      //   .transformVector([positionPixels[0], positionPixels[1], 0.0, 1.0]);
      projectionCenter = (0, _transformMat2.default)([], [positionPixels[0], positionPixels[1], 0.0, 1.0], viewProjectionMatrix);

      // Always apply uncentered projection matrix if available (shader adds center)
      viewMatrix = viewMatrixUncentered || viewMatrix;

      // Zero out 4th coordinate ("after" model matrix) - avoids further translations
      // viewMatrix = new Matrix4(viewMatrixUncentered || viewMatrix)
      //   .multiplyRight(VECTOR_TO_POINT_MATRIX);
      viewProjectionMatrix = (0, _multiply2.default)([], projectionMatrix, viewMatrix);
      viewProjectionMatrix = (0, _multiply2.default)([], viewProjectionMatrix, VECTOR_TO_POINT_MATRIX);
      break;

    default:
      throw new Error('Unknown projection mode');
  }

  return {
    viewMatrix: viewMatrix,
    viewProjectionMatrix: viewProjectionMatrix,
    projectionCenter: projectionCenter,
    cameraPos: viewport.cameraPosition
  };
}

/**
 * Returns uniforms for shaders based on current projection
 * includes: projection matrix suitable for shaders
 *
 * TODO - Ensure this works with any viewport, not just WebMercatorViewports
 *
 * @param {WebMercatorViewport} viewport -
 * @return {Float32Array} - 4x4 projection matrix that can be used in shaders
 */
function getUniformsFromViewport() {
  var _ref2 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      viewport = _ref2.viewport,
      _ref2$modelMatrix = _ref2.modelMatrix,
      modelMatrix = _ref2$modelMatrix === undefined ? null : _ref2$modelMatrix,
      _ref2$coordinateSyste = _ref2.coordinateSystem,
      coordinateSystem = _ref2$coordinateSyste === undefined ? _constants.COORDINATE_SYSTEM.LNGLAT : _ref2$coordinateSyste,
      _ref2$coordinateOrigi = _ref2.coordinateOrigin,
      coordinateOrigin = _ref2$coordinateOrigi === undefined ? DEFAULT_COORDINATE_ORIGIN : _ref2$coordinateOrigi,
      projectionMode = _ref2.projectionMode,
      positionOrigin = _ref2.positionOrigin;

  (0, _assert2.default)(viewport);

  if (projectionMode !== undefined) {
    _log2.default.removed('projectionMode', 'coordinateSystem');
  }
  if (positionOrigin !== undefined) {
    _log2.default.removed('positionOrigin', 'coordinateOrigin');
  }

  return Object.assign({
    project_uModelMatrix: modelMatrix || IDENTITY_MATRIX
  }, getMemoizedViewportUniforms({ viewport: viewport, coordinateSystem: coordinateSystem, coordinateOrigin: coordinateOrigin }));
}

function calculateViewportUniforms(_ref3) {
  var viewport = _ref3.viewport,
      coordinateSystem = _ref3.coordinateSystem,
      coordinateOrigin = _ref3.coordinateOrigin;

  var coordinateZoom = viewport.zoom;
  (0, _assert2.default)(coordinateZoom >= 0);

  var _calculateMatrixAndOf = calculateMatrixAndOffset({
    coordinateSystem: coordinateSystem,
    coordinateOrigin: coordinateOrigin,
    coordinateZoom: coordinateZoom,
    viewport: viewport
  }),
      projectionCenter = _calculateMatrixAndOf.projectionCenter,
      viewProjectionMatrix = _calculateMatrixAndOf.viewProjectionMatrix,
      cameraPos = _calculateMatrixAndOf.cameraPos;

  (0, _assert2.default)(viewProjectionMatrix, 'Viewport missing modelViewProjectionMatrix');

  // Calculate projection pixels per unit
  var distanceScales = viewport.getDistanceScales();

  // TODO - does this depend on useDevicePixels?
  var devicePixelRatio = typeof window !== 'undefined' && window.devicePixelRatio || 1;
  var viewportSize = [viewport.width * devicePixelRatio, viewport.height * devicePixelRatio];

  var uniforms = {
    // Projection mode values
    project_uCoordinateSystem: coordinateSystem,
    project_uCenter: projectionCenter,

    // Screen size
    project_uViewportSize: viewportSize,
    project_uDevicePixelRatio: devicePixelRatio,

    // Distance at which screen pixels are projected
    project_uFocalDistance: viewport.focalDistance || 1,
    project_uPixelsPerMeter: distanceScales.pixelsPerMeter,
    project_uPixelsPerDegree: distanceScales.pixelsPerDegree,
    project_uPixelsPerUnit: distanceScales.pixelsPerMeter,
    project_uPixelsPerUnit2: DEFAULT_PIXELS_PER_UNIT2,
    project_uScale: viewport.scale, // This is the mercator scale (2 ** zoom)

    project_uViewProjectionMatrix: viewProjectionMatrix,

    // This is for lighting calculations
    project_uCameraPosition: cameraPos
  };

  if (coordinateSystem === _constants.COORDINATE_SYSTEM.METER_OFFSETS) {
    var distanceScalesAtOrigin = viewport.getDistanceScales(coordinateOrigin);
    uniforms.project_uPixelsPerUnit = distanceScalesAtOrigin.pixelsPerMeter;
    uniforms.project_uPixelsPerUnit2 = distanceScalesAtOrigin.pixelsPerMeter2;
  }
  if (coordinateSystem === _constants.COORDINATE_SYSTEM.LNGLAT_OFFSETS) {
    var _distanceScalesAtOrigin = viewport.getDistanceScales(coordinateOrigin);
    uniforms.project_uPixelsPerUnit = _distanceScalesAtOrigin.pixelsPerDegree;
    uniforms.project_uPixelsPerUnit2 = _distanceScalesAtOrigin.pixelsPerDegree2;
  }

  return uniforms;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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