'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.lerp = lerp;
exports.isValid = isValid;
exports.getEndValueByShortestPath = getEndValueByShortestPath;
exports.extractViewportFrom = extractViewportFrom;

var _mathUtils = require('../utils/math-utils');

var VIEWPORT_PROPS = ['width', 'height', 'longitude', 'latitude', 'zoom', 'bearing', 'pitch', 'position'];

var WRAPPED_ANGULAR_PROPS = {
  longitude: 1,
  bearing: 1
};

function lerp(start, end, step) {
  if (Array.isArray(start)) {
    return start.map(function (element, index) {
      return lerp(element, end[index], step);
    });
  }
  return step * end + (1 - step) * start;
}

function isValid(prop) {
  return Number.isFinite(prop) || Array.isArray(prop);
}

function isWrappedAngularProp(propName) {
  return WRAPPED_ANGULAR_PROPS[propName];
}

function getEndValueByShortestPath(propName, startValue, endValue) {
  if (isWrappedAngularProp(propName) && Math.abs(endValue - startValue) > 180) {
    endValue = endValue < 0 ? endValue + 360 : endValue - 360;
  }
  return endValue;
}

// TODO/xiaoji: This should be merged with the controller's prop constraint system
function extractViewportFrom(props) {
  var viewport = {};
  VIEWPORT_PROPS.forEach(function (key) {
    var value = props[key];
    if (isValid(value)) {
      viewport[key] = value;
      // Normalize longitude and bearing into [-180, 180) range
      // This gurantees the props are in same range when they are interpolated.
      if (isWrappedAngularProp(key)) {
        viewport[key] = (0, _mathUtils.mod)(value + 180, 360) - 180;
      }
    }
  });
  return viewport;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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