// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import {COORDINATE_SYSTEM, Layer, experimental} from '../../core';
const {fp64LowPart, enable64bitSupport} = experimental;
import {GL, Model, Geometry} from 'luma.gl';

import vs from './path-layer-vertex.glsl';
import vs64 from './path-layer-vertex-64.glsl';
import fs from './path-layer-fragment.glsl';

const DEFAULT_COLOR = [0, 0, 0, 255];

const defaultProps = {
  widthScale: 1, // stroke width in meters
  widthMinPixels: 0, //  min stroke width in pixels
  widthMaxPixels: Number.MAX_SAFE_INTEGER, // max stroke width in pixels
  rounded: false,
  miterLimit: 4,
  fp64: false,
  dashJustified: false,

  getPath: object => object.path,
  getColor: object => object.color || DEFAULT_COLOR,
  getWidth: object => object.width || 1,
  getDashArray: null
};

const isClosed = path => {
  const firstPoint = path[0];
  const lastPoint = path[path.length - 1];
  return (
    firstPoint[0] === lastPoint[0] &&
    firstPoint[1] === lastPoint[1] &&
    firstPoint[2] === lastPoint[2]
  );
};

export default class PathLayer extends Layer {
  getShaders() {
    return enable64bitSupport(this.props)
      ? {vs: vs64, fs, modules: ['project64', 'picking']}
      : {vs, fs, modules: ['picking']}; // 'project' module added by default.
  }

  initializeState() {
    const attributeManager = this.getAttributeManager();
    /* eslint-disable max-len */
    attributeManager.addInstanced({
      instanceStartPositions: {size: 3, update: this.calculateStartPositions},
      instanceEndPositions: {size: 3, update: this.calculateEndPositions},
      instanceLeftDeltas: {size: 3, update: this.calculateLeftDeltas},
      instanceRightDeltas: {size: 3, update: this.calculateRightDeltas},
      instanceStrokeWidths: {size: 1, accessor: 'getWidth', update: this.calculateStrokeWidths},
      instanceDashArrays: {size: 2, accessor: 'getDashArray', update: this.calculateDashArrays},
      instanceColors: {
        size: 4,
        type: GL.UNSIGNED_BYTE,
        accessor: 'getColor',
        update: this.calculateColors
      },
      instancePickingColors: {size: 3, type: GL.UNSIGNED_BYTE, update: this.calculatePickingColors}
    });
    /* eslint-enable max-len */
  }

  updateAttribute({props, oldProps, changeFlags}) {
    if (props.fp64 !== oldProps.fp64) {
      const attributeManager = this.getAttributeManager();
      attributeManager.invalidateAll();

      if (props.fp64 && props.coordinateSystem === COORDINATE_SYSTEM.LNGLAT) {
        attributeManager.addInstanced({
          instanceStartEndPositions64xyLow: {
            size: 4,
            update: this.calculateInstanceStartEndPositions64xyLow
          }
        });
      } else {
        attributeManager.remove(['instanceStartEndPositions64xyLow']);
      }
    }
  }

  updateState({oldProps, props, changeFlags}) {
    super.updateState({props, oldProps, changeFlags});

    const {getPath} = this.props;
    const attributeManager = this.getAttributeManager();
    if (props.fp64 !== oldProps.fp64) {
      const {gl} = this.context;
      if (this.state.model) {
        this.state.model.delete();
      }
      this.setState({model: this._getModel(gl)});
    }
    this.updateAttribute({props, oldProps, changeFlags});

    const geometryChanged =
      changeFlags.dataChanged ||
      (changeFlags.updateTriggersChanged &&
        (changeFlags.updateTriggersChanged.all || changeFlags.updateTriggersChanged.getPath));

    if (geometryChanged) {
      // this.state.paths only stores point positions in each path
      const paths = props.data.map(getPath);
      const numInstances = paths.reduce((count, path) => count + path.length - 1, 0);

      this.setState({paths, numInstances});
      attributeManager.invalidateAll();
    }
  }

  draw({uniforms}) {
    const {
      rounded,
      miterLimit,
      widthScale,
      widthMinPixels,
      widthMaxPixels,
      dashJustified
    } = this.props;

    this.state.model.render(
      Object.assign({}, uniforms, {
        jointType: Number(rounded),
        alignMode: Number(dashJustified),
        widthScale,
        miterLimit,
        widthMinPixels,
        widthMaxPixels
      })
    );
  }

  _getModel(gl) {
    /*
     *       _
     *        "-_ 1                   3                       5
     *     _     "o---------------------o-------------------_-o
     *       -   / ""--..__              '.             _.-' /
     *   _     "@- - - - - ""--..__- - - - x - - - -_.@'    /
     *    "-_  /                   ""--..__ '.  _,-` :     /
     *       "o----------------------------""-o'    :     /
     *      0,2                            4 / '.  :     /
     *                                      /   '.:     /
     *                                     /     :'.   /
     *                                    /     :  ', /
     *                                   /     :     o
     */

    const SEGMENT_INDICES = [
      // start corner
      0,
      2,
      1,
      // body
      1,
      2,
      4,
      1,
      4,
      3,
      // end corner
      3,
      4,
      5
    ];

    // [0] position on segment - 0: start, 1: end
    // [1] side of path - -1: left, 0: center, 1: right
    // [2] role - 0: offset point 1: joint point
    const SEGMENT_POSITIONS = [
      // bevel start corner
      0,
      0,
      1,
      // start inner corner
      0,
      -1,
      0,
      // start outer corner
      0,
      1,
      0,
      // end inner corner
      1,
      -1,
      0,
      // end outer corner
      1,
      1,
      0,
      // bevel end corner
      1,
      0,
      1
    ];

    return new Model(
      gl,
      Object.assign({}, this.getShaders(), {
        id: this.props.id,
        geometry: new Geometry({
          drawMode: GL.TRIANGLES,
          attributes: {
            indices: new Uint16Array(SEGMENT_INDICES),
            positions: new Float32Array(SEGMENT_POSITIONS)
          }
        }),
        isInstanced: true,
        shaderCache: this.context.shaderCache
      })
    );
  }

  calculateStartPositions(attribute) {
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach(path => {
      const numSegments = path.length - 1;
      for (let ptIndex = 0; ptIndex < numSegments; ptIndex++) {
        const point = path[ptIndex];
        value[i++] = point[0];
        value[i++] = point[1];
        value[i++] = point[2] || 0;
      }
    });
  }

  calculateEndPositions(attribute) {
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach(path => {
      for (let ptIndex = 1; ptIndex < path.length; ptIndex++) {
        const point = path[ptIndex];
        value[i++] = point[0];
        value[i++] = point[1];
        value[i++] = point[2] || 0;
      }
    });
  }

  calculateInstanceStartEndPositions64xyLow(attribute) {
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach(path => {
      const numSegments = path.length - 1;
      for (let ptIndex = 0; ptIndex < numSegments; ptIndex++) {
        const startPoint = path[ptIndex];
        const endPoint = path[ptIndex + 1];
        value[i++] = fp64LowPart(startPoint[0]);
        value[i++] = fp64LowPart(startPoint[1]);
        value[i++] = fp64LowPart(endPoint[0]);
        value[i++] = fp64LowPart(endPoint[1]);
      }
    });
  }

  calculateLeftDeltas(attribute) {
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach(path => {
      const numSegments = path.length - 1;
      let prevPoint = isClosed(path) ? path[path.length - 2] : path[0];

      for (let ptIndex = 0; ptIndex < numSegments; ptIndex++) {
        const point = path[ptIndex];
        value[i++] = point[0] - prevPoint[0];
        value[i++] = point[1] - prevPoint[1];
        value[i++] = point[2] - prevPoint[2] || 0;
        prevPoint = point;
      }
    });
  }

  calculateRightDeltas(attribute) {
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach(path => {
      for (let ptIndex = 1; ptIndex < path.length; ptIndex++) {
        const point = path[ptIndex];
        let nextPoint = path[ptIndex + 1];
        if (!nextPoint) {
          nextPoint = isClosed(path) ? path[1] : point;
        }

        value[i++] = nextPoint[0] - point[0];
        value[i++] = nextPoint[1] - point[1];
        value[i++] = nextPoint[2] - point[2] || 0;
      }
    });
  }

  calculateStrokeWidths(attribute) {
    const {data, getWidth} = this.props;
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach((path, index) => {
      const width = getWidth(data[index], index);
      for (let ptIndex = 1; ptIndex < path.length; ptIndex++) {
        value[i++] = width;
      }
    });
  }

  calculateDashArrays(attribute) {
    const {data, getDashArray} = this.props;
    if (!getDashArray) {
      return;
    }

    const {paths} = this.state;
    const {value} = attribute;
    let i = 0;
    paths.forEach((path, index) => {
      const dashArray = getDashArray(data[index], index);
      for (let ptIndex = 1; ptIndex < path.length; ptIndex++) {
        value[i++] = dashArray[0];
        value[i++] = dashArray[1];
      }
    });
  }

  calculateColors(attribute) {
    const {data, getColor} = this.props;
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach((path, index) => {
      const pointColor = getColor(data[index], index);
      if (isNaN(pointColor[3])) {
        pointColor[3] = 255;
      }
      for (let ptIndex = 1; ptIndex < path.length; ptIndex++) {
        value[i++] = pointColor[0];
        value[i++] = pointColor[1];
        value[i++] = pointColor[2];
        value[i++] = pointColor[3];
      }
    });
  }

  // Override the default picking colors calculation
  calculatePickingColors(attribute) {
    const {paths} = this.state;
    const {value} = attribute;

    let i = 0;
    paths.forEach((path, index) => {
      const pickingColor = this.encodePickingColor(index);
      for (let ptIndex = 1; ptIndex < path.length; ptIndex++) {
        value[i++] = pickingColor[0];
        value[i++] = pickingColor[1];
        value[i++] = pickingColor[2];
      }
    });
  }
}

PathLayer.layerName = 'PathLayer';
PathLayer.defaultProps = defaultProps;
