// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
var R_EARTH = 6378000;

/**
 * Calculate density grid from an array of points
 * @param {array} points
 * @param {number} cellSize - cell size in meters
 * @param {function} getPosition - position accessor
 * @returns {object} - grid data, cell dimension
 */
export function pointToDensityGridData(points, cellSize, getPosition) {
  var _pointsToGridHashing2 = _pointsToGridHashing(points, cellSize, getPosition),
      gridHash = _pointsToGridHashing2.gridHash,
      gridOffset = _pointsToGridHashing2.gridOffset;

  var layerData = _getGridLayerDataFromGridHash(gridHash, gridOffset);

  return {
    gridOffset: gridOffset,
    layerData: layerData
  };
}

/**
 * Project points into each cell, return a hash table of cells
 * @param {array} points
 * @param {number} cellSize - unit size in meters
 * @param {function} getPosition - position accessor
 * @returns {object} - grid hash and cell dimension
 */
function _pointsToGridHashing() {
  var points = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
  var cellSize = arguments[1];
  var getPosition = arguments[2];

  // find the geometric center of sample points
  var latMin = Infinity;
  var latMax = -Infinity;
  var pLat = void 0;

  for (var p = 0; p < points.length; p++) {
    pLat = getPosition(points[p])[1];
    if (Number.isFinite(pLat)) {
      latMin = pLat < latMin ? pLat : latMin;
      latMax = pLat > latMax ? pLat : latMax;
    }
  }

  var centerLat = (latMin + latMax) / 2;

  var gridOffset = _calculateGridLatLonOffset(cellSize, centerLat);

  if (gridOffset.xOffset <= 0 || gridOffset.yOffset <= 0) {
    return { gridHash: {}, gridOffset: gridOffset };
  }
  // calculate count per cell
  var gridHash = points.reduce(function (accu, pt) {
    var lat = getPosition(pt)[1];
    var lng = getPosition(pt)[0];

    if (!Number.isFinite(lat) || !Number.isFinite(lng)) {
      return accu;
    }

    var latIdx = Math.floor((lat + 90) / gridOffset.yOffset);
    var lonIdx = Math.floor((lng + 180) / gridOffset.xOffset);
    var key = latIdx + '-' + lonIdx;

    accu[key] = accu[key] || { count: 0, points: [] };
    accu[key].count += 1;
    accu[key].points.push(pt);

    return accu;
  }, {});

  return { gridHash: gridHash, gridOffset: gridOffset };
}

function _getGridLayerDataFromGridHash(gridHash, gridOffset) {
  return Object.keys(gridHash).reduce(function (accu, key, i) {
    var idxs = key.split('-');
    var latIdx = parseInt(idxs[0], 10);
    var lonIdx = parseInt(idxs[1], 10);

    accu.push(Object.assign({
      index: i,
      position: [-180 + gridOffset.xOffset * lonIdx, -90 + gridOffset.yOffset * latIdx]
    }, gridHash[key]));

    return accu;
  }, []);
}

/**
 * calculate grid layer cell size in lat lon based on world unit size
 * and current latitude
 * @param {number} cellSize
 * @param {number} latitude
 * @returns {object} - lat delta and lon delta
 */
function _calculateGridLatLonOffset(cellSize, latitude) {
  var yOffset = _calculateLatOffset(cellSize);
  var xOffset = _calculateLonOffset(latitude, cellSize);
  return { yOffset: yOffset, xOffset: xOffset };
}

/**
 * with a given x-km change, calculate the increment of latitude
 * based on stackoverflow http://stackoverflow.com/questions/7477003
 * @param {number} dy - change in km
 * @return {number} - increment in latitude
 */
function _calculateLatOffset(dy) {
  return dy / R_EARTH * (180 / Math.PI);
}

/**
 * with a given x-km change, and current latitude
 * calculate the increment of longitude
 * based on stackoverflow http://stackoverflow.com/questions/7477003
 * @param {number} lat - latitude of current location (based on city)
 * @param {number} dx - change in km
 * @return {number} - increment in longitude
 */
function _calculateLonOffset(lat, dx) {
  return dx / R_EARTH * (180 / Math.PI) / Math.cos(lat * Math.PI / 180);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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