var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import assert from 'assert';
import { Framebuffer, ShaderCache } from 'luma.gl';
import seer from 'seer';
import Layer from './layer';
import { drawLayers as _drawLayers } from './draw-layers';
import { pickObject as _pickObject, pickVisibleObjects } from './pick-layers';
import { LIFECYCLE } from './constants';
import Viewport from '../viewports/viewport';
// TODO - remove, just for dummy initialization
import WebMercatorViewport from '../viewports/web-mercator-viewport';
import log from '../utils/log';
import { flatten } from '../utils/flatten';

import { setPropOverrides, layerEditListener, seerInitListener, initLayerInSeer, updateLayerInSeer } from './seer-integration';

var LOG_PRIORITY_LIFECYCLE = 2;
var LOG_PRIORITY_LIFECYCLE_MINOR = 4;

var initialContext = {
  uniforms: {},
  viewports: [],
  viewport: null,
  layerFilter: null,
  viewportChanged: true,
  pickingFBO: null,
  useDevicePixels: true,
  lastPickedInfo: {
    index: -1,
    layerId: null
  }
};

var layerName = function layerName(layer) {
  return layer instanceof Layer ? '' + layer : !layer ? 'null' : 'invalid';
};

var LayerManager = function () {
  // eslint-disable-next-line
  function LayerManager(gl) {
    var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        eventManager = _ref.eventManager;

    _classCallCheck(this, LayerManager);

    // Currently deck.gl expects the DeckGL.layers array to be different
    // whenever React rerenders. If the same layers array is used, the
    // LayerManager's diffing algorithm will generate a fatal error and
    // break the rendering.

    // `this.lastRenderedLayers` stores the UNFILTERED layers sent
    // down to LayerManager, so that `layers` reference can be compared.
    // If it's the same across two React render calls, the diffing logic
    // will be skipped.
    this.lastRenderedLayers = [];
    this.prevLayers = [];
    this.layers = [];

    this.oldContext = {};
    this.context = Object.assign({}, initialContext, {
      gl: gl,
      // Enabling luma.gl Program caching using private API (_cachePrograms)
      shaderCache: new ShaderCache({ gl: gl, _cachePrograms: true })
    });

    // List of view descriptors, gets re-evaluated when width/height changes
    this.width = 100;
    this.height = 100;
    this.viewDescriptors = [];
    this.viewDescriptorsChanged = true;
    this.viewports = []; // Generated viewports
    this._needsRedraw = 'Initial render';

    // Event handling
    this._pickingRadius = 0;

    this._eventManager = null;
    this._onLayerClick = null;
    this._onLayerHover = null;
    this._onClick = this._onClick.bind(this);
    this._onPointerMove = this._onPointerMove.bind(this);
    this._onPointerLeave = this._onPointerLeave.bind(this);
    this._pickAndCallback = this._pickAndCallback.bind(this);

    // Seer integration
    this._initSeer = this._initSeer.bind(this);
    this._editSeer = this._editSeer.bind(this);
    seerInitListener(this._initSeer);
    layerEditListener(this._editSeer);

    Object.seal(this);

    if (eventManager) {
      this._initEventHandling(eventManager);
    }

    // Init with dummy viewport
    this.setViewports([new WebMercatorViewport({ width: 1, height: 1, latitude: 0, longitude: 0, zoom: 1 })]);
  }

  /**
   * Method to call when the layer manager is not needed anymore.
   *
   * Currently used in the <DeckGL> componentWillUnmount lifecycle to unbind Seer listeners.
   */


  _createClass(LayerManager, [{
    key: 'finalize',
    value: function finalize() {
      seer.removeListener(this._initSeer);
      seer.removeListener(this._editSeer);
    }
  }, {
    key: 'needsRedraw',
    value: function needsRedraw() {
      var _ref2 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref2$clearRedrawFlag = _ref2.clearRedrawFlags,
          clearRedrawFlags = _ref2$clearRedrawFlag === undefined ? true : _ref2$clearRedrawFlag;

      return this._checkIfNeedsRedraw(clearRedrawFlags);
    }

    // Normally not called by app

  }, {
    key: 'setNeedsRedraw',
    value: function setNeedsRedraw(reason) {
      this._needsRedraw = this._needsRedraw || reason;
    }

    // Gets an (optionally) filtered list of layers

  }, {
    key: 'getLayers',
    value: function getLayers() {
      var _ref3 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref3$layerIds = _ref3.layerIds,
          layerIds = _ref3$layerIds === undefined ? null : _ref3$layerIds;

      // Filtering by layerId compares beginning of strings, so that sublayers will be included
      // Dependes on the convention of adding suffixes to the parent's layer name
      return layerIds ? this.layers.filter(function (layer) {
        return layerIds.find(function (layerId) {
          return layer.id.indexOf(layerId) === 0;
        });
      }) : this.layers;
    }

    // Get a set of viewports for a given width and height
    // TODO - Intention is for deck.gl to autodeduce width and height and drop the need for props

  }, {
    key: 'getViewports',
    value: function getViewports() {
      var _ref4 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          width = _ref4.width,
          height = _ref4.height;

      if (width !== this.width || height !== this.height || this.viewDescriptorsChanged) {
        this._rebuildViewportsFromViews({ viewDescriptors: this.viewDescriptors, width: width, height: height });
        this.width = width;
        this.height = height;
      }
      return this.viewports;
    }

    /**
     * Set parameters needed for layer rendering and picking.
     * Parameters are to be passed as a single object, with the following values:
     * @param {Boolean} useDevicePixels
     */

  }, {
    key: 'setParameters',
    value: function setParameters(parameters) {
      if ('eventManager' in parameters) {
        this._initEventHandling(parameters.eventManager);
      }

      if ('pickingRadius' in parameters || 'onLayerClick' in parameters || 'onLayerHover' in parameters) {
        this._setEventHandlingParameters(parameters);
      }

      // TODO - For now we set layers before viewports to preservenchangeFlags
      if ('layers' in parameters) {
        this.setLayers(parameters.layers);
      }

      if ('viewports' in parameters) {
        this.setViewports(parameters.viewports);
      }

      if ('layerFilter' in parameters) {
        this.context.layerFilter = parameters.layerFilter;
        if (this.context.layerFilter !== parameters.layerFilter) {
          this.setNeedsRedraw('layerFilter changed');
        }
      }

      if ('drawPickingColors' in parameters) {
        if (this.context.drawPickingColors !== parameters.drawPickingColors) {
          this.setNeedsRedraw('drawPickingColors changed');
        }
      }

      Object.assign(this.context, parameters);
    }

    // Update the view descriptor list and set change flag if needed

  }, {
    key: 'setViewports',
    value: function setViewports(viewports) {
      // Ensure viewports are wrapped in descriptors
      var viewDescriptors = flatten(viewports, { filter: Boolean }).map(function (viewport) {
        return viewport instanceof Viewport ? { viewport: viewport } : viewport;
      });

      this.viewDescriptorsChanged = this.viewDescriptorsChanged || this._diffViews(viewDescriptors, this.viewDescriptors);

      // Try to not actually rebuild the viewports until `getViewports` is called
      if (this.viewDescriptorsChanged) {
        this.viewDescriptors = viewDescriptors;
        this._rebuildViewportsFromViews({ viewDescriptors: this.viewDescriptors });
        this.viewDescriptorsChanged = false;
      }
    }

    // Supply a new layer list, initiating sublayer generation and layer matching

  }, {
    key: 'setLayers',
    value: function setLayers(newLayers) {
      assert(this.context.viewport, 'LayerManager.updateLayers: viewport not set');

      // TODO - something is generating state updates that cause rerender of the same
      if (newLayers === this.lastRenderedLayers) {
        log.log(3, 'Ignoring layer update due to layer array not changed');
        return this;
      }
      this.lastRenderedLayers = newLayers;

      newLayers = flatten(newLayers, { filter: Boolean });

      var _iteratorNormalCompletion = true;
      var _didIteratorError = false;
      var _iteratorError = undefined;

      try {
        for (var _iterator = newLayers[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
          var layer = _step.value;

          layer.context = this.context;
        }
      } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion && _iterator.return) {
            _iterator.return();
          }
        } finally {
          if (_didIteratorError) {
            throw _iteratorError;
          }
        }
      }

      this.prevLayers = this.layers;

      var _updateLayers2 = this._updateLayers({
        oldLayers: this.prevLayers,
        newLayers: newLayers
      }),
          error = _updateLayers2.error,
          generatedLayers = _updateLayers2.generatedLayers;

      this.layers = generatedLayers;
      // Throw first error found, if any
      if (error) {
        throw error;
      }
      return this;
    }
  }, {
    key: 'drawLayers',
    value: function drawLayers() {
      var _ref5 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref5$pass = _ref5.pass,
          pass = _ref5$pass === undefined ? 'render to screen' : _ref5$pass,
          _ref5$redrawReason = _ref5.redrawReason,
          redrawReason = _ref5$redrawReason === undefined ? 'unknown reason' : _ref5$redrawReason;

      var _context = this.context,
          gl = _context.gl,
          useDevicePixels = _context.useDevicePixels,
          drawPickingColors = _context.drawPickingColors;

      // render this viewport

      _drawLayers(gl, {
        layers: this.layers,
        viewports: this.getViewports(),
        onViewportActive: this._activateViewport.bind(this),
        useDevicePixels: useDevicePixels,
        drawPickingColors: drawPickingColors,
        pass: pass,
        layerFilter: this.context.layerFilter,
        redrawReason: redrawReason
      });
    }

    // Pick the closest info at given coordinate

  }, {
    key: 'pickObject',
    value: function pickObject(_ref6) {
      var x = _ref6.x,
          y = _ref6.y,
          mode = _ref6.mode,
          _ref6$radius = _ref6.radius,
          radius = _ref6$radius === undefined ? 0 : _ref6$radius,
          layerIds = _ref6.layerIds,
          layerFilter = _ref6.layerFilter;
      var _context2 = this.context,
          gl = _context2.gl,
          useDevicePixels = _context2.useDevicePixels;


      var layers = this.getLayers({ layerIds: layerIds });

      return _pickObject(gl, {
        // User params
        x: x,
        y: y,
        radius: radius,
        layers: layers,
        mode: mode,
        layerFilter: layerFilter,
        // Injected params
        viewports: this.getViewports(),
        onViewportActive: this._activateViewport.bind(this),
        pickingFBO: this._getPickingBuffer(),
        lastPickedInfo: this.context.lastPickedInfo,
        useDevicePixels: useDevicePixels
      });
    }

    // Get all unique infos within a bounding box

  }, {
    key: 'pickObjects',
    value: function pickObjects(_ref7) {
      var x = _ref7.x,
          y = _ref7.y,
          width = _ref7.width,
          height = _ref7.height,
          layerIds = _ref7.layerIds,
          layerFilter = _ref7.layerFilter;
      var _context3 = this.context,
          gl = _context3.gl,
          useDevicePixels = _context3.useDevicePixels;


      var layers = this.getLayers({ layerIds: layerIds });

      return pickVisibleObjects(gl, {
        x: x,
        y: y,
        width: width,
        height: height,
        layers: layers,
        layerFilter: layerFilter,
        mode: 'pickObjects',
        // TODO - how does this interact with multiple viewports?
        viewport: this.context.viewport,
        viewports: this.getViewports(),
        onViewportActive: this._activateViewport.bind(this),
        pickingFBO: this._getPickingBuffer(),
        useDevicePixels: useDevicePixels
      });
    }

    //
    // DEPRECATED METHODS in V5
    //

  }, {
    key: 'updateLayers',
    value: function updateLayers(_ref8) {
      var newLayers = _ref8.newLayers;

      log.deprecated('updateLayers', 'setLayers');
      this.setLayers(newLayers);
    }
  }, {
    key: 'setViewport',
    value: function setViewport(viewport) {
      log.deprecated('setViewport', 'setViewports');
      this.setViewports([viewport]);
      return this;
    }

    //
    // PRIVATE METHODS
    //

  }, {
    key: '_checkIfNeedsRedraw',
    value: function _checkIfNeedsRedraw(clearRedrawFlags) {
      var redraw = this._needsRedraw;
      if (clearRedrawFlags) {
        this._needsRedraw = false;
      }

      // This layers list doesn't include sublayers, relying on composite layers
      var _iteratorNormalCompletion2 = true;
      var _didIteratorError2 = false;
      var _iteratorError2 = undefined;

      try {
        for (var _iterator2 = this.layers[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
          var layer = _step2.value;

          // Call every layer to clear their flags
          var layerNeedsRedraw = layer.getNeedsRedraw({ clearRedrawFlags: clearRedrawFlags });
          redraw = redraw || layerNeedsRedraw;
        }
      } catch (err) {
        _didIteratorError2 = true;
        _iteratorError2 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion2 && _iterator2.return) {
            _iterator2.return();
          }
        } finally {
          if (_didIteratorError2) {
            throw _iteratorError2;
          }
        }
      }

      return redraw;
    }

    // Rebuilds viewports from descriptors towards a certain window size

  }, {
    key: '_rebuildViewportsFromViews',
    value: function _rebuildViewportsFromViews(_ref9) {
      var _this = this;

      var viewDescriptors = _ref9.viewDescriptors,
          width = _ref9.width,
          height = _ref9.height;

      var newViewports = viewDescriptors.map(function (viewDescriptor) {
        return (
          // If a `Viewport` instance was supplied, use it, otherwise build it
          viewDescriptor.viewport instanceof Viewport ? viewDescriptor.viewport : _this._makeViewportFromViewDescriptor({ viewDescriptor: viewDescriptor, width: width, height: height })
        );
      });

      this.setNeedsRedraw('Viewport(s) changed');

      // Ensure one viewport is activated, layers may expect it
      // TODO - handle empty viewport list (using dummy viewport), or assert
      // const oldViewports = this.context.viewports;
      // if (viewportsChanged) {

      var viewport = newViewports[0];
      assert(viewport instanceof Viewport, 'Invalid viewport');

      this.context.viewports = newViewports;
      this._activateViewport(viewport);
      // }

      // We've just rebuilt the viewports to match the descriptors, so clear the flag
      this.viewports = newViewports;
      this.viewDescriptorsChanged = false;
    }

    // Build a `Viewport` from a view descriptor
    // TODO - add support for autosizing viewports using width and height

  }, {
    key: '_makeViewportFromViewDescriptor',
    value: function _makeViewportFromViewDescriptor(_ref10) {
      var viewDescriptor = _ref10.viewDescriptor,
          width = _ref10.width,
          height = _ref10.height;

      // Get the type of the viewport
      // TODO - default to WebMercator?
      var ViewportType = viewDescriptor.type,
          viewState = viewDescriptor.viewState;

      // Resolve relative viewport dimensions
      // TODO - we need to have width and height available

      var viewportDimensions = this._getViewDimensions({ viewDescriptor: viewDescriptor });

      // Create the viewport, giving preference to view state in `viewState`
      return new ViewportType(Object.assign({}, viewDescriptor, viewportDimensions, viewState // Object.assign handles undefined
      ));
    }

    // Check if viewport array has changed, returns true if any change
    // Note that descriptors can be the same

  }, {
    key: '_diffViews',
    value: function _diffViews(newViews, oldViews) {
      var _this2 = this;

      if (newViews.length !== oldViews.length) {
        return true;
      }

      return newViews.some(function (_, i) {
        return _this2._diffView(newViews[i], oldViews[i]);
      });
    }
  }, {
    key: '_diffView',
    value: function _diffView(newView, oldView) {
      // `View` hiearchy supports an `equals` method
      if (newView.viewport) {
        return !oldView.viewport || !newView.viewport.equals(oldView.viewport);
      }
      // TODO - implement deep equal on view descriptors
      return newView !== oldView;
    }

    // Support for relative viewport dimensions (e.g {y: '50%', height: '50%'})

  }, {
    key: '_getViewDimensions',
    value: function _getViewDimensions(_ref11) {
      var viewDescriptor = _ref11.viewDescriptor,
          width = _ref11.width,
          height = _ref11.height;

      var parsePercent = function parsePercent(value, max) {
        return value;
      };
      // TODO - enable to support percent size specifiers
      // const parsePercent = (value, max) => value ?
      //   Math.round(parseFloat(value) / 100 * max) :
      //   (value === null ? max : value);

      return {
        x: parsePercent(viewDescriptor.x, width),
        y: parsePercent(viewDescriptor.y, height),
        width: parsePercent(viewDescriptor.width, width),
        height: parsePercent(viewDescriptor.height, height)
      };
    }

    /**
     * @param {Object} eventManager   A source of DOM input events
     */

  }, {
    key: '_initEventHandling',
    value: function _initEventHandling(eventManager) {
      this._eventManager = eventManager;

      // TODO: add/remove handlers on demand at runtime, not all at once on init.
      // Consider both top-level handlers like onLayerClick/Hover
      // and per-layer handlers attached to individual layers.
      // https://github.com/uber/deck.gl/issues/634
      this._eventManager.on({
        click: this._onClick,
        pointermove: this._onPointerMove,
        pointerleave: this._onPointerLeave
      });
    }

    // Set parameters for input event handling.

  }, {
    key: '_setEventHandlingParameters',
    value: function _setEventHandlingParameters(_ref12) {
      var pickingRadius = _ref12.pickingRadius,
          onLayerClick = _ref12.onLayerClick,
          onLayerHover = _ref12.onLayerHover;

      if (!isNaN(pickingRadius)) {
        this._pickingRadius = pickingRadius;
      }
      if (typeof onLayerClick !== 'undefined') {
        this._onLayerClick = onLayerClick;
      }
      if (typeof onLayerHover !== 'undefined') {
        this._onLayerHover = onLayerHover;
      }
      this._validateEventHandling();
    }

    // Make a viewport "current" in layer context, primed for draw

  }, {
    key: '_activateViewport',
    value: function _activateViewport(viewport) {
      // TODO - viewport change detection breaks METER_OFFSETS mode
      // const oldViewport = this.context.viewport;
      // const viewportChanged = !oldViewport || !viewport.equals(oldViewport);
      var viewportChanged = true;

      if (viewportChanged) {
        Object.assign(this.oldContext, this.context);
        this.context.viewport = viewport;
        this.context.viewportChanged = true;
        this.context.uniforms = {};
        log(4, viewport);

        // Update layers states
        // Let screen space layers update their state based on viewport
        // TODO - reimplement viewport change detection (single viewport optimization)
        // TODO - don't set viewportChanged during setViewports?
        if (this.context.viewportChanged) {
          var _iteratorNormalCompletion3 = true;
          var _didIteratorError3 = false;
          var _iteratorError3 = undefined;

          try {
            for (var _iterator3 = this.layers[Symbol.iterator](), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
              var layer = _step3.value;

              layer.setChangeFlags({ viewportChanged: 'Viewport changed' });
              this._updateLayer(layer);
            }
          } catch (err) {
            _didIteratorError3 = true;
            _iteratorError3 = err;
          } finally {
            try {
              if (!_iteratorNormalCompletion3 && _iterator3.return) {
                _iterator3.return();
              }
            } finally {
              if (_didIteratorError3) {
                throw _iteratorError3;
              }
            }
          }
        }
      }

      assert(this.context.viewport, 'LayerManager: viewport not set');

      return this;
    }
  }, {
    key: '_getPickingBuffer',
    value: function _getPickingBuffer() {
      var gl = this.context.gl;
      // Create a frame buffer if not already available

      this.context.pickingFBO = this.context.pickingFBO || new Framebuffer(gl);
      // Resize it to current canvas size (this is a noop if size hasn't changed)
      this.context.pickingFBO.resize({ width: gl.canvas.width, height: gl.canvas.height });
      return this.context.pickingFBO;
    }

    // Match all layers, checking for caught errors
    // To avoid having an exception in one layer disrupt other layers
    // TODO - mark layers with exceptions as bad and remove from rendering cycle?

  }, {
    key: '_updateLayers',
    value: function _updateLayers(_ref13) {
      var oldLayers = _ref13.oldLayers,
          newLayers = _ref13.newLayers;

      // Create old layer map
      var oldLayerMap = {};
      var _iteratorNormalCompletion4 = true;
      var _didIteratorError4 = false;
      var _iteratorError4 = undefined;

      try {
        for (var _iterator4 = oldLayers[Symbol.iterator](), _step4; !(_iteratorNormalCompletion4 = (_step4 = _iterator4.next()).done); _iteratorNormalCompletion4 = true) {
          var oldLayer = _step4.value;

          if (oldLayerMap[oldLayer.id]) {
            log.warn('Multiple old layers with same id ' + layerName(oldLayer));
          } else {
            oldLayerMap[oldLayer.id] = oldLayer;
          }
        }

        // Allocate array for generated layers
      } catch (err) {
        _didIteratorError4 = true;
        _iteratorError4 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion4 && _iterator4.return) {
            _iterator4.return();
          }
        } finally {
          if (_didIteratorError4) {
            throw _iteratorError4;
          }
        }
      }

      var generatedLayers = [];

      // Match sublayers
      var error = this._updateSublayersRecursively({
        newLayers: newLayers,
        oldLayerMap: oldLayerMap,
        generatedLayers: generatedLayers
      });

      // Finalize unmatched layers
      var error2 = this._finalizeOldLayers(oldLayerMap);

      var firstError = error || error2;
      return { error: firstError, generatedLayers: generatedLayers };
    }

    // Note: adds generated layers to `generatedLayers` array parameter

  }, {
    key: '_updateSublayersRecursively',
    value: function _updateSublayersRecursively(_ref14) {
      var newLayers = _ref14.newLayers,
          oldLayerMap = _ref14.oldLayerMap,
          generatedLayers = _ref14.generatedLayers;

      var error = null;

      var _iteratorNormalCompletion5 = true;
      var _didIteratorError5 = false;
      var _iteratorError5 = undefined;

      try {
        for (var _iterator5 = newLayers[Symbol.iterator](), _step5; !(_iteratorNormalCompletion5 = (_step5 = _iterator5.next()).done); _iteratorNormalCompletion5 = true) {
          var newLayer = _step5.value;

          newLayer.context = this.context;

          // Given a new coming layer, find its matching old layer (if any)
          var oldLayer = oldLayerMap[newLayer.id];
          if (oldLayer === null) {
            // null, rather than undefined, means this id was originally there
            log.warn('Multiple new layers with same id ' + layerName(newLayer));
          }
          // Remove the old layer from candidates, as it has been matched with this layer
          oldLayerMap[newLayer.id] = null;

          var sublayers = null;

          // We must not generate exceptions until after layer matching is complete
          try {
            if (!oldLayer) {
              this._initializeLayer(newLayer);
              initLayerInSeer(newLayer); // Initializes layer in seer chrome extension (if connected)
            } else {
              this._transferLayerState(oldLayer, newLayer);
              this._updateLayer(newLayer);
              updateLayerInSeer(newLayer); // Updates layer in seer chrome extension (if connected)
            }
            generatedLayers.push(newLayer);

            // Call layer lifecycle method: render sublayers
            sublayers = newLayer.isComposite && newLayer.getSubLayers();
            // End layer lifecycle method: render sublayers
          } catch (err) {
            log.warn('error during matching of ' + layerName(newLayer), err);
            error = error || err; // Record first exception
          }

          if (sublayers) {
            this._updateSublayersRecursively({
              newLayers: sublayers,
              oldLayerMap: oldLayerMap,
              generatedLayers: generatedLayers
            });
          }
        }
      } catch (err) {
        _didIteratorError5 = true;
        _iteratorError5 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion5 && _iterator5.return) {
            _iterator5.return();
          }
        } finally {
          if (_didIteratorError5) {
            throw _iteratorError5;
          }
        }
      }

      return error;
    }

    // Finalize any old layers that were not matched

  }, {
    key: '_finalizeOldLayers',
    value: function _finalizeOldLayers(oldLayerMap) {
      var error = null;
      for (var layerId in oldLayerMap) {
        var layer = oldLayerMap[layerId];
        if (layer) {
          error = error || this._finalizeLayer(layer);
        }
      }
      return error;
    }

    // Initializes a single layer, calling layer methods

  }, {
    key: '_initializeLayer',
    value: function _initializeLayer(layer) {
      assert(!layer.state);
      log(LOG_PRIORITY_LIFECYCLE, 'initializing ' + layerName(layer));

      var error = null;
      try {
        layer._initialize();
        layer.lifecycle = LIFECYCLE.INITIALIZED;
      } catch (err) {
        log.warn('error while initializing ' + layerName(layer) + '\n', err);
        error = error || err;
        // TODO - what should the lifecycle state be here? LIFECYCLE.INITIALIZATION_FAILED?
      }

      assert(layer.state);

      // Set back pointer (used in picking)
      layer.state.layer = layer;

      // Save layer on model for picking purposes
      // store on model.userData rather than directly on model
      var _iteratorNormalCompletion6 = true;
      var _didIteratorError6 = false;
      var _iteratorError6 = undefined;

      try {
        for (var _iterator6 = layer.getModels()[Symbol.iterator](), _step6; !(_iteratorNormalCompletion6 = (_step6 = _iterator6.next()).done); _iteratorNormalCompletion6 = true) {
          var model = _step6.value;

          model.userData.layer = layer;
        }
      } catch (err) {
        _didIteratorError6 = true;
        _iteratorError6 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion6 && _iterator6.return) {
            _iterator6.return();
          }
        } finally {
          if (_didIteratorError6) {
            throw _iteratorError6;
          }
        }
      }

      return error;
    }
  }, {
    key: '_transferLayerState',
    value: function _transferLayerState(oldLayer, newLayer) {
      if (newLayer !== oldLayer) {
        log(LOG_PRIORITY_LIFECYCLE_MINOR, 'matched ' + layerName(newLayer), oldLayer, '->', newLayer);
        newLayer.lifecycle = LIFECYCLE.MATCHED;
        oldLayer.lifecycle = LIFECYCLE.AWAITING_GC;
        newLayer._transferState(oldLayer);
      } else {
        log.log(LOG_PRIORITY_LIFECYCLE_MINOR, 'Matching layer is unchanged ' + newLayer.id);
        newLayer.lifecycle = LIFECYCLE.MATCHED;
        newLayer.oldProps = newLayer.props;
      }
    }

    // Updates a single layer, cleaning all flags

  }, {
    key: '_updateLayer',
    value: function _updateLayer(layer) {
      log.log(LOG_PRIORITY_LIFECYCLE_MINOR, 'updating ' + layer + ' because: ' + layer.printChangeFlags());
      var error = null;
      try {
        layer._update();
      } catch (err) {
        log.warn('error during update of ' + layerName(layer), err);
        // Save first error
        error = err;
      }
      return error;
    }

    // Finalizes a single layer

  }, {
    key: '_finalizeLayer',
    value: function _finalizeLayer(layer) {
      assert(layer.state);
      assert(layer.lifecycle !== LIFECYCLE.AWAITING_FINALIZATION);
      layer.lifecycle = LIFECYCLE.AWAITING_FINALIZATION;
      var error = null;
      this.setNeedsRedraw('finalized ' + layerName(layer));
      try {
        layer._finalize();
      } catch (err) {
        log.warn('error during finalization of ' + layerName(layer), err);
        error = err;
      }
      layer.lifecycle = LIFECYCLE.FINALIZED;
      log(LOG_PRIORITY_LIFECYCLE, 'finalizing ' + layerName(layer));
      return error;
    }

    /**
     * Warn if a deck-level mouse event has been specified,
     * but no layers are `pickable`.
     */

  }, {
    key: '_validateEventHandling',
    value: function _validateEventHandling() {
      if (this.onLayerClick || this.onLayerHover) {
        if (this.layers.length && !this.layers.some(function (layer) {
          return layer.props.pickable;
        })) {
          log.warn('You have supplied a top-level input event handler (e.g. `onLayerClick`), ' + 'but none of your layers have set the `pickable` flag.');
        }
      }
    }

    /**
     * Route click events to layers.
     * `pickLayer` will call the `onClick` prop of any picked layer,
     * and `onLayerClick` is called directly from here
     * with any picking info generated by `pickLayer`.
     * @param {Object} event  An object encapsulating an input event,
     *                        with the following shape:
     *                        {Object: {x, y}} offsetCenter: center of the event
     *                        {Object} srcEvent:             native JS Event object
     */

  }, {
    key: '_onClick',
    value: function _onClick(event) {
      if (!event.offsetCenter) {
        // Do not trigger onHover callbacks when click position is invalid.
        return;
      }
      this._pickAndCallback({
        callback: this._onLayerClick,
        event: event,
        mode: 'click'
      });
    }

    /**
     * Route click events to layers.
     * `pickLayer` will call the `onHover` prop of any picked layer,
     * and `onLayerHover` is called directly from here
     * with any picking info generated by `pickLayer`.
     * @param {Object} event  An object encapsulating an input event,
     *                        with the following shape:
     *                        {Object: {x, y}} offsetCenter: center of the event
     *                        {Object} srcEvent:             native JS Event object
     */

  }, {
    key: '_onPointerMove',
    value: function _onPointerMove(event) {
      if (event.leftButton || event.rightButton) {
        // Do not trigger onHover callbacks if mouse button is down.
        return;
      }
      this._pickAndCallback({
        callback: this._onLayerHover,
        event: event,
        mode: 'hover'
      });
    }
  }, {
    key: '_onPointerLeave',
    value: function _onPointerLeave(event) {
      this.pickObject({
        x: -1,
        y: -1,
        radius: this._pickingRadius,
        mode: 'hover'
      });
    }
  }, {
    key: '_pickAndCallback',
    value: function _pickAndCallback(options) {
      var pos = options.event.offsetCenter;
      var radius = this._pickingRadius;
      var selectedInfos = this.pickObject({ x: pos.x, y: pos.y, radius: radius, mode: options.mode });
      if (options.callback) {
        var firstInfo = selectedInfos.find(function (info) {
          return info.index >= 0;
        }) || null;
        // As per documentation, send null value when no valid object is picked.
        options.callback(firstInfo, selectedInfos, options.event.srcEvent);
      }
    }

    // SEER INTEGRATION

    /**
     * Called upon Seer initialization, manually sends layers data.
     */

  }, {
    key: '_initSeer',
    value: function _initSeer() {
      this.layers.forEach(function (layer) {
        initLayerInSeer(layer);
        updateLayerInSeer(layer);
      });
    }

    /**
     * On Seer property edition, set override and update layers.
     */

  }, {
    key: '_editSeer',
    value: function _editSeer(payload) {
      if (payload.type !== 'edit' || payload.valuePath[0] !== 'props') {
        return;
      }

      setPropOverrides(payload.itemKey, payload.valuePath.slice(1), payload.value);
      var newLayers = this.layers.map(function (layer) {
        return new layer.constructor(layer.props);
      });
      this.updateLayers({ newLayers: newLayers });
    }
  }]);

  return LayerManager;
}();

export default LayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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