// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

export default "// EXTERNAL CONSTANTS: these must match JavaScript constants in \"src/core/lib/constants.js\"\nconst float COORDINATE_SYSTEM_IDENTITY = 0.;\nconst float COORDINATE_SYSTEM_LNG_LAT = 1.;\nconst float COORDINATE_SYSTEM_METER_OFFSETS = 2.;\nconst float COORDINATE_SYSTEM_LNGLAT_OFFSETS = 3.;\n\nuniform float project_uCoordinateSystem;\nuniform float project_uScale;\nuniform vec3 project_uPixelsPerMeter;\nuniform vec3 project_uPixelsPerDegree;\nuniform vec3 project_uPixelsPerUnit;\nuniform vec3 project_uPixelsPerUnit2;\nuniform vec4 project_uCenter;\nuniform mat4 project_uModelMatrix;\nuniform mat4 project_uViewProjectionMatrix;\nuniform vec2 project_uViewportSize;\nuniform float project_uDevicePixelRatio;\nuniform float project_uFocalDistance;\nuniform vec3 project_uCameraPosition;\n\nconst float TILE_SIZE = 512.0;\nconst float PI = 3.1415926536;\nconst float WORLD_SCALE = TILE_SIZE / (PI * 2.0);\n\n//\n// Scaling offsets - scales meters to \"pixels\"\n// Note the scalar version of project_scale is for scaling the z component only\n//\nfloat project_scale(float meters) {\n  return meters * project_uPixelsPerMeter.z;\n}\n\nvec2 project_scale(vec2 meters) {\n  return meters * project_uPixelsPerMeter.xy;\n}\n\nvec3 project_scale(vec3 meters) {\n  return meters * project_uPixelsPerMeter;\n}\n\nvec4 project_scale(vec4 meters) {\n  return vec4(meters.xyz * project_uPixelsPerMeter, meters.w);\n}\n\n//\n// Projecting normal - transform deltas from current coordinate system to\n// normals in the worldspace\n//\nvec3 project_normal(vec3 vector) {\n  if (project_uCoordinateSystem == COORDINATE_SYSTEM_LNG_LAT ||\n    project_uCoordinateSystem == COORDINATE_SYSTEM_LNGLAT_OFFSETS) {\n    return normalize(vector * project_uPixelsPerDegree);\n  }\n  return normalize(vector * project_uPixelsPerMeter);\n}\n\nvec4 project_offset_(vec4 offset) {\n  vec3 pixelsPerUnit = project_uPixelsPerUnit + project_uPixelsPerUnit2 * offset.y;\n  return vec4(offset.xyz * pixelsPerUnit, offset.w);\n}\n\n//\n// Projecting positions - non-linear projection: lnglats => unit tile [0-1, 0-1]\n//\nvec2 project_mercator_(vec2 lnglat) {\n  return vec2(\n    radians(lnglat.x) + PI,\n    PI - log(tan_fp32(PI * 0.25 + radians(lnglat.y) * 0.5))\n  );\n}\n\n//\n// Projects lnglats (or meter offsets, depending on mode) to pixels\n//\nvec4 project_position(vec4 position) {\n  // TODO - why not simply subtract center and fall through?\n  if (project_uCoordinateSystem == COORDINATE_SYSTEM_LNG_LAT) {\n    return project_uModelMatrix * vec4(\n      project_mercator_(position.xy) * WORLD_SCALE * project_uScale,\n      project_scale(position.z),\n      position.w\n    );\n  }\n\n  if (project_uCoordinateSystem == COORDINATE_SYSTEM_LNGLAT_OFFSETS) {\n    return project_offset_(position);\n  }\n\n  // METER_OFFSETS or IDENTITY\n  // Apply model matrix\n  vec4 position_modelspace = project_uModelMatrix * position;\n  return project_offset_(position_modelspace);\n}\n\nvec3 project_position(vec3 position) {\n  vec4 projected_position = project_position(vec4(position, 1.0));\n  return projected_position.xyz;\n}\n\nvec2 project_position(vec2 position) {\n  vec4 projected_position = project_position(vec4(position, 0.0, 1.0));\n  return projected_position.xy;\n}\n\n//\n// Projects from \"world\" coordinates to clip space.\n// Uses project_uViewProjectionMatrix\n//\nvec4 project_to_clipspace(vec4 position) {\n  if (project_uCoordinateSystem == COORDINATE_SYSTEM_METER_OFFSETS ||\n    project_uCoordinateSystem == COORDINATE_SYSTEM_LNGLAT_OFFSETS) {\n    // Needs to be divided with project_uPixelsPerMeter\n    position.w *= project_uPixelsPerMeter.z;\n  }\n  return project_uViewProjectionMatrix * position + project_uCenter;\n}\n\n// Returns a clip space offset that corresponds to a given number of **non-device** pixels\nvec4 project_pixel_to_clipspace(vec2 pixels) {\n  vec2 offset = pixels / project_uViewportSize * project_uDevicePixelRatio;\n  return vec4(offset * project_uFocalDistance, 0.0, 0.0);\n}\n";
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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