// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

/* global window */
import mat4_multiply from 'gl-mat4/multiply';
import vec4_transformMat4 from 'gl-vec4/transformMat4';

import log from '../../utils/log';
import assert from 'assert';
import { COORDINATE_SYSTEM } from '../../lib/constants';
import memoize from '../../utils/memoize';

import { lngLatToWorld } from 'viewport-mercator-project';

// To quickly set a vector to zero
var ZERO_VECTOR = [0, 0, 0, 0];
// 4x4 matrix that drops 4th component of vector
var VECTOR_TO_POINT_MATRIX = [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0];
var IDENTITY_MATRIX = [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1];
var DEFAULT_PIXELS_PER_UNIT2 = [0, 0, 0];
var DEFAULT_COORDINATE_ORIGIN = [0, 0, 0];

var getMemoizedViewportUniforms = memoize(calculateViewportUniforms);

// The code that utilizes Matrix4 does the same calculation as their mat4 counterparts,
// has lower performance but provides error checking.
// Uncomment when debugging
function calculateMatrixAndOffset(_ref) {
  var viewport = _ref.viewport,
      coordinateSystem = _ref.coordinateSystem,
      coordinateOrigin = _ref.coordinateOrigin,
      coordinateZoom = _ref.coordinateZoom;
  var viewMatrixUncentered = viewport.viewMatrixUncentered;
  var viewMatrix = viewport.viewMatrix;
  var projectionMatrix = viewport.projectionMatrix;
  var viewProjectionMatrix = viewport.viewProjectionMatrix;


  var projectionCenter = void 0;

  switch (coordinateSystem) {
    case COORDINATE_SYSTEM.IDENTITY:
    case COORDINATE_SYSTEM.LNGLAT:
      projectionCenter = ZERO_VECTOR;
      break;

    // TODO: make lighting work for meter offset mode
    case COORDINATE_SYSTEM.LNGLAT_OFFSETS:
    case COORDINATE_SYSTEM.METER_OFFSETS:
      // Calculate transformed projectionCenter (using 64 bit precision JS)
      // This is the key to offset mode precision
      // (avoids doing this addition in 32 bit precision in GLSL)
      var positionPixels = lngLatToWorld(coordinateOrigin, Math.pow(2, coordinateZoom));
      // projectionCenter = new Matrix4(viewProjectionMatrix)
      //   .transformVector([positionPixels[0], positionPixels[1], 0.0, 1.0]);
      projectionCenter = vec4_transformMat4([], [positionPixels[0], positionPixels[1], 0.0, 1.0], viewProjectionMatrix);

      // Always apply uncentered projection matrix if available (shader adds center)
      viewMatrix = viewMatrixUncentered || viewMatrix;

      // Zero out 4th coordinate ("after" model matrix) - avoids further translations
      // viewMatrix = new Matrix4(viewMatrixUncentered || viewMatrix)
      //   .multiplyRight(VECTOR_TO_POINT_MATRIX);
      viewProjectionMatrix = mat4_multiply([], projectionMatrix, viewMatrix);
      viewProjectionMatrix = mat4_multiply([], viewProjectionMatrix, VECTOR_TO_POINT_MATRIX);
      break;

    default:
      throw new Error('Unknown projection mode');
  }

  return {
    viewMatrix: viewMatrix,
    viewProjectionMatrix: viewProjectionMatrix,
    projectionCenter: projectionCenter,
    cameraPos: viewport.cameraPosition
  };
}

/**
 * Returns uniforms for shaders based on current projection
 * includes: projection matrix suitable for shaders
 *
 * TODO - Ensure this works with any viewport, not just WebMercatorViewports
 *
 * @param {WebMercatorViewport} viewport -
 * @return {Float32Array} - 4x4 projection matrix that can be used in shaders
 */
export function getUniformsFromViewport() {
  var _ref2 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      viewport = _ref2.viewport,
      _ref2$modelMatrix = _ref2.modelMatrix,
      modelMatrix = _ref2$modelMatrix === undefined ? null : _ref2$modelMatrix,
      _ref2$coordinateSyste = _ref2.coordinateSystem,
      coordinateSystem = _ref2$coordinateSyste === undefined ? COORDINATE_SYSTEM.LNGLAT : _ref2$coordinateSyste,
      _ref2$coordinateOrigi = _ref2.coordinateOrigin,
      coordinateOrigin = _ref2$coordinateOrigi === undefined ? DEFAULT_COORDINATE_ORIGIN : _ref2$coordinateOrigi,
      projectionMode = _ref2.projectionMode,
      positionOrigin = _ref2.positionOrigin;

  assert(viewport);

  if (projectionMode !== undefined) {
    log.removed('projectionMode', 'coordinateSystem');
  }
  if (positionOrigin !== undefined) {
    log.removed('positionOrigin', 'coordinateOrigin');
  }

  return Object.assign({
    project_uModelMatrix: modelMatrix || IDENTITY_MATRIX
  }, getMemoizedViewportUniforms({ viewport: viewport, coordinateSystem: coordinateSystem, coordinateOrigin: coordinateOrigin }));
}

function calculateViewportUniforms(_ref3) {
  var viewport = _ref3.viewport,
      coordinateSystem = _ref3.coordinateSystem,
      coordinateOrigin = _ref3.coordinateOrigin;

  var coordinateZoom = viewport.zoom;
  assert(coordinateZoom >= 0);

  var _calculateMatrixAndOf = calculateMatrixAndOffset({
    coordinateSystem: coordinateSystem,
    coordinateOrigin: coordinateOrigin,
    coordinateZoom: coordinateZoom,
    viewport: viewport
  }),
      projectionCenter = _calculateMatrixAndOf.projectionCenter,
      viewProjectionMatrix = _calculateMatrixAndOf.viewProjectionMatrix,
      cameraPos = _calculateMatrixAndOf.cameraPos;

  assert(viewProjectionMatrix, 'Viewport missing modelViewProjectionMatrix');

  // Calculate projection pixels per unit
  var distanceScales = viewport.getDistanceScales();

  // TODO - does this depend on useDevicePixels?
  var devicePixelRatio = typeof window !== 'undefined' && window.devicePixelRatio || 1;
  var viewportSize = [viewport.width * devicePixelRatio, viewport.height * devicePixelRatio];

  var uniforms = {
    // Projection mode values
    project_uCoordinateSystem: coordinateSystem,
    project_uCenter: projectionCenter,

    // Screen size
    project_uViewportSize: viewportSize,
    project_uDevicePixelRatio: devicePixelRatio,

    // Distance at which screen pixels are projected
    project_uFocalDistance: viewport.focalDistance || 1,
    project_uPixelsPerMeter: distanceScales.pixelsPerMeter,
    project_uPixelsPerDegree: distanceScales.pixelsPerDegree,
    project_uPixelsPerUnit: distanceScales.pixelsPerMeter,
    project_uPixelsPerUnit2: DEFAULT_PIXELS_PER_UNIT2,
    project_uScale: viewport.scale, // This is the mercator scale (2 ** zoom)

    project_uViewProjectionMatrix: viewProjectionMatrix,

    // This is for lighting calculations
    project_uCameraPosition: cameraPos
  };

  if (coordinateSystem === COORDINATE_SYSTEM.METER_OFFSETS) {
    var distanceScalesAtOrigin = viewport.getDistanceScales(coordinateOrigin);
    uniforms.project_uPixelsPerUnit = distanceScalesAtOrigin.pixelsPerMeter;
    uniforms.project_uPixelsPerUnit2 = distanceScalesAtOrigin.pixelsPerMeter2;
  }
  if (coordinateSystem === COORDINATE_SYSTEM.LNGLAT_OFFSETS) {
    var _distanceScalesAtOrigin = viewport.getDistanceScales(coordinateOrigin);
    uniforms.project_uPixelsPerUnit = _distanceScalesAtOrigin.pixelsPerDegree;
    uniforms.project_uPixelsPerUnit2 = _distanceScalesAtOrigin.pixelsPerDegree2;
  }

  return uniforms;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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