// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import { experimental } from 'deck.gl';
var flattenVertices = experimental.flattenVertices;

import earcut from 'earcut';

// Basic polygon support
//
// Handles simple and complex polygons
// Simple polygons are arrays of vertices, implicitly "closed"
// Complex polygons are arrays of simple polygons, with the first polygon
// representing the outer hull and other polygons representing holes

/**
 * Check if this is a non-nested polygon (i.e. the first element of the first element is a number)
 * @param {Array} polygon - either a complex or simple polygon
 * @return {Boolean} - true if the polygon is a simple polygon (i.e. not an array of polygons)
 */
export function isSimple(polygon) {
  return polygon.length >= 1 && polygon[0].length >= 2 && Number.isFinite(polygon[0][0]);
}

/**
 * Normalize to ensure that all polygons in a list are complex - simplifies processing
 * @param {Array} polygon - either a complex or a simple polygon
 * @param {Object} opts
 * @param {Object} opts.dimensions - if 3, the coords will be padded with 0's if needed
 * @return {Array} - returns a complex polygons
 */
export function normalize(polygon) {
  var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref$dimensions = _ref.dimensions,
      dimensions = _ref$dimensions === undefined ? 3 : _ref$dimensions;

  return isSimple(polygon) ? [polygon] : polygon;
}

/**
 * Check if this is a non-nested polygon (i.e. the first element of the first element is a number)
 * @param {Array} polygon - either a complex or simple polygon
 * @return {Boolean} - true if the polygon is a simple polygon (i.e. not an array of polygons)
 */
export function getVertexCount(polygon) {
  return isSimple(polygon) ? polygon.length : polygon.reduce(function (length, simplePolygon) {
    return length + simplePolygon.length;
  }, 0);
}

// Return number of triangles needed to tesselate the polygon
export function getTriangleCount(polygon) {
  var triangleCount = 0;
  var first = true;
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = normalize(polygon)[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var simplePolygon = _step.value;

      var size = simplePolygon.length;
      if (first) {
        triangleCount += size >= 3 ? size - 2 : 0;
      } else {
        triangleCount += size + 1;
      }
      first = false;
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  return triangleCount;
}

export function forEachVertex(polygon, visitor) {
  if (isSimple(polygon)) {
    polygon.forEach(visitor);
    return;
  }

  polygon.forEach(function (simplePolygon) {
    simplePolygon.forEach(visitor);
  });
}

// Returns the offset of each hole polygon in the flattened array for that polygon
function getHoleIndices(complexPolygon) {
  var holeIndices = null;
  if (complexPolygon.length > 1) {
    var polygonStartIndex = 0;
    holeIndices = [];
    complexPolygon.forEach(function (polygon) {
      polygonStartIndex += polygon.length;
      holeIndices.push(polygonStartIndex);
    });
    // Last element points to end of the flat array, remove it
    holeIndices.pop();
  }
  return holeIndices;
}

/*
 * Get vertex indices for drawing complexPolygon mesh
 * @private
 * @param {[Number,Number,Number][][]} complexPolygon
 * @returns {[Number]} indices
 */
export function getSurfaceIndices(complexPolygon) {
  // Prepare an array of hole indices as expected by earcut
  var holeIndices = getHoleIndices(complexPolygon);
  // Flatten the polygon as expected by earcut
  var verts = flattenVertices(complexPolygon, { dimensions: 2, result: [] });
  // Let earcut triangulate the polygon
  return earcut(verts, holeIndices, 2);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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